package com.thebeastshop.common.http;

import org.apache.http.*;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;

public class HttpUtil {
    private final Logger logger  = LoggerFactory.getLogger(HttpUtil.class);

    private String                charset = "UTF-8";

    protected CloseableHttpClient httpClient;

    public HttpUtil() {
        httpClient = HttpClients.createDefault();
    }

    public String getCharset() {
        return charset;
    }

    public void setCharset(String charset) {
        this.charset = charset;
    }

    protected HttpGet initGetMethod(HttpRequest req) {
        return initGetMethod(req, null);
    }

    protected HttpGet initGetMethod(HttpRequest req, List<Header> headers) {
        String url = req.getUrl();
        String params = req.getParams();
        if (url.indexOf("?") == -1 && params != null && !"".equals(params)) {
            url = url + "?" + params;
        }

        HttpGet getMethod = new HttpGet(url);
        getMethod.addHeader("Connection", "Keep-Alive");
        getMethod
            .setHeader(
                "User-Agent",
                "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/32.0.1700.6 Safari/537.36");
        getMethod.setHeader("Accept", "*/*");

        if (headers != null) {
            for (Header header : headers) {
                getMethod.addHeader(header);
            }
        }

        String host = req.getHost();
        if (host != null && !"".equals(host)) {
            getMethod.addHeader("Host", host);
            if (getMethod.getHeaders("Referer") == null) {
                getMethod.addHeader("Referer", "http://" + host + "/");
            }
        }

        String cookies = req.getCookieStr();
        if (cookies != null && !"".equals(cookies)) {
            getMethod.addHeader("Cookie", cookies);
        }

        return getMethod;
    }

    protected HttpPost initPostMethod(HttpRequest req) {
        return initPostMethod(req, null);
    }

    protected HttpPost initPostMethod(HttpRequest req, List<Header> headers) {
        HttpPost postMethod = new HttpPost(req.getUrl());

        postMethod.addHeader("Connection", "Keep-Alive");
        postMethod
            .setHeader(
                "User-Agent",
                "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/32.0.1700.6 Safari/537.36");
        postMethod.setHeader("Accept", "*/*");

        if (headers != null) {
            for (Header header : headers) {
                postMethod.addHeader(header);
            }
        }

        String host = req.getHost();
        if (host != null && !"".equals(host)) {
            postMethod.addHeader("Host", host);
            if (postMethod.getHeaders("Referer") == null) {
                postMethod.addHeader("Referer", "http://" + host + "/");
            }
        }

        String cookies = req.getCookieStr();
        if (cookies != null && !"".equals(cookies)) {
            if (cookies != null && !"".equals(cookies)) {
                postMethod.addHeader("Cookie", cookies);
            }
        }

        String params = req.getParams();
        if (params != null && !"".equals(params)) {
            String[] _params = params.split("&");
            List<NameValuePair> nvps = new ArrayList<NameValuePair>();
            for (int i = 0; i < _params.length; i++) {
                String _p = _params[i];
                int j = _p.indexOf("=");
                nvps.add(new BasicNameValuePair(_p.substring(0, j), (_p.length() > j + 1) ? _p
                    .substring(j + 1) : ""));
            }
            UrlEncodedFormEntity entity = new UrlEncodedFormEntity(nvps, Consts.UTF_8);
            postMethod.setEntity(entity);
        }

        return postMethod;
    }

    protected Map<String, String> getCookiesByResponseHeaders(HeaderIterator headerIterator) {
        Map<String, String> m = new HashMap<String, String>();
        while (headerIterator.hasNext()) {
            Header header = (Header) headerIterator.next();
            if ("Set-Cookie".equals(header.getName())) {
                for (String s : header.getValue().split(";")) {
                    s = s.trim();
                    int i = s.indexOf("=");
                    if (i != -1) {
                        m.put(s.substring(0, i), (s.length() > i + 1) ? s.substring(i + 1) : "");
                    }
                }
            }
        }
        return m;
    }

    protected Map<String, String> getHeadersByResponseHeaders(HeaderIterator headerIterator) {
        Map<String, String> m = new HashMap<String, String>();
        while (headerIterator.hasNext()) {
            Header header = (Header) headerIterator.next();
            if (!"Set-Cookie".equals(header.getName())) {
                m.put(header.getName(), header.getValue());
            }
        }
        return m;
    }

    public HttpResponse request(HttpRequest req) throws HttpException, IOException {
        return request(req, null);
    }

    public HttpResponse request(HttpRequest req, List<Header> headers) throws HttpException,
                                                                      IOException {
        return request(req, headers, HttpStatus.SC_OK);
    }

    public HttpResponse request(HttpRequest req, int successStatus) throws HttpException,
                                                                   IOException {
        return request(req, null, successStatus);
    }

    public HttpResponse request(HttpRequest req, List<Header> headers, int successStatus)
                                                                                         throws HttpException,
                                                                                         IOException {
        return request(req, headers, successStatus, false);
    }

    public HttpResponse request(HttpRequest req, List<Header> headers, int successStatus,
                                boolean isPost) throws HttpException, IOException {
        HttpRequestBase httpMethod = isPost ? initPostMethod(req, headers) : initGetMethod(req,
            headers);
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(req.getSoTimeout())
            .setConnectTimeout(req.getTimeOut()).build();
        httpMethod.setConfig(requestConfig);

        HttpResponse resp = new HttpResponse();
        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(httpMethod);
            int statusCode = response.getStatusLine().getStatusCode();
            logger.debug("request statusCode:" + statusCode + "," + req.getUrl());
            resp.setStatusCode(statusCode);

            if (successStatus == -1 || statusCode == successStatus) {
                if (req.isNeedCookie()) {
                    Map<String, String> _cookies = getCookiesByResponseHeaders(response
                        .headerIterator());
                    resp.setCookieMap(_cookies);
                }
                if (req.isNeedHeaderMap()) {
                    Map<String, String> _headers = getHeadersByResponseHeaders(response
                        .headerIterator());
                    resp.setHeaderMap(_headers);
                }
                HttpEntity entity = response.getEntity();
                if (req.isNeedBody()) {
                    Header contentType = entity.getContentType();
                    Header contentEncoding = entity.getContentEncoding();
                    boolean isGzip = (contentEncoding != null && contentEncoding.getValue()
                        .toLowerCase().indexOf("gzip") > -1);

                    if (contentType != null && needWrap(contentType.getValue())) {
                        resp.setBody(getResBody(entity, isGzip));
                    } else {
                        resp.setByteBody(getResByteBody(entity, isGzip));
                    }
                }
            } else {
                throw new RuntimeException("request failed(status):" + statusCode + "/"
                                           + successStatus + "," + req.getUrl());
            }
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (response != null) {
                response.close();
            }
            if (httpMethod != null) {
                httpMethod.releaseConnection();
            }
        }
        return resp;
    }

    public void requestNoResp(HttpRequest req) throws HttpException, IOException {
        requestNoResp(req, null);
    }

    public void requestNoResp(HttpRequest req, List<Header> headers) throws HttpException,
                                                                    IOException {
        requestNoResp(req, headers, false);
    }

    public void requestNoResp(HttpRequest req, List<Header> headers, boolean isPost)
                                                                                    throws HttpException,
                                                                                    IOException {
        HttpRequestBase httpMethod = isPost ? initPostMethod(req, headers) : initGetMethod(req,
            headers);
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(req.getSoTimeout())
            .setConnectTimeout(req.getTimeOut()).build();
        httpMethod.setConfig(requestConfig);

        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(httpMethod);
            int statusCode = response.getStatusLine().getStatusCode();
            logger.debug("requestNoResp statusCode:" + statusCode + "," + req.getUrl());
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            if (response != null) {
                response.close();
            }
            if (httpMethod != null) {
                httpMethod.releaseConnection();
            }
        }
    }

    private boolean needWrap(String contentType) {
        if (contentType == null) {
            return false;
        } else if (contentType.contains("text/xml") || contentType.contains("text/html")
                   || contentType.contains("text/css") || contentType.contains("javascript")
                   || contentType.contains("application/json")) {
            return true;
        }
        return false;
    }

    private String getResBody(HttpEntity entity, boolean isGzip) throws IOException {
        StringBuilder sb = new StringBuilder(128);
        InputStream is = null;

        try {
            if (isGzip) {
                is = new GZIPInputStream(entity.getContent());
            } else {
                is = entity.getContent();
            }
            BufferedReader br = new BufferedReader(new InputStreamReader(is, charset));
            String str = null;
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
        } catch (Exception e) {
            logger.warn(e.toString());
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    logger.warn(e.toString());
                }
            }
        }

        return sb.toString();
    }

    private byte[] getResByteBody(HttpEntity entity, boolean isGzip) {
        InputStream is = null;

        List<Byte> resList = null;
        try {
            resList = new ArrayList<Byte>();
            if (isGzip) {
                is = new GZIPInputStream(entity.getContent());
            } else {
                is = entity.getContent();
            }
            byte[] b = new byte[1];
            while (is.read(b) != -1) {
                resList.add(new Byte(b[0]));
                b = new byte[1];
            }
        } catch (Exception e) {
            logger.warn(e.toString());
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    logger.warn(e.toString());
                }
            }
        }

        byte[] res = new byte[resList.size()];
        for (int i = 0; i < resList.size(); i++) {
            res[i] = ((Byte) resList.get(i)).byteValue();
        }

        return res;
    }
}
