package com.thebeastshop.kit.rocketmq.springboot;

import cn.hutool.core.util.ReflectUtil;
import com.thebeastshop.kit.rocketmq.RDelay;
import com.thebeastshop.kit.rocketmq.RMessage;
import com.thebeastshop.kit.rocketmq.RTopicInvoker;
import com.thebeastshop.kit.rocketmq.RocketMQTopicException;
import com.thebeastshop.kit.rocketmq.Topic;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyContext;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.MessageExt;
import org.apache.rocketmq.spring.core.RocketMQListener;
import org.apache.rocketmq.spring.core.RocketMQPushConsumerLifecycleListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

public class RocketMQConsumer implements RocketMQListener<String>, RocketMQPushConsumerLifecycleListener {

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private final Map<String, RTopicInvoker> TOPIC_INVOKER_CACHE = new ConcurrentHashMap<>();

    @Override
    public void onMessage(String s) {
    }


    @Override
    public void prepareStart(DefaultMQPushConsumer consumer) {
        final Class<?> clazz = this.getClass();
        final Method[] methods =  ReflectUtil.getMethods(clazz);

        for (final Method method : methods) {
            final Class<?>[] paramTypes = method.getParameterTypes();
            final Topic topicAnnotation = method.getAnnotation(Topic.class);
            if (topicAnnotation == null) {
                continue;
            }
            String topic = topicAnnotation.value();
            if (StringUtils.isBlank(topic)) {
                topic = "";
            } else {
                topic = topic.trim();
            }
            try {
                consumer.subscribe(topic, "*");
            } catch (MQClientException e) {
                throw new RuntimeException("[RocketMQ] 订阅Topic[\"" + topic + "\"; tags = \"*\"]失败: ", e);
            }
            final Class<?> paramType = paramTypes[0];
            TOPIC_INVOKER_CACHE.put(topic, new RTopicInvoker(paramType, method));
        }

        consumer.registerMessageListener((final List<MessageExt> messages, final ConsumeConcurrentlyContext context) -> {
            if (CollectionUtils.isNotEmpty(messages)) {
                for (final MessageExt message : messages) {
                    try {
                        final String topic = message.getTopic();
                        final RTopicInvoker invoker = TOPIC_INVOKER_CACHE.get(topic);
                        if (invoker == null) {
                            throw new RocketMQTopicException(topic);
                        }
                        logMessage(message);
                        invoker.invoke(this, message);
                    } catch (Throwable th) {
                        logMessageError(message, context, th);
                        return ConsumeConcurrentlyStatus.RECONSUME_LATER;
                    }
                }
            }
            return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
        });
    }

    private void logMessageError(
            final MessageExt message,
            final ConsumeConcurrentlyContext context,
            Throwable th) {
        try {
            final String topic = message.getTopic();
            final StringBuilder builder = new StringBuilder("[RocketMQ] 执行消费者任务时发生错误: ")
                    .append(th.getMessage())
                    .append("\n\n");
            builder.append("==========================================================================================\n");
            final String text = new String(message.getBody());
            final String id = message.getMsgId();
            final String tags = message.getTags();
            builder.append("\tid = ").append(id)
                    .append(", topic = ").append(topic)
                    .append(", ack index = ").append(context.getAckIndex());
            if (StringUtils.isNotEmpty(tags)) {
                builder.append(", tags = ")
                        .append(tags);
            }
            final String delay = RDelay.timeLevelToString(message.getDelayTimeLevel());
            if (StringUtils.isNotEmpty(delay)) {
                builder.append(", delay = ")
                        .append(delay);
            }
            builder.append("\n\tbody: " + text).append("\n");
            builder.append("==========================================================================================\n");
            log.error(builder.toString(), th);
        } catch (Throwable oth) {
            log.error("[RocketMQ] 打印错误日志时出现错误: ", oth);
        }
    }

    private void logMessage(MessageExt message) {
        StringBuilder content = new StringBuilder("[RocketMQ 消费者] 接受消息(")
                .append("id = ")
                .append(message.getMsgId())
                .append(", topic = ")
                .append(message.getTopic());
        String tags = message.getTags();
        if (StringUtils.isNotEmpty(tags)) {
            content.append(", tags = ")
                    .append(tags);
        }
        String delay = RDelay.timeLevelToString(message.getDelayTimeLevel());
        if (StringUtils.isNotEmpty(delay)) {
            content.append(", delay = ")
                    .append(delay);
        }
        content.append("):\n\t");
        content.append(new String(message.getBody(), StandardCharsets.UTF_8));
        log.info(content.toString());
    }

}
