package com.thebeastshop.kit.redis.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import com.google.common.collect.Lists;
import com.thebeastshop.kit.redis.lock.RedisDistributLock;
import com.thebeastshop.kit.redis.script.RedisNamedScript;
import com.thebeastshop.kit.redis.script.RedisScriptManager;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.Redisson;
import org.redisson.api.*;
import org.redisson.api.LocalCachedMapOptions.EvictionPolicy;
import org.redisson.api.RScript.Mode;
import org.redisson.client.RedisClientConfig;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisException;
import org.redisson.client.codec.Codec;
import org.redisson.client.codec.StringCodec;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.codec.JsonJacksonCodec;
import org.redisson.codec.KryoCodec;
import org.redisson.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.ParameterizedType;
import java.util.*;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * 统一Redis客户端
 * @author bryan.zhang
 * @date 2017-6-19
 * @description Redis包装实现
 */
@SuppressWarnings({"unchecked","rawtypes"})
public class RedisClient {

	private final Logger log = LoggerFactory.getLogger(getClass());

	private RedissonClient redissonClient;

	private RedisScriptManager redisScriptManager = RedisScriptManager.scriptManager();

	private Map<Integer, RedissonClient> otherRedissClientMap = new HashMap<Integer, RedissonClient>();

	private RedisConnection redisConnection;

	private RedisDistributLock redisDistributLock;

	public RedisClient() {
	}

	public RedisClient(RedissonClient redissonClient) {
		this.redissonClient = redissonClient;
	}

	public boolean hasKey(String key){
		return redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE).isExists();
	}

	public <T> Map<String,T> mutiGet(List<String> keys){
		return redissonClient.getBuckets(JsonJacksonCodec.INSTANCE).get(keys.toArray(new String[keys.size()]));
	}

	public <T> Map<String,T> mutiGet(String... keys){
		return mutiGet(Lists.newArrayList(keys));
	}


	public RedisConnection getRedisConnection() {
		if (redisConnection == null) {
			synchronized (this) {
				if (redisConnection == null)
					initRedisCommondConnection();
			}
		}
		return redisConnection;
	}

	/**
	 * 带超时的普通KV存放
	 * @param key
	 * @param t
	 * @param timeOut
	 */
	public <T> void putCache(String key, T t, Long timeOut){
		putCache(key, t, timeOut, TimeUnit.SECONDS, JsonJacksonCodec.INSTANCE);
	}

	/**
	 * 带超时的普通KV存放
	 * @param key
	 * @param t
	 * @param timeOut
	 */
	public <T> void putCache(String key, T t, Long timeOut, TimeUnit unit){
		putCache(key, t, timeOut, unit, JsonJacksonCodec.INSTANCE);
	}

	/**
	 * 不带超时时间的普通KV存放
	 * @param key 域key
	 * @param t 数据
	 */
	public <T> void putCache(String key, T t){
		putCache(key, t, null, null, JsonJacksonCodec.INSTANCE);
	}

	public void putStringCache(String key, String value){
		putCache(key, value, null, null, StringCodec.INSTANCE);
	}

	public void putIntegerCache(String key, Integer value){
		putCache(key, value, null, null, StringCodec.INSTANCE);
	}

	public void putLongCache(String key, Long value){
		putCache(key, value, null, null, StringCodec.INSTANCE);
	}

	public <T> void putByteCache(String key, T t){
		putCache(key, t, null, null, new KryoCodec());
	}

	public <T> void putByteCache(String key, T t, Long timeout, TimeUnit unit){
		putCache(key, t, timeout, unit, new KryoCodec());
	}

	/**
	 * 不带超时时间的共用KV存放
	 * @param key 域key
	 * @param t 数据
	 */
	public <T> void putCommCache(String key, T t){
		putCommCache(key, t, null, null);
	}

	/**
	 * 带超时时间的共用KV存放
	 * @param key
	 * @param t
	 * @param timeout
	 * @param unit
	 */
	public <T> void putCommCache(String key, T t, Long timeout, TimeUnit unit){
		if(otherRedissClientMap.get(0) == null){
			initOtherRedissonClient(0);
		}
		Transaction trans = Cat.newTransaction("Redis","set");
		Cat.logEvent("RedisKey", key);
		try{
			RBucket<T> bucket = otherRedissClientMap.get(0).getBucket(key, JsonJacksonCodec.INSTANCE);
			if(timeout == null){
				bucket.set(t);
			}else{
				bucket.set(t, timeout, unit);
			}
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> void putCacheWithDbIndex(Integer dbIndex, String key, T t, Long timeout, TimeUnit unit){
		if(otherRedissClientMap.get(dbIndex) == null){
			initOtherRedissonClient(dbIndex);
		}
		Transaction trans = Cat.newTransaction("Redis","set");
		Cat.logEvent("RedisKey", key);
		try{
			RBucket<T> bucket = otherRedissClientMap.get(dbIndex).getBucket(key, JsonJacksonCodec.INSTANCE);
			if(timeout == null){
				bucket.set(t);
			}else{
				bucket.set(t, timeout, unit);
			}
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> void putHashCacheWithDbIndex(Integer dbIndex, String key, String hashKey, T t){
		if(otherRedissClientMap.get(dbIndex) == null){
			initOtherRedissonClient(dbIndex);
		}
		Transaction trans = Cat.newTransaction("Redis","hset");
		Cat.logEvent("RedisKey", key);
		try{
			RMap<String, T> rmap = otherRedissClientMap.get(dbIndex).getMap(key, JsonJacksonCodec.INSTANCE);
			rmap.fastPut(hashKey, t);
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	private <T> void putCache(String key, T t, Long timeOut, TimeUnit unit, Codec codec){
		Transaction trans = Cat.newTransaction("Redis","set");
		Cat.logEvent("RedisKey", key);
		try{
			RBucket<T> bucket = redissonClient.getBucket(key,codec);
			if(timeOut == null){
				bucket.set(t);
			}else{
				bucket.set(t, timeOut, unit);
			}
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> void putCacheAsyn(String key, T t){
		putCacheAsyn(key, t, null, null);
	}

	public <T> void putCacheAsyn(String key, T t, Long timeOut, TimeUnit unit){
		Transaction trans = Cat.newTransaction("Redis","set");
		Cat.logEvent("RedisKey", key);
		try{
			RBucket<T> bucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
			if(timeOut == null){
				bucket.setAsync(t);
			}else{
				bucket.setAsync(t, timeOut, unit);
			}
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> void putAllCache(Map<String, T> map, Long liveTimeSec){
		try{
			RBatch batch = redissonClient.createBatch();
			RBucketAsync<T> rMapAsync = null;

			for(Map.Entry<String, T> entry : map.entrySet()){
				rMapAsync = batch.getBucket(entry.getKey(), JsonJacksonCodec.INSTANCE);
				rMapAsync.setAsync(entry.getValue(), liveTimeSec, TimeUnit.SECONDS);
			}
			batch.execute();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public <T> void putAllCache(List<RedisKVInstance<T>> list){
		try{
			RBatch batch = redissonClient.createBatch();
			RBucketAsync<T> rMapAsync = null;

			for(RedisKVInstance<T> item : list){
				rMapAsync = batch.getBucket(item.getKey(), JsonJacksonCodec.INSTANCE);
				rMapAsync.setAsync(item.getValue(), item.getLiveTimeSec(), TimeUnit.SECONDS);
			}

			batch.execute();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public <T> void putAllCache(Map<String, T> map){
		try{
			RBatch batch = redissonClient.createBatch();
			RBucketAsync<T> rMapAsync = null;

			for(Map.Entry<String, T> entry : map.entrySet()){
				rMapAsync = batch.getBucket(entry.getKey(), JsonJacksonCodec.INSTANCE);
				rMapAsync.setAsync(entry.getValue());
			}
			batch.execute();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public void putAllStringCache(Map<String, String> map,Long liveTimeSec){
		try{
			RBatch batch = redissonClient.createBatch();
			RBucketAsync<String> rMapAsync = null;

			for(Map.Entry<String, String> entry : map.entrySet()){
				rMapAsync = batch.getBucket(entry.getKey(), new StringCodec());
				rMapAsync.setAsync(entry.getValue(), liveTimeSec, TimeUnit.SECONDS);
			}
			batch.execute();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public void putAllStringCache(Map<String, String> map){
		try{
			RBatch batch = redissonClient.createBatch();
			RBucketAsync<String> rMapAsync = null;

			for(Map.Entry<String, String> entry : map.entrySet()){
				rMapAsync = batch.getBucket(entry.getKey(), new StringCodec());
				rMapAsync.setAsync(entry.getValue());
			}
			batch.execute();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	/**
	 * 取普通缓存数据
	 * @param key
	 * @return
	 */
	public <T> T getCache(String key){
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			RBucket<T> rBucket = redissonClient.getBucket(key, JsonJacksonCodec.INSTANCE);
			T t = rBucket.get();
			trans.setSuccessStatus();
			return t;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> T getSimpleJsonCache(String key) {
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			RBucket<String> bucket = redissonClient.getBucket(key, new StringCodec());
			String str = bucket.get();
			T t = JSONObject.parseObject(str, new TypeReference<T>() {});
			trans.setSuccessStatus();
			return t;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	/**
	 * 取普通缓存数据
	 * @param key
	 * @return
	 */
	public String getStringCache(String key){
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			String str = (String)redissonClient.getBucket(key, new StringCodec()).get();
			trans.setSuccessStatus();
			return str;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public Integer getIntegerCache(String key){
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			String str = (String)redissonClient.getBucket(key, new StringCodec()).get();
			trans.setSuccessStatus();
			return Integer.parseInt(str);
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public Long getLongCache(String key){
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			String str = (String)redissonClient.getBucket(key, new StringCodec()).get();
			trans.setSuccessStatus();
			return Long.parseLong(str);
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> T getByteCache(String key){
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			T t = (T)redissonClient.getBucket(key, new KryoCodec()).get();
			trans.setSuccessStatus();
			return t;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> T getCommCache(String key){
		if(otherRedissClientMap.get(0) == null){
			initOtherRedissonClient(0);
		}
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			T t = (T)otherRedissClientMap.get(0).getBucket(key).get();
			trans.setSuccessStatus();
			return t;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> T getCacheWithDbIndex(Integer dbIndex, String key){
		if(otherRedissClientMap.get(dbIndex) == null){
			initOtherRedissonClient(dbIndex);
		}
		Transaction trans = Cat.newTransaction("Redis","get");
		Cat.logEvent("RedisKey", key);
		try{
			T t = (T)otherRedissClientMap.get(dbIndex).getBucket(key,JsonJacksonCodec.INSTANCE).get();
			trans.setSuccessStatus();
			return t;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> T getCache(String key,DataGetter<T> dataGetter){
		return getCache(key,dataGetter,null,null);
	}

	public <T> T getCache(String key,DataGetter<T> dataGetter,Long ttl,TimeUnit unit){
		T t = getCache(key);
		if(t == null){
			log.info("key[{}] is not exist,get data and put into redis",key);
			t = dataGetter.getData();
			putCache(key,t,ttl,unit);
		}
		return t;
	}

	/**
	 * 带有超时时间的HASH缓存存放
	 * @param key 域key
	 */
	public <T> void putAllHashCache(String key, Map<String, T> map){
		Transaction trans = Cat.newTransaction("Redis","setHash");
		Cat.logEvent("RedisKey", key);
		try{
			RBatch batch = redissonClient.createBatch();
			RMapAsync<String, T> rMapAsync = batch.getMap(key, JsonJacksonCodec.INSTANCE);

			for(Map.Entry<String, T> entry : map.entrySet()){
				rMapAsync.fastPutAsync(entry.getKey(), entry.getValue());
			}

			batch.execute();

			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> void putAllHashCache(String key, Map<String, T> map, Long liveTimeSec){
		Transaction trans = Cat.newTransaction("Redis","setHash");
		Cat.logEvent("RedisKey", key);
		try{
			RBatch batch = redissonClient.createBatch();
			RMapAsync<String, T> rmapAsync = batch.getMap(key, JsonJacksonCodec.INSTANCE);

			for(Map.Entry<String, T> entry : map.entrySet()){
				rmapAsync.fastPutAsync(entry.getKey(), entry.getValue());
			}

			rmapAsync.expireAsync(liveTimeSec, TimeUnit.SECONDS);

			batch.execute();
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> void putHashCache(String key, String hashKey, T t){
		Transaction trans = Cat.newTransaction("Redis","setHash");
		Cat.logEvent("RedisKey", key);
		try{
			RMap<String, T> rmap = redissonClient.getMap(key, JsonJacksonCodec.INSTANCE);
			rmap.fastPut(hashKey, t);
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public void deleteHashCache(String key, String hashKey){
		Transaction trans = Cat.newTransaction("Redis","deleteHash");
		Cat.logEvent("RedisKey", key);
		try{
			RMap rmap = redissonClient.getMap(key, JsonJacksonCodec.INSTANCE);
			rmap.remove(hashKey);
			trans.setSuccessStatus();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	/**
	 * 取HASH缓存数据
	 * @param key 域key
	 * @param hashKey HashKey
	 * @return T
	 */
	public <T> T getHashCache(String key, String hashKey){
		Transaction trans = Cat.newTransaction("Redis","getHash");
		Cat.logEvent("RedisKey", key);
		try{
			T t = (T)redissonClient.getMap(key, JsonJacksonCodec.INSTANCE).get(hashKey);
			trans.setSuccessStatus();
			return t;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public <T> Map<String,T> getHashCache(String key){
		Transaction trans = Cat.newTransaction("Redis","getHash");
		Cat.logEvent("RedisKey", key);
		try{
			RMap<String,T> rmap = redissonClient.getMap(key, JsonJacksonCodec.INSTANCE);
			Map<String,T> map = rmap.readAllMap();
			trans.setSuccessStatus();
			return map;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	private final LocalCachedMapOptions options = LocalCachedMapOptions.defaults()
			.evictionPolicy(EvictionPolicy.LFU)
			.cacheSize(1000)
			.reconnectionStrategy(LocalCachedMapOptions.ReconnectionStrategy.CLEAR)
			.syncStrategy(LocalCachedMapOptions.SyncStrategy.INVALIDATE)
			.timeToLive(60, TimeUnit.SECONDS)
			.maxIdle(60, TimeUnit.SECONDS);

	private Map<String, RLocalCachedMap> localCachedMaps = new HashMap<>();

	public <T> List<T> mutiHashGet(String key,List<String> hashKeyList){
		Transaction trans = Cat.newTransaction("Redis","mutiGetHash");
		Cat.logEvent("RedisKey", key);
		try{
			RLocalCachedMap<String, T> localCachedMap;
			if (localCachedMaps.containsKey(key)){
				localCachedMap = localCachedMaps.get(key);
			}else{
				localCachedMap = redissonClient.getLocalCachedMap(key, JsonJacksonCodec.INSTANCE ,options);
				localCachedMaps.put(key, localCachedMap);
			}

			List<T> result = new ArrayList<>();
			for(String hashKey : hashKeyList){
				result.add(localCachedMap.get(hashKey));
			}

			trans.setSuccessStatus();
			return result;
		}catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw new RuntimeException("redis操作出现异常", th);
		}finally{
			trans.complete();
		}
	}

	public long incr(String key){
		try{
			RAtomicLong atomicLong = redissonClient.getAtomicLong(key);
			return atomicLong.incrementAndGet();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public long incr(String key,int step){
		try{
			RAtomicLong atomicLong = redissonClient.getAtomicLong(key);
			return atomicLong.addAndGet(step);
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public long decr(String key){
		try{
			RAtomicLong atomicLong = redissonClient.getAtomicLong(key);
			return atomicLong.decrementAndGet();
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public long decr(String key,int step){
		try{
			RAtomicLong atomicLong = redissonClient.getAtomicLong(key);
			return atomicLong.addAndGet(-step);
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public void expire(String key,long timeToLive){
		try{
			RBucket bucket = redissonClient.getBucket(key);
			bucket.expire(timeToLive, TimeUnit.SECONDS);
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	public void expire(String key,Date date){
		try{
			RBucket bucket = redissonClient.getBucket(key);
			bucket.expireAt(date);
		}catch(Throwable th){
			log.error("redis操作异常", th);
			throw new RuntimeException("redis操作出现异常", th);
		}
	}

	/**
	 * 获取本地缓存映射
	 * @param key map的KEY
	 * @param cacheSize  本地缓存的最大个数
	 * @param timeToLiveSec 本地缓存每项最多存活时间
	 * @return
	 */
	public <T> RLocalCachedMap<String, T> getLocalCacheMap(String key,Integer cacheSize,Long timeToLiveSec){
		return getLocalCacheMap(key, cacheSize, timeToLiveSec, false);
	}

	/**
	 * 获取本地缓存映射
	 * @param key map的KEY
	 * @param cacheSize 本地缓存的最大个数
	 * @param timeToLiveSec 本地缓存每项最多存活时间
	 * @param syncOnChange 如果该值是true时，在该实例执行更新和删除操作的同时，将向其他所有的相同实例发
	 * 					      送针对该元素的淘汰消息。其他相同实例在收到该消息以后，会同时删除自身的缓存。下次读取
	 * 					      该元素时会从Redis服务器获取。
	 * @return
	 */
	public <T> RLocalCachedMap<String, T> getLocalCacheMap(String key,Integer cacheSize,Long timeToLiveSec,boolean syncOnChange){
		LocalCachedMapOptions options = LocalCachedMapOptions.defaults()
				.evictionPolicy(EvictionPolicy.LFU)
				.cacheSize(cacheSize)
				.syncStrategy(syncOnChange?LocalCachedMapOptions.SyncStrategy.INVALIDATE: LocalCachedMapOptions.SyncStrategy.NONE)
				.timeToLive(timeToLiveSec, TimeUnit.SECONDS)
				.maxIdle(0);
		RLocalCachedMap<String, T> map = redissonClient.getLocalCachedMap(key, options);
		return map;
	}

	/**
	 * 获得一个redis的分布式队列
	 * 内部采用发布订阅模式，在Redis节点故障转移（主从切换）或断线重连以后，内置的相关话题监听器将自动完成话题的重新订阅。
	 * @param queueName 队列名称
	 * @return
	 */
	public <T> BlockingQueue<T> getReddisonQueue(String queueName){
		return redissonClient.getBlockingQueue(queueName, JsonJacksonCodec.INSTANCE);
	}

	public boolean delCache(String key){
		return redissonClient.getBucket(key).delete();
	}

	public long delCacheByPattern(String pattern){
		if(StringUtils.isBlank(pattern)){
			return 0;
		}
		if(pattern.equals("*") || pattern.length() <= 3){
			return 0;
		}

		RKeys rKeys = redissonClient.getKeys();
		return rKeys.deleteByPattern(pattern);
	}

	public boolean delCommCache(String key){
		if(otherRedissClientMap.get(0) == null){
			initOtherRedissonClient(0);
		}
		return otherRedissClientMap.get(0).getBucket(key).delete();
	}


	public <T> T executeScript(String scriptStr){
		return redissonClient.getScript().eval(Mode.READ_ONLY, scriptStr, RScript.ReturnType.VALUE);
	}

	public Future<Object> executeScriptAsync(String scriptStr){
		RScript s = redissonClient.getScript();
		String sha = s.scriptLoad(scriptStr);
		return s.evalShaAsync(Mode.READ_ONLY, sha, RScript.ReturnType.VALUE,Collections.emptyList());
	}



/*
	private  <T> T executeNamedScript(String name, Class<T> returnType, Mode mode, int count, Object... args) {
		Transaction trans = Cat.newTransaction("Redis","exec-lua " + name);
		Cat.logEvent("luaName", name);
		Cat.logEvent("luaArg", JSON.toJSONString(args));
		RedisNamedScript namedScript = null;
		try{
			namedScript = redisScriptManager.getNamedScript(name);
			if (namedScript == null) {
				return null;
			}
			T ret = namedScript.eval(redissonClient, returnType, mode, args);
			trans.setSuccessStatus();
			return ret;
		} catch (RedisException ex) {
			if (ex.getMessage().indexOf("NOSCRIPT") > -1) {
				namedScript.load(redissonClient);
				if (count > 0) {
					return executeNamedScript(name, returnType, mode, count - 1, args);
				}
				throw ex;
			}
			else {
				log.error("redis操作异常", ex);
				Cat.logError(ex);
				trans.setStatus(ex);
				throw ex;
			}
		} catch (Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw th;
		} finally{
			trans.complete();
		}
	}
*/


	private  <T, V> T executeNamedScript(String name, Object returnType, Mode mode, int count, Object... args) {
		String argsJson = JSON.toJSONString(args);
		log.info("[Redis] start eval lua: [" + name + "] args: " + argsJson);
		Transaction trans = Cat.newTransaction("Redis","exec-lua " + name);
		Cat.logEvent("luaName", name);
		Cat.logEvent("luaArg", argsJson);
		RedisNamedScript namedScript = null;
		try{
			namedScript = redisScriptManager.getNamedScript(name);
			if (namedScript == null) {
				return null;
			}
			T ret = null;
			if (returnType instanceof Class) {
				ret = namedScript.eval(redissonClient, (Class<T>) returnType, mode, args);
			}
			else if (returnType instanceof TypeReference) {
				ret = namedScript.eval(redissonClient, (TypeReference<T>) returnType, mode, args);
			}
			trans.setSuccessStatus();
			return ret;
		} catch (RedisException ex) {
			if (ex.getMessage().indexOf("NOSCRIPT") > -1) {
				namedScript.load(redissonClient);
				if (count > 0) {
					return executeNamedScript(name, returnType, mode, count - 1, args);
				}
				throw ex;
			}
			else {
				log.error("redis操作异常", ex);
				Cat.logError(ex);
				trans.setStatus(ex);
				throw ex;
			}
		} catch(Throwable th){
			log.error("redis操作异常", th);
			Cat.logError(th);
			trans.setStatus(th);
			throw th;
		} finally {
			trans.complete();
		}
	}


	/**
	 * 根据脚本名称执行脚本<BR/>
	 * 模式：读写
	 * @param name 脚本名称
	 * @param <T>
	 * @return
	 */
	public <T> T executeNamedScript(String name, Class<T> returnType, Object... args) {
		return executeNamedScript(name, returnType, Mode.READ_WRITE, 2, args);
	}


	/**
	 * 根据脚本名称执行脚本<BR/>
	 * 模式：读写
	 * @param name 脚本名称
	 * @param <T>
     * @return
     */
	public <T> T executeNamedScript(String name, TypeReference<T> returnType, Object... args) {
		return executeNamedScript(name, returnType, Mode.READ_WRITE, 2, args);
	}




	/**
	 * 根据脚本名称执行脚本<BR/>
	 * 模式：只读
	 * @param name 脚本名称
	 * @param <T>
	 * @return
	 */
	public <T> T executeNamedScriptReadOnly(String name, Class<T> returnType, Object... args) {
		return executeNamedScript(name, returnType, Mode.READ_ONLY, 2, args);
	}



	/**
	 * 根据脚本名称执行脚本<BR/>
	 * 模式：只读
	 * @param name 脚本名称
	 * @param <T>
	 * @return
	 */
	public <T> T executeNamedScriptReadOnly(String name, TypeReference<T> returnType, Object... args) {
		return executeNamedScript(name, returnType, Mode.READ_ONLY, 2, args);
	}

	public void initRedisCommondConnection(){
		String address = getMasterAddress(redissonClient);
		String password = redissonClient.getConfig().useSentinelServers().getPassword();
		int database = redissonClient.getConfig().useSentinelServers().getDatabase();

		RedisClientConfig config = new RedisClientConfig();
		config.setAddress(address) // 或者用rediss://使用加密连接
				.setPassword(password)
				.setDatabase(database)
				.setClientName("redisClient");
		org.redisson.client.RedisClient redisClient = org.redisson.client.RedisClient.create(config);
		redisConnection = redisClient.connect();
	}

	public void watch(String... keys){
		if(redisConnection == null){
			initRedisCommondConnection();
		}
        redisConnection.sync(StringCodec.INSTANCE, RedisCommands.WATCH, keys);
	}

	public void putAllCacheInTx(Map<String,?> map){
		if(redisConnection == null){
			initRedisCommondConnection();
		}
		redisConnection.sync(StringCodec.INSTANCE, RedisCommands.MULTI);
		for(Map.Entry<String,?> entry : map.entrySet()){
			redisConnection.sync(StringCodec.INSTANCE, RedisCommands.SET,entry.getKey(),entry.getValue());
		}
		List result = redisConnection.sync(StringCodec.INSTANCE, RedisCommands.EXEC);
		if(CollectionUtils.isEmpty(result)){
			throw  new RuntimeException("事务被中断");
		}
	}

	public String getMasterAddress(RedissonClient client){
		for(Node redisNode : redissonClient.getNodesGroup().getNodes()){
			if(redisNode.getType().equals(NodeType.MASTER)){
				return "redis://" + redisNode.getAddr().getHostName() + ":" + redisNode.getAddr().getPort();
			}
		}
		return null;
	}


	private void initOtherRedissonClient(int dbIndex){
		Config oldConfig = redissonClient.getConfig();
		Config config = new Config(oldConfig);
		config.useSentinelServers().setDatabase(dbIndex);
		RedissonClient redissonClient = Redisson.create(config);
		this.otherRedissClientMap.put(dbIndex, redissonClient);
	}


	public RedissonClient getRedissonClient() {
		return redissonClient;
	}


	public void setRedissonClient(RedissonClient redissonClient) {
		this.redissonClient = redissonClient;
	}

	public static class RedisKVInstance<T>{
		private String key;

		private T value;

		private Long liveTimeSec;

		public RedisKVInstance(String key, T value, Long liveTimeSec) {
			this.key = key;
			this.value = value;
			this.liveTimeSec = liveTimeSec;
		}

		public String getKey() {
			return key;
		}

		public void setKey(String key) {
			this.key = key;
		}

		public T getValue() {
			return value;
		}

		public void setValue(T value) {
			this.value = value;
		}

		public Long getLiveTimeSec() {
			return liveTimeSec;
		}

		public void setLiveTimeSec(Long liveTimeSec) {
			this.liveTimeSec = liveTimeSec;
		}
	}
}
