package com.thebeastshop.kit.kafka.utils;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.thebeastshop.kit.prop.PropConstants;
import org.apache.kafka.clients.CommonClientConfigs;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.CreateTopicsResult;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.KafkaFuture;
import org.springframework.kafka.core.DefaultKafkaConsumerFactory;
import org.springframework.kafka.core.DefaultKafkaProducerFactory;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.listener.*;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Random;

/**
 * kafka驱动器
 * @author Bryan.Zhang
 * @Date 2019/1/4
 */
public class KafkaDriver {

    private static KafkaDriver kafkaDriver;

    private String bootstrapServers;

    private KafkaTemplate kafkaTemplate;

    private AdminClient adminClient;

    private String customTopic;

    public static KafkaDriver drive(String bootstrapServers){
        if(kafkaDriver == null){
            kafkaDriver = new KafkaDriver(bootstrapServers, false, false);
        }
        return kafkaDriver;
    }

    public static KafkaDriver drive(String bootstrapServers, boolean autoFlush){
        if(kafkaDriver == null){
            kafkaDriver = new KafkaDriver(bootstrapServers, false, autoFlush);
        }
        return kafkaDriver;
    }

    public static KafkaDriver drive(String bootstrapServers, boolean autoFlush, boolean initAdminClient){
        if(kafkaDriver == null){
            kafkaDriver = new KafkaDriver(bootstrapServers, initAdminClient, autoFlush);
        }
        return kafkaDriver;
    }

    public static KafkaDriver reDrive(String bootstrapServers, boolean autoFlush){
        kafkaDriver = new KafkaDriver(bootstrapServers, false, autoFlush);
        return kafkaDriver;
    }

    public static KafkaDriver newDriverWithTopic(String bootstrapServers, String topic, boolean autoFlush){
        KafkaDriver kafkaDriver = new KafkaDriver(bootstrapServers, false, autoFlush);
        kafkaDriver.setCustomTopic(topic);
        return kafkaDriver;
    }

    public static KafkaDriver newDriver(String bootstrapServers, boolean autoFlush){
        return new KafkaDriver(bootstrapServers, false, autoFlush);
    }

    public void setDefaultTopic(String defaultTopic){
        if(kafkaTemplate == null){
            throw new RuntimeException("please init kafka driver first");
        }else{
            kafkaTemplate.setDefaultTopic(defaultTopic);
        }
    }

    public KafkaDriver(String bootstrapServers) {
        this(bootstrapServers, false, true);
    }

    public KafkaDriver(String bootstrapServers, boolean autoFlush) {
        this(bootstrapServers, false, autoFlush);
    }

    public KafkaDriver(String bootstrapServers, boolean isInitAdminClient, boolean autoFlush) {
        this.bootstrapServers = bootstrapServers;
        Map<String,Object> kafkaTemplateProperties = initKafkaTemplateProperties(bootstrapServers);
        DefaultKafkaProducerFactory producerFactory = new DefaultKafkaProducerFactory(kafkaTemplateProperties);
        kafkaTemplate = new KafkaTemplate(producerFactory, autoFlush);

        if (isInitAdminClient){
            Properties properties = new Properties();
            properties.put(CommonClientConfigs.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);
            properties.put(CommonClientConfigs.CLIENT_ID_CONFIG,PropConstants.getAppId());
            adminClient = AdminClient.create(properties);
        }
    }

    public ConcurrentMessageListenerContainer startConsumer(String topicName, String groupId, MessageListener listener,int concurrency,
                                                            ContainerProperties.AckMode ackMode){
        Map<String,Object> propertiesMap = initKafkaConsumerProperties(groupId);
        DefaultKafkaConsumerFactory consumerFactory = new DefaultKafkaConsumerFactory(propertiesMap);
        ContainerProperties containerProperties = new ContainerProperties(topicName);
        containerProperties.setMessageListener(listener);
        containerProperties.setAckMode(ackMode == null ? ContainerProperties.AckMode.BATCH : ackMode);
        ConcurrentMessageListenerContainer container = new ConcurrentMessageListenerContainer(consumerFactory,containerProperties);
        container.setConcurrency(concurrency);
        container.start();
        return container;
    }

    public ConcurrentMessageListenerContainer startBatchConsumer(String topicName, String groupId, BatchAcknowledgingMessageListener listener, int concurrency){
        Map<String,Object> propertiesMap = initKafkaConsumerProperties(groupId);
        DefaultKafkaConsumerFactory consumerFactory = new DefaultKafkaConsumerFactory(propertiesMap);
        ContainerProperties containerProperties = new ContainerProperties(topicName);
        containerProperties.setMessageListener(listener);
        containerProperties.setAckMode(ContainerProperties.AckMode.MANUAL);
        ConcurrentMessageListenerContainer container = new ConcurrentMessageListenerContainer(consumerFactory,containerProperties);
        container.setConcurrency(concurrency);
        container.start();
        return container;
    }

    public void createTopic(String topicName,int numPartitions) throws Exception{
        NewTopic newTopic = new NewTopic(topicName,numPartitions,(short)1);
        CreateTopicsResult result = adminClient.createTopics(Lists.newArrayList(newTopic));
        KafkaFuture<Void> kafkaFuture = result.all();
        kafkaFuture.get();
    }

    public void removeTopic(String topicName){
        adminClient.deleteTopics(Lists.newArrayList(topicName));
    }

    public void send(Object data,String topic){
        kafkaTemplate.send(topic, null, JSON.toJSONString(data));
    }

    public void send(Object data){
        kafkaTemplate.sendDefault(data);
    }

    public void sendWithTopic(Object data){
        kafkaTemplate.send(this.getCustomTopic(), null, JSON.toJSONString(data));
    }

    private Map<String,Object> initKafkaConsumerProperties(String groupId){
        Map<String,Object> kafkaConsumerProperties =new HashMap<String, Object>();
        kafkaConsumerProperties.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);
        kafkaConsumerProperties.put(ConsumerConfig.GROUP_ID_CONFIG, groupId);
        kafkaConsumerProperties.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, "false");
        kafkaConsumerProperties.put(ConsumerConfig.SESSION_TIMEOUT_MS_CONFIG, "60000");
        kafkaConsumerProperties.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringDeserializer");
        kafkaConsumerProperties.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG,"org.apache.kafka.common.serialization.StringDeserializer");
        kafkaConsumerProperties.put(ConsumerConfig.MAX_POLL_INTERVAL_MS_CONFIG,"1200000");
        kafkaConsumerProperties.put(ProducerConfig.CLIENT_ID_CONFIG, PropConstants.getAppId() + "_" + new Random().nextInt(1000));
        return kafkaConsumerProperties;
    }

    private Map<String,Object> initKafkaTemplateProperties(String kafkaBootstrapServers){
        Map<String,Object> kafkaTemplateProperties =new HashMap<String, Object>();
        kafkaTemplateProperties.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, kafkaBootstrapServers);
        kafkaTemplateProperties.put(ProducerConfig.RETRIES_CONFIG, 3);
        kafkaTemplateProperties.put(ProducerConfig.BATCH_SIZE_CONFIG, 16384);
        kafkaTemplateProperties.put(ProducerConfig.LINGER_MS_CONFIG,1);
        kafkaTemplateProperties.put(ProducerConfig.BUFFER_MEMORY_CONFIG,33554432);
        kafkaTemplateProperties.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG,"org.apache.kafka.common.serialization.StringSerializer");
        kafkaTemplateProperties.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG,"org.apache.kafka.common.serialization.StringSerializer");
        kafkaTemplateProperties.put(ProducerConfig.CLIENT_ID_CONFIG, PropConstants.getAppId() + "_" + new Random().nextInt(1000));
        return kafkaTemplateProperties;
    }

    public AdminClient getAdminClient() {
        return adminClient;
    }

    public void setAdminClient(AdminClient adminClient) {
        this.adminClient = adminClient;
    }

    public KafkaTemplate getKafkaTemplate() {
        return kafkaTemplate;
    }

    public void setKafkaTemplate(KafkaTemplate kafkaTemplate) {
        this.kafkaTemplate = kafkaTemplate;
    }

    public String getCustomTopic() {
        return customTopic;
    }

    public void setCustomTopic(String customTopic) {
        this.customTopic = customTopic;
    }

    public String getBootstrapServers() {
        return bootstrapServers;
    }
}