package com.thebeastshop.common.aop;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.exception.BusinessException;
import com.thebeastshop.common.log.AopLogger;
import com.thebeastshop.common.spring.SpringAware;
import com.thebeastshop.common.utils.GZIPHelper;
import com.thebeastshop.common.utils.MetaUtil;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.naming.ldap.Control;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Bryan.Zhang
 * @Date 2019-04-28
 */
@Aspect
public class ServiceAspect implements InitializingBean {

    private final AopLogger log = new AopLogger(ServiceAspect.class);

    private final static ConcurrentHashMap<Class<?>, Controller> ControllerAnnotationCache = new ConcurrentHashMap();

    private final static ConcurrentHashMap<Method, RequestMapping> RequestMappingAnnotationCache = new ConcurrentHashMap();

    @Pointcut("(@within(org.springframework.web.bind.annotation.RestController) || @within(org.springframework.stereotype.Controller) || @within(org.apache.dubbo.config.annotation.DubboService)) && within(com.thebeastshop..*)")
    public void cut(){}

    private int argLength;

    private boolean needCompress;

    private Environment environment = SpringAware.getBean(Environment.class);

    @Override
    public void afterPropertiesSet() throws Exception {
        String argLengthStr = environment.getProperty("serviceAspect.argLength","200");
        argLength = Integer.parseInt(argLengthStr);
        String needCompressStr = environment.getProperty("serviceAspect.needCompress","false");
        needCompress = Boolean.parseBoolean(needCompressStr);
    }

    @Around("cut()")
    public Object around(ProceedingJoinPoint pJoinPoint){
        try{
            Class clazz = pJoinPoint.getTarget().getClass();
            String className = clazz.getSimpleName();
            Signature signature = pJoinPoint.getSignature();
            MethodSignature methodSignature = (MethodSignature)signature;
            Method method = methodSignature.getMethod();
            String methodName = method.getName();

            Controller controllerAnn = ControllerAnnotationCache.computeIfAbsent(clazz,
                    key -> AnnotationUtils.getAnnotation(clazz, Controller.class));
            RequestMapping requestMappingAnn = null;

            if (controllerAnn != null) {
                requestMappingAnn = RequestMappingAnnotationCache.computeIfAbsent(method,
                        key -> AnnotationUtils.findAnnotation(method, RequestMapping.class));
            }

            if (requestMappingAnn != null) {
                log.info("Controller " + Arrays.toString(requestMappingAnn.path()) + " 开始调用", className, methodName);
            }
            log.info("方法[{}.{}]开始调用", className, methodName);
            Object arg = null;
            StringBuilder argsBuilder = new StringBuilder("(");
            int argsLen = pJoinPoint.getArgs().length;
            try {
                for (int i = 0; i < argsLen; i++) {
                    arg = pJoinPoint.getArgs()[i];
                    int argIndex = i + 1;
                    String argValue = getArgValue(arg);
                    argsBuilder.append("参数" + argIndex + ": " + argValue);
                    if (i < argsLen - 1) {
                        argsBuilder.append(", ");
                    }
                    log.info("方法[{}.{}]参数{}:{}", className, methodName, argIndex, argValue);
                }
                argsBuilder.append(")");
                Cat.logEvent("PigeonService.args",
                        className + "." + methodName,
                        Message.SUCCESS,
                        argsBuilder.toString());
            } catch (Exception e) {
                log.error("" + e);
            }
            long start = System.currentTimeMillis();
            Object result = null;
            try {
                result = pJoinPoint.proceed();
            } catch (Throwable t) {
                if (t instanceof BusinessException) {
                    log.info(t.getMessage());
                }
                else {
                    log.error("系统发生异常: ", t);
                }
                Class retType = method.getReturnType();
                if (ServiceResp.class.isAssignableFrom(retType)) {
                    result = ServiceResp.newInstanceFail(t);
                    try {
                        // 回滚当前事务
                        SpringAware.rollBack();
                    } catch (Throwable t2) {
                    }
                }
                else {
                    throw t;
                }
            }
            // 添加Service元信息
            if (result != null && result instanceof ServiceResp) {
                ServiceResp resp = (ServiceResp) result;
                resp.setAppId(MetaUtil.APP_NAME);
            }
            long end = System.currentTimeMillis();

            log.info("方法[{}.{}]结束调用,总共耗时{}毫秒", className, methodName, end - start);
            return result;
        }catch(Throwable t){
            if (t instanceof RuntimeException) {
                throw (RuntimeException) t;
            }
            throw new RuntimeException(t);
        }
    }


    private String getArgValue(Object arg) {
        String argValue;
        if (arg == null) {
            return "null";
        }
        if (arg instanceof CharSequence) {
            return "\"" + arg.toString() + "\"";
        }
        if (int.class.isAssignableFrom(arg.getClass()) ||
                long.class.isAssignableFrom(arg.getClass()) ||
                short.class.isAssignableFrom(arg.getClass()) ||
                byte.class.isAssignableFrom(arg.getClass()) ||
                boolean.class.isAssignableFrom(arg.getClass())) {
            return String.valueOf(arg);
        }
        try {
            argValue = JSON.toJSONString(arg, SerializerFeature.DisableCircularReferenceDetect);

            if(argLength == -1 && needCompress) {
                argValue = GZIPHelper.zipAndSerialize(argValue);
            }else if(argLength != -1 && argValue.length() > argLength){
                argValue = argValue.substring(0,argLength) + " ...";
            }
        } catch (Exception ex) {
            try {
                argValue = arg.getClass().getSimpleName() + " " + String.valueOf(arg);
            } catch (Exception ex2) {
                argValue = "null";
            }
        }
        return argValue;
    }


    public int getArgLength() {
        return argLength;
    }


    public void setArgLength(int argLength) {
        this.argLength = argLength;
    }


    public boolean isNeedCompress() {
        return needCompress;
    }


    public void setNeedCompress(boolean needCompress) {
        this.needCompress = needCompress;
    }
}
