package com.thebeastshop.common.zk;

import com.alibaba.fastjson.JSON;
import com.thebeastshop.common.exception.ZkConnectException;
import com.thebeastshop.common.exception.ZkConnectionNotInitException;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.cache.*;
import org.apache.curator.retry.RetryNTimes;
import org.apache.zookeeper.CreateMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * zookeeper驱动器
 * @author Bryan.Zhang
 * @Date 2019-01-03
 */
public class ZkDriver {

    private static Logger log = LoggerFactory.getLogger(ZkDriver.class);

    private static ZkDriver zkDriver;

    private CuratorFramework curatorClient;

    public static ZkDriver drive(String zkAddress){
        if(zkDriver == null){
            zkDriver = new ZkDriver(zkAddress);
        }
        return zkDriver;
    }

    public static ZkDriver drive(){
        return zkDriver;
    }

    private void validate(){
        if(curatorClient == null){
            throw new ZkConnectionNotInitException("请先初始化zk连接");
        }
    }

    public boolean checkExist(String nodePath){
        try{
            validate();
            return !(curatorClient.checkExists().forPath(nodePath) == null);
        }catch (Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    /**
     * 获取子节点路径列表
     * @param parentPath
     * @return
     * @throws Exception
     */
    public List<String> getNodePathList(String parentPath) throws Exception {
        List<String> paths = curatorClient.getChildren().forPath(parentPath);
        return paths;
    }

    public String getNodeDataText(String nodePath){
        try{
            validate();
            byte[] data = curatorClient.getData().forPath(nodePath);
            if(data == null || data.length == 0){
                return null;
            }
            return new String(data);
        }catch (Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    public <T> T getNodeData(String nodePath,Class<T> clazz){
        try{
            String data = getNodeDataText(nodePath);
            return JSON.parseObject(data,clazz);
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    public String getNodeData(String nodePath){
        try{
            return getNodeDataText(nodePath);
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    public void putData2EphemeralNode(String nodePath, String data){
        try{
            validate();
            if(!checkExist(nodePath)){
                createEphemeralNode(nodePath,data);
            }else{
                curatorClient.setData().forPath(nodePath, data.getBytes());
            }
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }


    public void putData2Node(String nodePath, String data){
        try{
            validate();
            if(!checkExist(nodePath)){
                createNode(nodePath,data);
            }else{
                curatorClient.setData().forPath(nodePath, data.getBytes());
            }
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    /**
     * 创建临时节点
     * @param nodePath
     * @param data
     */
    public void createEphemeralNode(String nodePath, String data) {
        try{
            validate();
            curatorClient.create().creatingParentsIfNeeded().withMode(CreateMode.EPHEMERAL)
                    .forPath(nodePath, data.getBytes());
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }


    /**
     * 创建默认（持久化）节点
     * @param nodePath
     * @param data
     */
    public void createNode(String nodePath, String data){
        try{
            validate();
            curatorClient.create().creatingParentsIfNeeded().forPath(nodePath,data.getBytes());
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    public void createListener(String nodePath, ZkListener listener, boolean executeOnStart){
        try{
            final NodeCache cache = new NodeCache(curatorClient,nodePath);
            listener.setCache(cache);
            cache.start(!executeOnStart);
            cache.getListenable().addListener(listener);
        }catch(Exception e){
            throw new ZkConnectException("zk连接异常",e);
        }
    }

    public void createListener(String nodePath, ZkListener listener) throws Exception{
        createListener(nodePath,listener,true);
    }

    public void createChildrenListener(String nodePath, ZkChildrenListener listener) throws Exception {
        final PathChildrenCache cache = new PathChildrenCache(curatorClient, nodePath, true);
        cache.start();
        cache.getListenable().addListener(listener);
    }


    public ZkDriver(String zkAddress) {
        curatorClient = CuratorFrameworkFactory.newClient(
                zkAddress,
                new RetryNTimes(10, 5000)
        );
        curatorClient.start();
    }
}
