package com.thebeastshop.common.utils;

import com.github.junrar.Archive;
import com.github.junrar.rarfile.FileHeader;
import com.thebeastshop.common.enums.FileTypeEnum;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.FileWriterWithEncoding;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Zip;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.unidal.helper.Files;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.*;
import java.util.List;
import java.util.Map.Entry;
import java.util.zip.ZipOutputStream;

/**
 * 读取文件工具类
 *
 * @author everywhere.z
 * @date 2015-3-5
 * @description
 */
public class FileUtil {

	private final static Logger logger = LoggerFactory.getLogger(FileUtil.class);

	public final static String LINE_SEPARATOR = System.getProperty("line.separator");

	private static final String[] IMAGE_NAMES = {
			"bmp", "dib", "gif", "jfif", "jpe", "jpeg", "jpg", "png", "tif", "tiff", "ico"
	};

	@SuppressWarnings("serial")
	private final static Map<MagicFileNumber, FileTypeEnum> FILE_TYPE = new HashMap<MagicFileNumber, FileTypeEnum>() {
		{
			// 常用文件类型的魔数
			put(new MagicFileNumber("FFD8FF", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.JPG);
			put(new MagicFileNumber("89504E47", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.PNG);
			put(new MagicFileNumber("47494638", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.GIF);
			put(new MagicFileNumber("5249464", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.WAV);
			put(new MagicFileNumber("4944330", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.MP3);
			put(new MagicFileNumber("41564920", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.AVI);
			put(new MagicFileNumber("66747970", MagicNumPositionEnum.middle, 8, 16), FileTypeEnum.MP4);
		}
	};

	private enum MagicNumPositionEnum{
		front,middle,end;
	}

	private static class MagicFileNumber{
		private String magicNumber;

		private MagicNumPositionEnum position;

		private int middleStart;

		private int middleEnd;

		public MagicFileNumber(String magicNumber, MagicNumPositionEnum position, int middleStart, int middleEnd) {
			super();
			this.magicNumber = magicNumber;
			this.position = position;
			this.middleStart = middleStart;
			this.middleEnd = middleEnd;
		}

		public String getMagicNumber() {
			return magicNumber;
		}

		public void setMagicNumber(String magicNumber) {
			this.magicNumber = magicNumber;
		}

		public MagicNumPositionEnum getPosition() {
			return position;
		}

		public void setPosition(MagicNumPositionEnum position) {
			this.position = position;
		}

		public int getMiddleStart() {
			return middleStart;
		}

		public void setMiddleStart(int middleStart) {
			this.middleStart = middleStart;
		}

		public int getMiddleEnd() {
			return middleEnd;
		}

		public void setMiddleEnd(int middleEnd) {
			this.middleEnd = middleEnd;
		}
	}

	public static boolean createDirectory(File file) {
		if (file.exists()) {
			return true;
		}
		return file.mkdirs();
	}

	public static boolean createDirectory(String dirname) {
		return createDirectory(new File(dirname));
	}

	public static String readFile2String(String filename) throws IOException {
		return readFile2String(new File(filename));
	}

	public static String readFile2String(File file) throws IOException {
		if ((file == null) || !file.exists() || file.isDirectory()) {
			return null;
		}
		return readInputStream2String(new FileInputStream(file));
	}

	public static String readInputStream2String(InputStream is) throws IOException {
		return readInputStream2String(is, "UTF-8");
	}

	public static String readInputStream2String(InputStream is, String charset) throws IOException {
		BufferedReader br = null;
		StringBuilder sb = new StringBuilder();
		try {
			br = new BufferedReader(new InputStreamReader(is, charset));
			for (String str = null; (str = br.readLine()) != null;) {
				sb.append(str).append(LINE_SEPARATOR);
			}
		} finally {
			closeIO(br);
		}
		return sb.toString().trim();
	}

	public static List<String> readFile2List(String filename) throws IOException {
		return readFile2List(new File(filename));
	}

	public static List<String> readFile2List(File file) throws IOException {
		if ((file == null) || !file.exists() || file.isDirectory()) {
			return null;
		}
		BufferedReader br = null;
		List<String> list = new ArrayList<String>();
		try {
			br = new BufferedReader(new FileReader(file));
			for (String str = null; (str = br.readLine()) != null;) {
				list.add(str);
			}
		} finally {
			closeIO(br);
		}
		return list;
	}

	public static void writeString2File(String str, String filename) throws IOException {
		writeString2File(str, new File(filename));
	}

	public static void writeString2File(String str, File file) throws IOException {
		BufferedWriter bw = null;
		try {
			bw = new BufferedWriter(new FileWriterWithEncoding(file, "UTF-8"));
			bw.write(str);
			bw.flush();
		} finally {
			closeIO(bw);
		}
	}

	public static void writeList2File(List<String> list, String filename) throws IOException {
		writeList2File(list, new File(filename), LINE_SEPARATOR);
	}

	public static void writeList2File(List<String> list, File file) throws IOException {
		writeList2File(list, file, LINE_SEPARATOR);
	}

	public static void writeList2File(List<String> list, String filename, String lineSeparator) throws IOException {
		writeList2File(list, new File(filename), lineSeparator);
	}

	public static void writeList2File(List<String> list, File file, String lineSeparator) throws IOException {
		StringBuffer sb = new StringBuffer();
		for (int i = 0, k = list.size(); i < k; i++) {
			if (i > 0) {
				sb.append(lineSeparator);
			}
			sb.append(list.get(i));
		}
		writeString2File(sb.toString(), file);
	}

	public static void closeIO(Closeable io) throws IOException {
		if (io != null) {
			io.close();
		}
	}

	public static void zipFile(File inFile, File outFile) throws IOException {
		ZipOutputStream zos = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(outFile)));
		ZipEntry ze = new ZipEntry(inFile.getName());
		zos.putNextEntry(ze);

		byte[] buf = new byte[2048];
		int readLen = 0;
		InputStream is = new BufferedInputStream(new FileInputStream(inFile));
		while ((readLen = is.read(buf, 0, 2048)) != -1) {
			zos.write(buf, 0, readLen);
		}
		is.close();

		zos.close();
	}

	/**
	 * 获得指定文件的byte数组
	 */
	public static byte[] getBytes(String filePath) {
		byte[] buffer = null;
		try {
			File file = new File(filePath);
			FileInputStream fis = new FileInputStream(file);
			ByteArrayOutputStream bos = new ByteArrayOutputStream(1000);
			byte[] b = new byte[1000];
			int n;
			while ((n = fis.read(b)) != -1) {
				bos.write(b, 0, n);
			}
			fis.close();
			bos.close();
			buffer = bos.toByteArray();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return buffer;
	}

	/**
	 * 获取文件类型
	 *
	 * @param filePath
	 * @return
	 */
	public static FileTypeEnum getFileType(String filePath) {
		return getFileType(new File(filePath));
	}

	/**
	 * 获取文件类型
	 *
	 * @param file
	 * @return
	 */
	public static FileTypeEnum getFileType(File file) {
		FileTypeEnum fileType = null;
		try {
			fileType = getFileType(new FileInputStream(file));
		} catch (Exception e) {
			logger.error("获取文件类型异常: {}", e);
		}
		return fileType;
	}

	/**
	 * 获取文件类型
	 *
	 * @param inputStream
	 * @return
	 */
	public static FileTypeEnum getFileType(InputStream inputStream) {
		FileTypeEnum fileType = null;
		// 数据流提取出字节数组
		try {
			byte[] b = new byte[28];
			inputStream.read(b);
			String hex = ByteUtil.bytes2Hex(b).toUpperCase();
			logger.info("文件魔数值: {}", hex);

			String middleHex = null;
			for(Entry<MagicFileNumber, FileTypeEnum> entry : FILE_TYPE.entrySet()) {
				if(entry.getKey().getPosition().equals(MagicNumPositionEnum.front)) {
					if(hex.startsWith(entry.getKey().getMagicNumber())) {
						fileType = entry.getValue();
						break;
					}
				}else if(entry.getKey().getPosition().equals(MagicNumPositionEnum.middle)) {
					middleHex = hex.substring(entry.getKey().getMiddleStart(),entry.getKey().getMiddleEnd());
					if(middleHex.equals(entry.getKey().getMagicNumber())) {
						fileType = entry.getValue();
						break;
					}
				}
			}
		} catch (Exception e) {
			logger.error("获取文件类型异常: {}", e);
		}
		return fileType;
	}

    /**
     * 获取文件扩展名
     *
     * @param fileName 文件名
     * @return
     */
    public static String getExtName(String fileName) {
        if (EmptyUtil.isEmpty(fileName)) {
            return "";
        }
        if (!fileName.contains(".")) {
            return "";
        }
        return fileName.substring(fileName.lastIndexOf(".") + 1);
    }
	/**
	 * 执行压缩操作
	 * @param srcPathName  需要被压缩的文件/文件夹
	 * @param zipFile  生成的压缩文件
	 */
	public static boolean compressZip(String srcPathName, File zipFile) {
		File srcdir = new File(srcPathName);
		try{
			if (!srcdir.exists()){
				throw new RuntimeException(srcPathName + "不存在！");
			}
			Project prj = new Project();
			Zip zip = new Zip();
			zip.setProject(prj);
			zip.setDestFile(zipFile);
			FileSet fileSet = new FileSet();
			fileSet.setProject(prj);
			fileSet.setDir(srcdir);
			//fileSet.setIncludes("**/*.java"); //包括哪些文件或文件夹 eg:zip.setIncludes("*.java");
			//fileSet.setExcludes(...); //排除哪些文件或文件夹
			zip.addFileset(fileSet);
			zip.execute();
			return true;
		}catch (Exception e){
			return false;
		}
	}
    /**
	 * 判断一个文件扩展名是否是图片格式
	 *
	 * @param extName 文件扩展名
	 * @return
	 */
	public static boolean isImageExtName(String extName) {
		boolean result = false;
		if (EmptyUtil.isNotEmpty(extName)) {
			if (ArrayUtils.contains(IMAGE_NAMES, extName.toLowerCase())) {
				result = true;
			}
		}
		return result;
	}
	public static Boolean resize(int width, int height, String filePath) throws IOException {
		width = Math.max(width, 1);
		height = Math.max(height, 1);
		Image targetImage = ImageIO.read(new File(filePath));
		BufferedImage image = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_RGB);
		image.getGraphics().drawImage(targetImage, 0, 0, width, height, Color.WHITE, null);
		String formatName = filePath.substring(filePath.lastIndexOf(".") + 1);
		return ImageIO.write(image, formatName, new File(filePath));
	}
	public static boolean isPdf(String extName) {
		boolean result = false;
		if (EmptyUtil.isNotEmpty(extName)) {
			if (extName.toLowerCase().equals(FILES.PDF.value)) {
				result = true;
			}
		}
		return result;
	}
	/**
	 * 判断一个文件扩展名是否是压缩格式(zip/rar)
	 *
	 * @param extName 文件扩展名
	 * @return
	 */
	public static boolean isCompressed(String extName) {
		boolean result = false;
		if (EmptyUtil.isNotEmpty(extName)) {
			if (extName.toLowerCase().equals(FILES.ZIP.value) ||
					extName.toLowerCase().equals(FILES.RAR.value)) {
				result = true;
			}
		}
		return result;
	}
	public static java.util.List<File> unCompressedFile(File originFile,String parentPath) {
		if (!originFile.exists()){
			return null;
		}
		if (isZIP(getExtName(originFile.getName()))){
			return unZipFile(originFile,parentPath);
		}else if(isRAR(getExtName(originFile.getName()))){
			return unRarFile(originFile,parentPath);
		}
		return null;
	}
	/**
	 * 根据原始rar路径，解压到指定文件夹下.
	 * @param originfile 原始rar路径
	 */
	public static synchronized java.util.List<File> unRarFile(File originfile,String parentPath) {
		if (!originfile.exists()){
			return null;
		}
		java.util.List<File> files = new ArrayList<File>();
		Archive archive = null;
		try {
			archive = new Archive(originfile);
			if (archive != null) {
				FileHeader fh = archive.nextFileHeader();
				while (fh != null) {
					if (fh.isDirectory()) { // 文件夹
						fh = archive.nextFileHeader();
						continue;
					} else { // 文件
						String fileName=fh.getFileNameW();// 中文名称
						if (EmptyUtil.isEmpty(fileName)){
							fileName = fh.getFileNameString();
						}
						// 过滤掉mac压缩文件时，产生的临时文件
						if (fileName.contains("__MACOSX")){
							continue;
						}
						File unRarFile = new File(parentPath,fileName);
						if (!new File(unRarFile.getParent()).exists()){
							new File(unRarFile.getParent()).mkdirs();
						}
						try {// 之所以这么写try，是因为万一这里面有了异常，不影响继续解压.
							FileOutputStream os = new FileOutputStream(unRarFile);
							archive.extractFile(fh, os);
							os.close();
						} catch (Exception ex) {
							ex.printStackTrace();
						}
						files.add(unRarFile);
						fh = archive.nextFileHeader();
					}
				}
			}else{
				return null;
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (archive != null){
				try {
					archive.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return files;
	}
	/**
	 * 判断一个文件扩展名是否是RAR压缩格式
	 *
	 * @param extName 文件扩展名
	 * @return
	 */
	public static boolean isRAR(String extName) {
		boolean result = false;
		if (EmptyUtil.isNotEmpty(extName)) {
			if (extName.toLowerCase().equals(FILES.RAR.value)) {
				result = true;
			}
		}
		return result;
	}
	public static synchronized java.util.List<File> unZipFile(File originfile,String parentPath){
		String encodeing = System.getProperty("sun.jnu.encoding");
		System.setProperty("sun.zip.encoding", encodeing);
		if (!originfile.exists()){
			return null;
		}
		java.util.List<File> files = new ArrayList<File>();
		// 默认按GBK解压
		ergodicZipFile(originfile,null,files,parentPath);
		return files;
	}
	private static void ergodicZipFile(File originfile,String encoding,java.util.List<File> files,String parentPath){
		ZipFile zipFile = null;
		BufferedOutputStream bout=null;
		try {
			if (encoding != null){
				zipFile = new ZipFile(originfile,encoding);
			}else{
				zipFile = new ZipFile(originfile);
			}
			for (Enumeration<? extends ZipEntry> e = zipFile.getEntries(); e.hasMoreElements();) {
				ZipEntry entry = e.nextElement();
				if (entry.isDirectory()) {
					continue;
				}
				// 过滤掉mac压缩文件时，产生的临时文件
				if (entry.getName().toUpperCase().contains("__MACOSX")) {
					continue;
				}
				File unZipFile = new File(parentPath,entry.getName());
				if (!new File(unZipFile.getParent()).exists()) {
					new File(unZipFile.getParent()).mkdirs();
				}
				bout = new BufferedOutputStream(new FileOutputStream(unZipFile));
				byte[] buf1 = new byte[1024];
				int len;
				InputStream in = zipFile.getInputStream(entry);
				while ((len = in.read(buf1)) > 0) {
					bout.write(buf1, 0, len);
				}
				bout.close();
				files.add(unZipFile);
			}
		}catch (Exception e1) {
			//e1.printStackTrace();
			files.clear();
			ergodicZipFile(originfile,"GBK",files,parentPath);
		}finally {
			if(bout != null){
				try {
					bout.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	public static void writeFileToLocal(byte[] data,String parentPath,String fileName){
		File rootFile = new File(parentPath);
		if (!rootFile.exists()){
			rootFile.mkdirs();
		}
		writeFileToLocal(data, parentPath+fileName);
	}
	/**
	 * 将字节流文件写入到本地磁盘
	 * @param data
	 * @param destPath
	 */
	public static void writeFileToLocal(byte[] data, String destPath){
		FileOutputStream outStream = null;
		try{
			//new一个文件对象用来保存文件，默认保存当前工程根目录
			File imageFile = new File(destPath);
			//创建输出流
			outStream = new FileOutputStream(imageFile);
			//写入数据
			outStream.write(data);
			//关闭输出流
			outStream.close();
		}catch(Exception e){
			e.printStackTrace();
		}finally {
			if (outStream != null){
				try {
					outStream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	/**
	 * 判断一个文件扩展名是否是ZIP压缩格式
	 *
	 * @param extName 文件扩展名
	 * @return
	 */
	public static boolean isZIP(String extName) {
		boolean result = false;
		if (EmptyUtil.isNotEmpty(extName)) {
			if (extName.toLowerCase().equals(FILES.ZIP.value)) {
				result = true;
			}
		}
		return result;
	}
	public static void resize(int width, int height, String filePath, boolean fillWhite) {
		try {
			double ratio = 0.0; //缩放比例
			File file = new File(filePath);
			BufferedImage tempBufferedImage = ImageIO.read(file);
			Image targetImage = tempBufferedImage.getScaledInstance(width, height, tempBufferedImage.SCALE_SMOOTH);
			//计算比例
			if ((tempBufferedImage.getHeight() > height) || (tempBufferedImage.getWidth() > width)) {
				if (tempBufferedImage.getHeight() > tempBufferedImage.getWidth()) {
					ratio = (new Integer(height)).doubleValue() / tempBufferedImage.getHeight();
				} else {
					ratio = (new Integer(width)).doubleValue() / tempBufferedImage.getWidth();
				}
				AffineTransformOp op = new AffineTransformOp(AffineTransform.getScaleInstance(ratio, ratio), null);
				targetImage = op.filter(tempBufferedImage, null);
			}
			//是否补白
			if (fillWhite) {
				BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
				Graphics2D graphics2D = image.createGraphics();
				graphics2D.setColor(Color.white);
				graphics2D.fillRect(0, 0, width, height);
				if (width == targetImage.getWidth(null)) {
					graphics2D.drawImage(targetImage, 0, (height - targetImage.getHeight(null)) / 2, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
				} else {
					graphics2D.drawImage(targetImage, (width - targetImage.getWidth(null)) / 2, 0, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
				}
				graphics2D.dispose();
				targetImage = image;
			}
			ImageIO.write((BufferedImage) targetImage, "jpg", file);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	public static String getFileName(String fileName){
		if (EmptyUtil.isEmpty(fileName)) {
			return "";
		}
		if (!fileName.contains(".")) {
			return fileName;
		}
		return fileName.substring(0,fileName.lastIndexOf("."));
	}
	private static enum FILES {
		PDF("pdf"),HTML("html"),EXCEL("excel"),DOC("doc"),ZIP("zip"),RAR("rar");
		private String value;

		private FILES(String value) {
			this.value = value;
		}
	}
	public static void downloadPic(String picUrl,String parentPath,String fileName){
		File rootFile = new File(parentPath);
		if (!rootFile.exists()){
			rootFile.mkdirs();
		}
		downloadPic(picUrl,parentPath+fileName);
	}
	public static void downloadPic(String picUrl,String destPath){
		FileOutputStream outStream = null;
		try{
			URL url = new URL(picUrl);
			//打开链接
			HttpURLConnection conn = (HttpURLConnection)url.openConnection();
			//设置请求方式为"GET"
			conn.setRequestMethod("GET");
			//超时响应时间为5秒
			conn.setConnectTimeout(5 * 1000);
			//通过输入流获取图片数据
			InputStream inStream = conn.getInputStream();
			//得到图片的二进制数据，以二进制封装得到数据，具有通用性
			byte[] data = readInputStream(inStream);
			//new一个文件对象用来保存图片，默认保存当前工程根目录
			File imageFile = new File(destPath);
			//创建输出流
			outStream = new FileOutputStream(imageFile);
			//写入数据
			outStream.write(data);
			//关闭输出流
			outStream.close();
		}catch(Exception e){
			e.printStackTrace();
		}finally {
			if (outStream != null){
				try {
					outStream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	public static byte[] downloadPic(String picUrl){
		InputStream inStream = null;
		try{
			URL url = new URL(picUrl);
			//打开链接
			HttpURLConnection conn = (HttpURLConnection)url.openConnection();
			//设置请求方式为"GET"
			conn.setRequestMethod("GET");
			//超时响应时间为5秒
			conn.setConnectTimeout(5 * 1000);
//			log.info(conn.getContentType());
			//通过输入流获取图片数据
			inStream = conn.getInputStream();
			//得到图片的二进制数据，以二进制封装得到数据，具有通用性
			byte[] data = readInputStream(inStream);
			return  data;
		}catch(Exception e){
			e.printStackTrace();
		}finally {
			if (inStream != null){
				try {
					inStream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return null;
	}
	/**
	 * 删除目录（文件夹）以及目录下的文件
	 * @param   sPath 被删除目录的文件路径
	 * @return  目录删除成功返回true，否则返回false
	 */
	public static boolean deleteDirectory(String sPath) {
		//如果sPath不以文件分隔符结尾，自动添加文件分隔符
		if (!sPath.endsWith(File.separator)) {
			sPath = sPath + File.separator;
		}
		File dirFile = new File(sPath);
		//如果dir对应的文件不存在，或者不是一个目录，则退出
		if (!dirFile.exists() || !dirFile.isDirectory()) {
			return false;
		}
		boolean flag = true;
		//删除文件夹下的所有文件(包括子目录)
		File[] files = dirFile.listFiles();
		for (int i = 0; i < files.length; i++) {
			//删除子文件
			if (files[i].isFile()) {
				flag = deleteFile(files[i].getAbsolutePath());
				if (!flag) {
					break;
				}
			} //删除子目录
			else {
				flag = deleteDirectory(files[i].getAbsolutePath());
				if (!flag) {
					break;
				}
			}
		}
		if (!flag) {
			return false;
		}
		//删除当前目录
		if (dirFile.delete()) {
			return true;
		} else {
			return false;
		}
	}
	/**
	 * 删除单个文件
	 * @param   sPath    被删除文件的文件名
	 * @return 单个文件删除成功返回true，否则返回false
	 */
	public static boolean deleteFile(String sPath) {
		boolean flag = false;
		File file = new File(sPath);
		// 路径为文件且不为空则进行删除
		if (file.isFile() && file.exists()) {
			file.delete();
			flag = true;
		}
		return flag;
	}
	//递归删除文件夹
	public static void deleteFile(File file) {
		if (file.exists()) {//判断文件是否存在
			if (file.isFile()) {//判断是否是文件
				file.delete();//删除文件
			} else if (file.isDirectory()) {//否则如果它是一个目录
				File[] files = file.listFiles();//声明目录下所有的文件 files[];
				for (int i = 0;i < files.length;i ++) {//遍历目录下所有的文件
					deleteFile(files[i]);//把每个文件用这个方法进行迭代
				}
				file.delete();//删除文件夹
			}
		}
	}
	public static byte[] readInputStream(InputStream inStream) throws Exception{
		ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		//创建一个Buffer字符串
		byte[] buffer = new byte[1024];
		//每次读取的字符串长度，如果为-1，代表全部读取完毕
		int len = 0;
		//使用一个输入流从buffer里把数据读取出来
		while( (len=inStream.read(buffer)) != -1 ){
			//用输出流往buffer里写入数据，中间参数代表从哪个位置开始读，len代表读取的长度
			outStream.write(buffer, 0, len);
		}
		//关闭输入流
		inStream.close();
		//把outStream里的数据写入内存
		return outStream.toByteArray();
	}
	/**
	 * 根据文件路径判断其是否存在
	 *
	 * @param filePath
	 * @return
	 */
	public static Boolean exists(String filePath) {
		Boolean result = false;
		if (EmptyUtil.isNotEmpty(filePath)) {
			File file = new File(filePath);
			if (file.exists() && file.canRead()) {
				result = true;
			}
		}
		return result;
	}

	/**
	 * 根据文件路径获取数据
	 *
	 * @param filePath
	 * @return byte[]
	 * @throws IOException
	 */
	public static byte[] getByteFileData(String filePath) throws IOException {
		if (FileUtil.exists(filePath)) {
			File file = new File(filePath);
			return FileUtils.readFileToByteArray(file);
		} else {
			throw new IOException("file is not exist");
		}
	}

	/**
	 * 把一个输入流转成文件
	 *
	 * @param in
	 * @param filePath
	 * @throws IOException
	 */
	public static Boolean InputStreamTOFile(InputStream in, String filePath) throws IOException {
		if (EmptyUtil.isNotEmpty(in) && EmptyUtil.isNotEmpty(filePath)) {
			File file = new File(filePath);
			byte[] b = IOUtils.toByteArray(in);
			FileUtils.writeByteArrayToFile(file, b);
			IOUtils.closeQuietly(in);
			return true;
		}
		return false;
	}

	/**
	 * 重置图片大小
	 *
	 * @param width
	 * @param height
	 * @param filePath
	 * @return
	 * @throws IOException
	 */
	public static Boolean resizeImage(int width, int height, String filePath) throws IOException {
		width = Math.max(width, 1);
		height = Math.max(height, 1);
		Image targetImage = ImageIO.read(new File(filePath));
		BufferedImage image = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_RGB);
		image.getGraphics().drawImage(targetImage, 0, 0, width, height, Color.WHITE, null);
		String formatName = filePath.substring(filePath.lastIndexOf(".") + 1);
		return ImageIO.write(image, formatName, new File(filePath));
	}


	/**
	 * 重置图片大小并补白
	 * @param width
	 * @param height
	 * @param filePath
	 * @param fillWhite
	 */
	public static void resizeImage(int width, int height, String filePath, boolean fillWhite) {
		try {
			double ratio = 0.0; //缩放比例
			File file = new File(filePath);
			BufferedImage tempBufferedImage = ImageIO.read(file);
			Image targetImage = tempBufferedImage.getScaledInstance(width, height, tempBufferedImage.SCALE_SMOOTH);
			//计算比例
			if ((tempBufferedImage.getHeight() > height) || (tempBufferedImage.getWidth() > width)) {
				if (tempBufferedImage.getHeight() > tempBufferedImage.getWidth()) {
					ratio = (new Integer(height)).doubleValue() / tempBufferedImage.getHeight();
				} else {
					ratio = (new Integer(width)).doubleValue() / tempBufferedImage.getWidth();
				}
				AffineTransformOp op = new AffineTransformOp(AffineTransform.getScaleInstance(ratio, ratio), null);
				targetImage = op.filter(tempBufferedImage, null);
			}
			//是否补白
			if (fillWhite) {
				BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
				Graphics2D graphics2D = image.createGraphics();
				graphics2D.setColor(Color.white);
				graphics2D.fillRect(0, 0, width, height);
				if (width == targetImage.getWidth(null)) {
					graphics2D.drawImage(targetImage, 0, (height - targetImage.getHeight(null)) / 2, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
				} else {
					graphics2D.drawImage(targetImage, (width - targetImage.getWidth(null)) / 2, 0, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
				}
				graphics2D.dispose();
				targetImage = image;
			}
			ImageIO.write((BufferedImage) targetImage, "jpg", file);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}


	public static void main(String[] args) {
		FileTypeEnum fileType = getFileType(new File("D:/1.png"));
		System.out.println(fileType);
	}
}
