package com.thebeastshop.common.validation;

import com.thebeastshop.common.exception.BaseErrorCode;
import com.thebeastshop.common.exception.CommonErrorCode;
import com.thebeastshop.common.exception.ValidationException;
import com.thebeastshop.common.utils.EmptyUtil;
import com.thebeastshop.common.utils.MetaUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Map;

/**
 * 验证工具类
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-09-28 16:16
 */
public class Validation {

    private final static Logger log = LoggerFactory.getLogger(Validation.class);

    /**
     * 断言：引用不为NULL
     * @param object
     * @param errorCode
     * @param msg
     */
    public static void assertNotNull(Object object, BaseErrorCode errorCode, String msg) {
        if (object == null) {
            throwValidationException(errorCode, msg);
        }
    }

    /**
     * 断言：引用不为NULL
     * @param object
     * @param msg
     */
    public static void assertNotNull(Object object,  String msg) {
        if (object == null) {
            throwValidationException(CommonErrorCode.RESULT_NULL, msg);
        }
    }

    /**
     * 断言：集合不为空
     * @param collection 集合
     * @param errorCode
     * @param msg
     */
    public static void assertNotEmpty(Collection collection, BaseErrorCode errorCode, String msg) {
        if (CollectionUtils.isEmpty(collection)) {
            throwValidationException(errorCode, msg);
        }
    }

    /**
     * 断言：集合不为空
     * @param collection 集合
     * @param msg
     */
    public static void assertNotEmpty(Collection collection, String msg) {
        if (CollectionUtils.isEmpty(collection)) {
            throwValidationException(CommonErrorCode.RESULT_EMPTY, msg);
        }
    }

    /**
     * 断言：字符串不为空
     * @param str 字符串
     * @param errorCode
     * @param msg
     */
    public static void assertNotEmpty(CharSequence str, BaseErrorCode errorCode, String msg) {
        if (EmptyUtil.isEmpty(str)) {
            throwValidationException(errorCode, msg);
        }
    }

    /**
     * 断言：字符串不为空
     * @param str 字符串
     * @param msg
     */
    public static void assertNotEmpty(String str, String msg) {
        if (EmptyUtil.isEmpty(str)) {
            throwValidationException(CommonErrorCode.RESULT_EMPTY, msg);
        }
    }

    /**
     * 断言：Map不为空
     * @param map Map
     * @param errorCode
     * @param msg
     */
    public static void assertNotEmpty(Map map, BaseErrorCode errorCode, String msg) {
        if (MapUtils.isEmpty(map)) {
            throwValidationException(errorCode, msg);
        }
    }

    /**
     * 断言：Map不为空
     * @param map
     * @param msg
     */
    public static void assertNotEmpty(Map map, String msg) {
        if (MapUtils.isEmpty(map)) {
            throwValidationException(CommonErrorCode.RESULT_EMPTY, msg);
        }
    }

    /**
     * 断言：必须为真
     * @param condition
     * @param errorCode
     * @param msg
     */
    public static void assertTrue(boolean condition, BaseErrorCode errorCode, String msg) {
        if (!condition) {
            throwValidationException(errorCode, msg);
        }
    }

    /**
     * 断言：必须为真
     * @param condition
     * @param msg
     */
    public static void assertTrue(boolean condition, String msg) {
        if (!condition) {
            throwValidationException(CommonErrorCode.RESULT_NOT_EXPECTED, msg);
        }
    }

    /**
     * 断言：必须为真
     * @param condition
     * @param exceptionCode
     */
    public static void assertTrue(boolean condition, BaseErrorCode exceptionCode) {
        assertTrue(condition, exceptionCode, exceptionCode.getMessage());
    }

    /**
     * 断言：必须为真
     * @param condition 条件
     */
    public static void assertTrue(boolean condition) {
        assertTrue(condition, CommonErrorCode.RESULT_NOT_EXPECTED, CommonErrorCode.RESULT_NOT_EXPECTED.getMessage());
    }


    /**
     * 断言：必须相等
     * @param excepted 期望值
     * @param source 实际值
     */
    public static void assertEquals(Object excepted, Object source) {
        assertEquals(excepted, source, CommonErrorCode.RESULT_NOT_EXPECTED, CommonErrorCode.RESULT_NOT_EXPECTED.getMessage());
    }

    /**
     * 断言：必须相等
     * @param excepted 期望值
     * @param source 实际值
     * @param errorCode
     */
    public static void assertEquals(Object excepted, Object source, BaseErrorCode errorCode) {
        assertEquals(excepted, source, CommonErrorCode.RESULT_NOT_EXPECTED, CommonErrorCode.RESULT_NOT_EXPECTED.getMessage());
    }

    /**
     * 断言：必须相等
     * @param excepted 期望值
     * @param source 实际值
     * @param msg
     */
    public static void assertEquals(Object excepted, Object source, String msg) {
        assertEquals(excepted, source, CommonErrorCode.RESULT_NOT_EXPECTED, msg);
    }

    /**
     * 断言：必须相等
     * @param excepted 期望值
     * @param source 实际值
     * @param errorCode
     * @param msg
     */
    public static void assertEquals(Object excepted, Object source, BaseErrorCode errorCode, String msg) {
        if (excepted == source) {
            return;
        }
        if (excepted == null || source == null || !excepted.equals(source)) {
            String throwMsg;
            String defaultMsg = "\tExpected: " + excepted + "\n\tSource: " + source;
            if (msg != null && msg.length() > 0) {
                throwMsg = msg;
            }
            else {
                throwMsg = "\n" + defaultMsg;
            }
            throwParamException(errorCode, throwMsg);
        }
    }

    /**
     * 条件为真时拒绝
     * @param condition 条件
     * @param exceptionCode
     * @param msg
     */
    public static void rejectIf(Boolean condition, BaseErrorCode exceptionCode, String msg) {
        if (condition) {
            throwValidationException(exceptionCode, msg);
        }
    }

    /**
     * 条件为真时拒绝
     * @param condition 条件
     * @param exceptionCode
     */
    public static void rejectIf(Boolean condition, BaseErrorCode exceptionCode) {
        rejectIf(condition, exceptionCode, exceptionCode.getMessage());
    }

    /**
     * 条件为真时拒绝
     * @param condition 条件
     * @param msg
     */
    public static void rejectIf(Boolean condition, String msg) {
        if (condition) {
            throwValidationException(CommonErrorCode.VALID_FAIL, msg);
        }
    }

    /**
     * 验证：参数不为NULL
     * @param obj 参数
     * @param msg
     */
    public static void paramNotNull(Object obj, String msg) {
        if (obj == null) {
            throwParamException(CommonErrorCode.NULL_PARAM, msg);
        }
    }

    /**
     * 验证：参数不为空
     * @param collection 集合
     * @param msg
     */
    public static void paramNotEmpty(Collection collection, String msg) {
        if (CollectionUtils.isEmpty(collection)) {
            throwParamException(CommonErrorCode.EMPTY_PARAM, msg);
        }
    }

    /**
     * 验证：参数不为空
     * @param str 字符串
     * @param msg
     */
    public static void paramNotEmpty(CharSequence str, String msg) {
        if (StringUtils.isEmpty(str)) {
            throwParamException(CommonErrorCode.EMPTY_PARAM, msg);
        }
    }

    /**
     * 验证：参数不为空
     * @param map Map
     * @param msg
     */
    public static void paramNotEmpty(Map map, String msg) {
        if (MapUtils.isEmpty(map)) {
            throwParamException(CommonErrorCode.EMPTY_PARAM, msg);
        }
    }

    /**
     * 验证：参数必须满足某个条件
     * @param condition 条件
     * @param msg
     */
    public static void paramMust(Boolean condition, String msg) {
        if (!condition) {
            throwParamException(CommonErrorCode.ILLEGAL_PARAM, msg);
        }
    }

    private static StackTraceElement getStackElement(int index) {
        StackTraceElement stackTraceElement = Thread.currentThread().getStackTrace()[index];
        return stackTraceElement;
    }

    private static void throwValidationException(BaseErrorCode errorCode, String msg) {
        throwValidationException("数据校验错误", errorCode, msg, 5);
    }

    private static void throwParamException(BaseErrorCode errorCode, String msg) {
        throwValidationException("参数校验错误", errorCode, msg, 5);
    }

    private static void throwValidationException(String type, BaseErrorCode errorCode, String msg, int index) {
        StackTraceElement stackTraceElement = getStackElement(index);
        String class_method = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
        String appId = MetaUtil.APP_NAME;
        if (msg != null) {
            String text = "[" + appId + "] [" + type + ": " + errorCode.getErrorCode() + "] 调用方法[" + class_method + "]时\n\n\t" + errorCode.getMessage() + ":\n\t\t" + msg + "\n\n";
            log.error(text);
            throw new ValidationException(errorCode, errorCode.getMessage() + ": " + msg);
        } else {
            String text = "[" + appId + "] [" + type + ": " + errorCode.getErrorCode() + "] 调用方法[" + class_method + "]时\n\n\t" + errorCode.getMessage() + "\n";
            log.error(text);
            throw new ValidationException(errorCode, errorCode.getMessage());
        }
    }


}
