package com.thebeastshop.common.utils;

import com.thebeastshop.common.enums.FileTypeEnum;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.FileWriterWithEncoding;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 读取文件工具类
 *
 * @author everywhere.z
 * @date 2015-3-5
 * @description
 */
public class FileUtil {

	private final static Logger logger = LoggerFactory.getLogger(FileUtil.class);

	public final static String LINE_SEPARATOR = System.getProperty("line.separator");

	private static final String[] IMAGE_NAMES = {
			"bmp", "dib", "gif", "jfif", "jpe", "jpeg", "jpg", "png", "tif", "tiff", "ico"
	};

	@SuppressWarnings("serial")
	private final static Map<MagicFileNumber, FileTypeEnum> FILE_TYPE = new HashMap<MagicFileNumber, FileTypeEnum>() {
		{
			// 常用文件类型的魔数
			put(new MagicFileNumber("FFD8FF", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.JPG);
			put(new MagicFileNumber("89504E47", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.PNG);
			put(new MagicFileNumber("47494638", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.GIF);
			put(new MagicFileNumber("5249464", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.WAV);
			put(new MagicFileNumber("4944330", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.MP3);
			put(new MagicFileNumber("41564920", MagicNumPositionEnum.front, 0, 0), FileTypeEnum.AVI);
			put(new MagicFileNumber("66747970", MagicNumPositionEnum.middle, 8, 16), FileTypeEnum.MP4);
		}
	};

	private enum MagicNumPositionEnum{
		front,middle,end;
	}

	private static class MagicFileNumber{
		private String magicNumber;

		private MagicNumPositionEnum position;

		private int middleStart;

		private int middleEnd;

		public MagicFileNumber(String magicNumber, MagicNumPositionEnum position, int middleStart, int middleEnd) {
			super();
			this.magicNumber = magicNumber;
			this.position = position;
			this.middleStart = middleStart;
			this.middleEnd = middleEnd;
		}

		public String getMagicNumber() {
			return magicNumber;
		}

		public void setMagicNumber(String magicNumber) {
			this.magicNumber = magicNumber;
		}

		public MagicNumPositionEnum getPosition() {
			return position;
		}

		public void setPosition(MagicNumPositionEnum position) {
			this.position = position;
		}

		public int getMiddleStart() {
			return middleStart;
		}

		public void setMiddleStart(int middleStart) {
			this.middleStart = middleStart;
		}

		public int getMiddleEnd() {
			return middleEnd;
		}

		public void setMiddleEnd(int middleEnd) {
			this.middleEnd = middleEnd;
		}
	}

	public static boolean createDirectory(File file) {
		if (file.exists()) {
			return true;
		}
		return file.mkdirs();
	}

	public static boolean createDirectory(String dirname) {
		return createDirectory(new File(dirname));
	}

	public static String readFile2String(String filename) throws IOException {
		return readFile2String(new File(filename));
	}

	public static String readFile2String(File file) throws IOException {
		if ((file == null) || !file.exists() || file.isDirectory()) {
			return null;
		}
		return readInputStream2String(new FileInputStream(file));
	}

	public static String readInputStream2String(InputStream is) throws IOException {
		return readInputStream2String(is, "UTF-8");
	}

	public static String readInputStream2String(InputStream is, String charset) throws IOException {
		BufferedReader br = null;
		StringBuilder sb = new StringBuilder();
		try {
			br = new BufferedReader(new InputStreamReader(is, charset));
			for (String str = null; (str = br.readLine()) != null;) {
				sb.append(str).append(LINE_SEPARATOR);
			}
		} finally {
			closeIO(br);
		}
		return sb.toString().trim();
	}

	public static List<String> readFile2List(String filename) throws IOException {
		return readFile2List(new File(filename));
	}

	public static List<String> readFile2List(File file) throws IOException {
		if ((file == null) || !file.exists() || file.isDirectory()) {
			return null;
		}
		BufferedReader br = null;
		List<String> list = new ArrayList<String>();
		try {
			br = new BufferedReader(new FileReader(file));
			for (String str = null; (str = br.readLine()) != null;) {
				list.add(str);
			}
		} finally {
			closeIO(br);
		}
		return list;
	}

	public static void writeString2File(String str, String filename) throws IOException {
		writeString2File(str, new File(filename));
	}

	public static void writeString2File(String str, File file) throws IOException {
		BufferedWriter bw = null;
		try {
			bw = new BufferedWriter(new FileWriterWithEncoding(file, "UTF-8"));
			bw.write(str);
			bw.flush();
		} finally {
			closeIO(bw);
		}
	}

	public static void writeList2File(List<String> list, String filename) throws IOException {
		writeList2File(list, new File(filename), LINE_SEPARATOR);
	}

	public static void writeList2File(List<String> list, File file) throws IOException {
		writeList2File(list, file, LINE_SEPARATOR);
	}

	public static void writeList2File(List<String> list, String filename, String lineSeparator) throws IOException {
		writeList2File(list, new File(filename), lineSeparator);
	}

	public static void writeList2File(List<String> list, File file, String lineSeparator) throws IOException {
		StringBuffer sb = new StringBuffer();
		for (int i = 0, k = list.size(); i < k; i++) {
			if (i > 0) {
				sb.append(lineSeparator);
			}
			sb.append(list.get(i));
		}
		writeString2File(sb.toString(), file);
	}

	public static void closeIO(Closeable io) throws IOException {
		if (io != null) {
			io.close();
		}
	}

	public static void zipFile(File inFile, File outFile) throws IOException {
		ZipOutputStream zos = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(outFile)));
		ZipEntry ze = new ZipEntry(inFile.getName());
		zos.putNextEntry(ze);

		byte[] buf = new byte[2048];
		int readLen = 0;
		InputStream is = new BufferedInputStream(new FileInputStream(inFile));
		while ((readLen = is.read(buf, 0, 2048)) != -1) {
			zos.write(buf, 0, readLen);
		}
		is.close();

		zos.close();
	}

	/**
	 * 获得指定文件的byte数组
	 */
	public static byte[] getBytes(String filePath) {
		byte[] buffer = null;
		try {
			File file = new File(filePath);
			FileInputStream fis = new FileInputStream(file);
			ByteArrayOutputStream bos = new ByteArrayOutputStream(1000);
			byte[] b = new byte[1000];
			int n;
			while ((n = fis.read(b)) != -1) {
				bos.write(b, 0, n);
			}
			fis.close();
			bos.close();
			buffer = bos.toByteArray();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return buffer;
	}

	/**
	 * 获取文件类型
	 *
	 * @param filePath
	 * @return
	 */
	public static FileTypeEnum getFileType(String filePath) {
		return getFileType(new File(filePath));
	}

	/**
	 * 获取文件类型
	 *
	 * @param file
	 * @return
	 */
	public static FileTypeEnum getFileType(File file) {
		FileTypeEnum fileType = null;
		try {
			fileType = getFileType(new FileInputStream(file));
		} catch (Exception e) {
			logger.error("获取文件类型异常: {}", e);
		}
		return fileType;
	}

	/**
	 * 获取文件类型
	 *
	 * @param inputStream
	 * @return
	 */
	public static FileTypeEnum getFileType(InputStream inputStream) {
		FileTypeEnum fileType = null;
		// 数据流提取出字节数组
		try {
			byte[] b = new byte[28];
			inputStream.read(b);
			String hex = ByteUtil.bytes2Hex(b).toUpperCase();
			logger.info("文件魔数值: {}", hex);

			String middleHex = null;
			for(Entry<MagicFileNumber, FileTypeEnum> entry : FILE_TYPE.entrySet()) {
				if(entry.getKey().getPosition().equals(MagicNumPositionEnum.front)) {
					if(hex.startsWith(entry.getKey().getMagicNumber())) {
						fileType = entry.getValue();
						break;
					}
				}else if(entry.getKey().getPosition().equals(MagicNumPositionEnum.middle)) {
					middleHex = hex.substring(entry.getKey().getMiddleStart(),entry.getKey().getMiddleEnd());
					if(middleHex.equals(entry.getKey().getMagicNumber())) {
						fileType = entry.getValue();
						break;
					}
				}
			}
		} catch (Exception e) {
			logger.error("获取文件类型异常: {}", e);
		}
		return fileType;
	}

    /**
     * 获取文件扩展名
     *
     * @param fileName 文件名
     * @return
     */
    public static String getExtName(String fileName) {
        if (EmptyUtil.isEmpty(fileName)) {
            return "";
        }
        if (!fileName.contains(".")) {
            return "";
        }
        return fileName.substring(fileName.lastIndexOf(".") + 1);
    }

    /**
	 * 判断一个文件扩展名是否是图片格式
	 *
	 * @param extName 文件扩展名
	 * @return
	 */
	public static boolean isImageExtName(String extName) {
		boolean result = false;
		if (EmptyUtil.isNotEmpty(extName)) {
			if (ArrayUtils.contains(IMAGE_NAMES, extName.toLowerCase()))
				result = true;
		}
		return result;
	}


	/**
	 * 根据文件路径判断其是否存在
	 *
	 * @param filePath
	 * @return
	 */
	public static Boolean exists(String filePath) {
		Boolean result = false;
		if (EmptyUtil.isNotEmpty(filePath)) {
			File file = new File(filePath);
			if (file.exists() && file.canRead()) {
				result = true;
			}
		}
		return result;
	}

	/**
	 * 根据文件路径获取数据
	 *
	 * @param filePath
	 * @return byte[]
	 * @throws IOException
	 */
	public static byte[] getByteFileData(String filePath) throws IOException {
		if (FileUtil.exists(filePath)) {
			File file = new File(filePath);
			return FileUtils.readFileToByteArray(file);
		} else {
			throw new IOException("file is not exist");
		}
	}

	/**
	 * 把一个输入流转成文件
	 *
	 * @param in
	 * @param filePath
	 * @throws IOException
	 */
	public static Boolean InputStreamTOFile(InputStream in, String filePath) throws IOException {
		if (EmptyUtil.isNotEmpty(in) && EmptyUtil.isNotEmpty(filePath)) {
			File file = new File(filePath);
			byte[] b = IOUtils.toByteArray(in);
			FileUtils.writeByteArrayToFile(file, b);
			IOUtils.closeQuietly(in);
			return true;
		}
		return false;
	}

	/**
	 * 重置图片大小
	 *
	 * @param width
	 * @param height
	 * @param filePath
	 * @return
	 * @throws IOException
	 */
	public static Boolean resizeImage(int width, int height, String filePath) throws IOException {
		width = Math.max(width, 1);
		height = Math.max(height, 1);
		Image targetImage = ImageIO.read(new File(filePath));
		BufferedImage image = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_RGB);
		image.getGraphics().drawImage(targetImage, 0, 0, width, height, Color.WHITE, null);
		String formatName = filePath.substring(filePath.lastIndexOf(".") + 1);
		return ImageIO.write(image, formatName, new File(filePath));
	}


	/**
	 * 重置图片大小并补白
	 * @param width
	 * @param height
	 * @param filePath
	 * @param fillWhite
	 */
	public static void resizeImage(int width, int height, String filePath, boolean fillWhite) {
		try {
			double ratio = 0.0; //缩放比例
			File file = new File(filePath);
			BufferedImage tempBufferedImage = ImageIO.read(file);
			Image targetImage = tempBufferedImage.getScaledInstance(width, height, tempBufferedImage.SCALE_SMOOTH);
			//计算比例
			if ((tempBufferedImage.getHeight() > height) || (tempBufferedImage.getWidth() > width)) {
				if (tempBufferedImage.getHeight() > tempBufferedImage.getWidth()) {
					ratio = (new Integer(height)).doubleValue() / tempBufferedImage.getHeight();
				} else {
					ratio = (new Integer(width)).doubleValue() / tempBufferedImage.getWidth();
				}
				AffineTransformOp op = new AffineTransformOp(AffineTransform.getScaleInstance(ratio, ratio), null);
				targetImage = op.filter(tempBufferedImage, null);
			}
			//是否补白
			if (fillWhite) {
				BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
				Graphics2D graphics2D = image.createGraphics();
				graphics2D.setColor(Color.white);
				graphics2D.fillRect(0, 0, width, height);
				if (width == targetImage.getWidth(null)) {
					graphics2D.drawImage(targetImage, 0, (height - targetImage.getHeight(null)) / 2, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
				} else {
					graphics2D.drawImage(targetImage, (width - targetImage.getWidth(null)) / 2, 0, targetImage.getWidth(null), targetImage.getHeight(null), Color.white, null);
				}
				graphics2D.dispose();
				targetImage = image;
			}
			ImageIO.write((BufferedImage) targetImage, "jpg", file);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}


	public static void main(String[] args) {
		FileTypeEnum fileType = getFileType(new File("D:/1.png"));
		System.out.println(fileType);
	}
}
