package com.thebeastshop.common.utils;

import com.thebeastshop.common.enums.BooleanValue;
import com.thebeastshop.common.enums.CodeEnum;
import com.thebeastshop.common.enums.IdEnum;
import com.thebeastshop.common.validation.Validation;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-02-23 20:48
 */
public class EnumUtil {

    private final static ConcurrentHashMap<Class, Map<Object, Object>> enumCodeCache = new ConcurrentHashMap<>();
    private final static ConcurrentHashMap<Class, Map<Object, Object>> enumIdCache = new ConcurrentHashMap<>();
    private final static ConcurrentHashMap<Class, Object[]> booleanValueCache = new ConcurrentHashMap<>();

    /**
     * 动态获取枚举对象属性
     * @param id
     * @param cls
     * @param <T>
     * @return
     */
    public static <T, V> V getPropertyById(Integer id, Class<T> cls, String propertyName) {
        if (id == null) return null;
        try {
            Method method = cls.getMethod("getEnumById", new Class[] {Integer.class});
            if (method == null) return null;
            T result = (T) method.invoke(null,new Object[]{id}).toString();
            if (result == null) return null;
            String getterName = "get" + (propertyName.charAt(0) + "").toUpperCase() + propertyName.substring(1);
            Method getterMethod = cls.getMethod(getterName, new Class[] {});
            if (getterMethod == null) return null;
            return (V) getterMethod.invoke(null, new Object[] {});
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 动态获取枚举对象名称
     * @param id
     * @param cls
     * @param <T>
     * @return
     */
    public static <T> String getNameById(Integer id, Class<T> cls) {
        String result = getPropertyById(id, cls, "name");
        return result == null ? "" : result;
    }


    public static <T extends CodeEnum> T getByName(Class<T> enumClass, String name) {
        Validation.paramNotNull(enumClass,"enumClass为空");
        Validation.paramNotEmpty(name, "name为空");
        T[] arr = getAll(enumClass);
        for (T item : arr) {
            if (name.equals(item.getName())) {
                return item;
            }
        }
        return null;
    }


    public static <T extends BooleanValue> T getByBooleanValue(Class<T> enumClass, Boolean value) {
        Validation.paramNotNull(enumClass,"enumClass为空");
        Validation.paramNotNull(value, "value为空");
        Object[] values = booleanValueCache.get(enumClass);
        if (values == null) {
            synchronized (enumClass) {
                if (!booleanValueCache.contains(enumClass)) {
                    T[] arr = getAll(enumClass);
                    values = new Object[2];
                    for (T item : arr) {
                        values[item.toBooleanValue() ? 0 : 1] = item;
                    }
                    booleanValueCache.put(enumClass, values);
                }
            }
        }
        return  (T) values[value ? 0 : 1];
    }


    public static <C, T extends CodeEnum<C>> T getByCode(Class<T> enumClass, C code) {
        Validation.paramNotNull(enumClass,"enumClass为空");
        Validation.paramNotNull(code, "code为空");
        Map<Object, Object> map = enumCodeCache.get(enumClass);
        if (map == null) {
            synchronized (enumClass) {
                if (!enumCodeCache.contains(enumClass)) {
                    T[] arr = getAll(enumClass);
                    map = new HashMap<>();
                    for (T item : arr) {
                        map.put(item.getCode(), item);
                    }
                    enumCodeCache.put(enumClass, map);
                }
            }
        }
        return  (T) map.get(code);
    }


    public static <T extends IdEnum, ID> T getById(Class<T> enumClass, ID id) {
        Validation.paramNotNull(enumClass,"enumClass为空");
        Validation.paramNotNull(id, "id为空");
        Map<Object, Object> map = enumIdCache.get(enumClass);
        if (map == null) {
            synchronized (enumClass) {
                if (!enumIdCache.contains(enumClass)) {
                    T[] arr = getAll(enumClass);
                    map = new HashMap<>();
                    for (T item : arr) {
                        map.put(item.getId(), item);
                    }
                    enumIdCache.put(enumClass, map);
                }
            }
        }
        return (T) map.get(id);
    }



    public static <T extends Enum> List<T> getAllAsList(Class<T> enumClass) {
        T[] arr = getAll(enumClass);
        List<T> list = new ArrayList<>();
        for (T item : arr) {
            list.add(item);
        }
        return list;
    }

    public static <T> T[] getAll(Class<T> enumClass) {
        if (!Enum.class.isAssignableFrom(enumClass)) {
            throw new IllegalArgumentException("参数必须为枚举类Class");
        }
        try {
            Method valuesMethod = enumClass.getDeclaredMethod("values", new Class[0]);
            Object values = valuesMethod.invoke(enumClass);
            Validation.assertTrue(values.getClass().isArray(),
                    "" + enumClass.getName() + ".values() 结果和预期不同");
            return (T[])values;
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }


}
