package com.thebeast.kit.alitool;

import com.aliyun.oss.ClientException;
import com.aliyun.oss.HttpMethod;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.*;
import com.thebeastshop.common.enums.FileTypeEnum;
import com.thebeastshop.common.utils.DateUtil;
import com.thebeastshop.common.utils.EmptyUtil;
import com.thebeastshop.common.utils.StringUtil;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * @author Paul-xiong
 * @date 2017年7月25日
 * @description 文件上传操作类
 */
public class FileUploadClient {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /**
     * 基础路径
     */
    private static final String BASE_PATH = "file/";
    private static final String KEY_PDF = "pdf/";
    private OSSClient ossClient;

    private String ossEndPoint;

    private String ossAccessId;

    private String ossAccessKey;

    private String ossBucketName;

    private String baseUrl;

    private final String KEY_IMG = "image/";

    private static final String KEY_FILE = "document/";

    /**
     * 获得oss的文件key
     *
     * @param fileUrl
     * @return
     */
    private String getKey(String fileUrl) {
        return fileUrl.replace(baseUrl, "");
    }

    /**
     * 处理额外路径
     *
     * @param extraPath
     * @return
     */
    private String buildExtraPath(String extraPath) {
        if (StringUtils.isNotBlank(extraPath)) {
            extraPath = extraPath.indexOf("/") == 0 ? extraPath.substring(1) : extraPath;
            extraPath += extraPath.lastIndexOf("/") == -1 ? "/" : "";
        } else {
            extraPath = "";
        }
        return extraPath;
    }

    /**
     * 获取ossClient
     *
     * @return
     */
    public OSSClient getOssClient() {
        if (ossClient == null) {
            ossClient = new OSSClient(ossEndPoint, ossAccessId, ossAccessKey);
        }
        return ossClient;
    }

    // --------------------getter setter----------------------

    public String getOssEndPoint() {
        return ossEndPoint;
    }

    public void setOssEndPoint(String ossEndPoint) {
        this.ossEndPoint = ossEndPoint;
    }

    public String getOssAccessId() {
        return ossAccessId;
    }

    public void setOssAccessId(String ossAccessId) {
        this.ossAccessId = ossAccessId;
    }

    public String getOssAccessKey() {
        return ossAccessKey;
    }

    public void setOssAccessKey(String ossAccessKey) {
        this.ossAccessKey = ossAccessKey;
    }

    public String getOssBucketName() {
        return ossBucketName;
    }

    public void setOssBucketName(String ossBucketName) {
        this.ossBucketName = ossBucketName;
    }

    // -------------------------------------------------------

    public String getBaseUrl() {
        return baseUrl;
    }

    public void setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    /**
     * 上传文件
     *
     * @param localFilePath 本地文件路径
     * @param extraPath     上传的文件自定义路径
     * @param fileType      文件类型
     * @return
     */
    public FileUploadResp uploadFile(String localFilePath, String extraPath, FileTypeEnum fileType) {
        File file = new File(localFilePath);
        return this.uploadFile(file, extraPath, fileType);
    }

    public String uploadImg(String suffix, InputStream content) {
        String fileName = DateUtil.formatDate(new Date(), "yyyyMMddHHmmss")
                + StringUtil.getRandomString(6, StringUtil.randomInts);

        return uploadImg(fileName, suffix, content);
    }

    public String uploadImg(String fileName, String suffix, InputStream content) {
        String key = KEY_IMG + fileName + "." + suffix;

        // 创建上传Object的Metadata
        ObjectMetadata meta = new ObjectMetadata();

        // 必须设置ContentLength
        //        meta.setContentLength(file.length());

        MimeType mimeType = MimeType.findByName(suffix);
        if (mimeType != null) {
            meta.setContentType(mimeType.getContentType());
        }
        logger.info("-------ossBucketName:" + ossBucketName);
        getOssClient().putObject(ossBucketName, key, content,meta);
        return key;
    }
    public String uploadFile(String subdirectory,String fileName, String suffix, InputStream content) {

        String key = KEY_FILE + (EmptyUtil.isNotEmpty(subdirectory) ? (subdirectory + File.separator) : "") + fileName + "." + suffix;
        // 创建上传Object的Metadata
        ObjectMetadata meta = new ObjectMetadata();
        // 必须设置ContentLength
        //        meta.setContentLength(file.length());

        MimeType mimeType = MimeType.findByName(suffix);
        if (mimeType != null) {
            meta.setContentType(mimeType.getContentType());
        }

        getOssClient().putObject(ossBucketName, key, content,meta);
        return key;
    }

    public String uploadFileNew(String subdirectory,String fileName, String suffix, InputStream content) throws URIException, UnsupportedEncodingException {
        String reNameFileName =  "uploadFile" + System.currentTimeMillis();
        StringBuilder keySb = new StringBuilder().append(KEY_FILE).append((EmptyUtil.isNotEmpty(subdirectory) ? (subdirectory + File.separator) : ""))
                .append(reNameFileName).append(".").append(suffix);
        // 创建上传Object的Metadata
        ObjectMetadata meta = new ObjectMetadata();
        MimeType mimeType = MimeType.findByName(suffix);
        if (mimeType != null) {
            meta.setContentType(mimeType.getContentType());
        }

        String endCodefileName = new StringBuilder().append(URLEncoder.encode(fileName, StandardCharsets.UTF_8.name())).append(".").append(suffix).toString();
        String  disposition = String.format("attachment;filename=%s;filename*=UTF-8''%s", endCodefileName, endCodefileName);
        meta.setContentDisposition(disposition);

        PutObjectResult putObjectResult = getOssClient().putObject(ossBucketName, keySb.toString(), content,meta);
        return keySb.toString();
    }

    public FileInfo uploadFile(InputStream inputStream, String extraPath) throws IOException {
        try {
            // 数据流提取出字节数组
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            int len;
            while ((len = inputStream.read(buffer)) > -1) {
                baos.write(buffer, 0, len);
            }
            ByteArrayInputStream inputStreamc = new ByteArrayInputStream(baos.toByteArray());

            // check文件类型
            byte[] b = Arrays.copyOf(baos.toByteArray(), 28);
            String hex = bytes2Hex(b).toUpperCase();
            Set<String> keys = FileConstant.FILE_TYPE.keySet();
            ObjectMetadata meta = new ObjectMetadata();
            String extraPaths = extraPath.substring(extraPath.lastIndexOf(".") + 1, extraPath.length());
            logger.info("extraPaths={}", extraPaths);
            MimeType mimeType = MimeType.findByName(extraPaths);
            if (mimeType != null) {
                meta.setContentType(mimeType.getContentType());
            }
            getOssClient().putObject(ossBucketName, extraPath, inputStreamc, meta);

            FileInfo info = new FileInfo(inputStream, extraPath);
            inputStream.close();
            inputStreamc.close();
            baos.flush();
            baos.close();
            return info;

        } catch (Exception e) {
            logger.error("上传文件异常: {}", e);
            inputStream.close();
            return null;
        }
    }
    public static String bytes2Hex(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        String tmp = null;
        for (byte b : bytes) {
            // 将每个字节与0xFF进行与运算，然后转化为10进制，然后借助于Integer再转化为16进制
            tmp = Integer.toHexString(0xFF & b);
            if (tmp.length() == 1)// 每个字节8为，转为16进制标志，2个16进制位
            {
                tmp = "0" + tmp;
            }
            sb.append(tmp);
        }

        return sb.toString();
    }
    /**
     * 上传文件
     *
     * @param file      文件类
     * @param extraPath 上传的文件自定义路径
     * @param fileType  文件类型
     * @return
     */
    public FileUploadResp uploadFile(File file, String extraPath, FileTypeEnum fileType) {
        FileUploadResp info = null;
        try {
            info = this.uploadFile(new FileInputStream(file), extraPath, fileType);
        } catch (Exception e) {
            logger.error("上传文件失败: {}", e);
        }
        return info;
    }

    /**
     * 上传文件
     *
     * @param inputStream 文件流
     * @param extraPath   上传的文件自定义路径
     * @param fileType    文件类型
     * @return
     */
    public FileUploadResp uploadFile(InputStream inputStream, String extraPath, FileTypeEnum fileType) {
        FileUploadResp info = null;
        try {
            info = this.uploadFile(inputStream, extraPath, fileType, "");
        } catch (Exception e) {
            logger.error("上传文件异常: {}", e);
        }
        return info;
    }

    /**
     * 上传文件
     *
     * @param inputStream
     * @param extraPath
     * @param fileType
     * @return
     */
    public FileUploadResp uploadFile(InputStream inputStream, String extraPath, String fileType) {
        return uploadFile(inputStream, extraPath, fileType, "");
    }

    /**
     * 上传文件
     *
     * @param file
     * @param extraPath
     * @param fileType
     * @param fileName
     * @return
     */
    public FileUploadResp uploadFile(File file, String extraPath, FileTypeEnum fileType, String fileName) {
        FileUploadResp info = null;
        try {
            info = this.uploadFile(new FileInputStream(file), extraPath, fileType, fileName);
        } catch (Exception e) {
            logger.error("上传文件异常: {}", e);
        }
        return info;
    }

    public FileUploadResp uploadFile(InputStream inputStream, String extraPath, FileTypeEnum fileType, String fileName) {
        FileUploadResp info = null;
        try {
            if (StringUtils.isBlank(fileName)) {
                fileName = UUID.randomUUID().toString().replace("-", "");
            }
            String key = BASE_PATH + this.buildExtraPath(extraPath) + fileName + "." + fileType.getName();
            getOssClient().putObject(ossBucketName, key, inputStream);
            info = new FileUploadResp(inputStream, baseUrl + key);
        } catch (Exception e) {
            logger.error("上传文件异常: {}", e);
        }
        return info;
    }

    /**
     * 上传文件
     *
     * @param inputStream
     * @param extraPath
     * @param fileType
     * @param fileName
     * @return
     */
    public FileUploadResp uploadFile(InputStream inputStream, String extraPath, String fileType, String fileName) {
        FileTypeEnum fileTypeEnum = FileTypeEnum.getEnumByName(fileType);
        if (fileTypeEnum == null) {
            throw new RuntimeException("文件类型不支持");
        }
        return uploadFile(inputStream, extraPath, fileTypeEnum, fileName);
    }

    /**
     * 获取文件列表
     *
     * @param extraPath 上传的文件自定义路径
     * @return
     */
    public List<FileUploadResp> listFiles(String extraPath) {
        List<FileUploadResp> list = null;
        try {
            ObjectListing listing = getOssClient().listObjects(ossBucketName, BASE_PATH + this.buildExtraPath(extraPath));
            list = new LinkedList<FileUploadResp>();
            for (OSSObjectSummary objectSummary : listing.getObjectSummaries()) {
                list.add(this.getFile(baseUrl + objectSummary.getKey()));
            }
        } catch (Exception e) {
            logger.error("获取文件列表异常: {}", e);
        }
        return list;
    }

    /**
     * 获取远程目录下的所有文件（排除目录）
     * @param remoteDir
     * @return
     */
    public List<String> listFileExcludeDir(String remoteDir) {
        List<String> keyList = new ArrayList<>();
        try {
            ossClient = ossClient == null ? getOssClient() : ossClient;
            // 列出目录下所有文件
            ObjectListing listing = ossClient.listObjects(ossBucketName, remoteDir);
            for (OSSObjectSummary objectSummary : listing.getObjectSummaries()) {
                String objectKey = objectSummary.getKey();
                // 跳过目录本身
                if (objectKey.endsWith("/")) {
                    continue;
                }
                OSSObject ossObject =  ossClient.getObject(
                        new GetObjectRequest(ossBucketName, objectKey)
                );
                logger.info(objectKey);
                keyList.add(objectKey);
            }
        } catch (Exception e) {
            logger.error("获取文件列表异常: {}", e);
        }
        return keyList;
    }

    /**
     * 获取上传文件
     *
     * @param fileUrl 文件路径
     * @return
     */
    public FileUploadResp getFile(String fileUrl) {
        FileUploadResp info = null;
        try {
            OSSObject ossObject = getOssClient().getObject(ossBucketName, this.getKey(fileUrl));
            info = new FileUploadResp(ossObject.getObjectContent(), fileUrl);
        } catch (Exception e) {
            logger.error("获取上传文件异常: {}", e);
        }
        return info;
    }

    /**
     * 删除上传文件
     *
     * @param fileUrl 文件路径
     */
    public void deleteFile(String fileUrl) {
        try {
            String key = this.getKey(fileUrl);
            getOssClient().deleteObject(ossBucketName, key);
        } catch (Exception e) {
            logger.error("删除上传文件异常: {}", e);
        }
    }

    public String uploadPdf(String fileName, String suffix, InputStream content) {
        String key = KEY_PDF + fileName + "." + suffix;

        // 创建上传Object的Metadata
        ObjectMetadata meta = new ObjectMetadata();

        // 必须设置ContentLength
        //        meta.setContentLength(file.length());

        MimeType mimeType = MimeType.findByName(suffix);
        if (mimeType != null) {
            meta.setContentType(mimeType.getContentType());
        }

        getOssClient().putObject(ossBucketName,key, content, meta);

        return key;
    }

    public URL generateSignedUrl(String key) {
        Date expiration = new Date(new Date().getTime() + 3600 * 1000L);
        // 生成签名URL (2.x版本的API略有不同)
        GeneratePresignedUrlRequest request = new GeneratePresignedUrlRequest(ossBucketName, key);
        request.setExpiration(expiration);
        // 2.x版本需要显式设置HTTP方法
        request.setMethod(HttpMethod.GET);
        URL signedUrl = getOssClient().generatePresignedUrl(request);

        return signedUrl;
    }
    public OSSObject downloadByKey(String key) {
        return getOssClient().getObject(ossBucketName, key);
    }


    public OSSObject download(String key) {
        return getOssClient().getObject(ossBucketName, key);
    }


}
