package com.thebeastshop.common.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * 日期工具类
 *
 * @author bryan.zhang
 */
public class DateUtil {

    private static Logger logger = LoggerFactory.getLogger(DateUtil.class);

    /** yyyy-MM-dd */
    public static final String FORMAT_YYYY_MM_DD = "yyyy-MM-dd";

    /** yyyyMMdd */
    public static final String FORMAT_YYYYMMDD = "yyyyMMdd";

    /** yyMMdd */
    public static final String FORMAT_YYMMDD = "yyMMdd";

    /** yyyy-MM-dd HH:mm:ss */
    public static final String FORMAT_YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    /** yyyyMMddHHmmss */
    public static final String FORMAT_YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

    /** yyyyMMddHHmm */
    public static final String FORMAT_YYYYMMDDHHMM = "yyyyMMddHHmm";

    /** HH:mm:ss */
    public static final String FORMAT_HH_MM_SS = "HH:mm:ss";

    /** HHmmss */
    public static final String FORMAT_HHMMSS = "HHmmss";

    /** dd */
    public static final String FORMAT_DD = "dd";

    /** MMdd */
    public static final String FORMAT_MMDD = "MMdd";

    public static final String FORMAT_YYYY_MM_DD_HH_MM = "yyyy-MM-dd HH:mm";

    public static final String FORMAT_UTC = "yyyy-MM-dd'T'HH:mm:ss'Z'";

    public static final String FORMAT_MICROSECOND = "yyyyMMddHHmmssmmm";
    public static final String FORMAT_DATE_2 = "yyyyMMdd";
    public static final String FORMAT_DATE_3 = "MM/dd/yyyy";
    public static final String FORMAT_DATE_4 = "MM/dd/yyyy HH:mm";
    public static final String FORMAT_MM_DD = "MM-dd";
    public static final String FORMAT_M_YUE_D_RI = "M月d日";

    public static final String MONGOSTRINGTODATE = "EEE MMM dd HH:mm:ss zzz yyyy";

    /**
     * 将日期转换成字符格式
     *
     * @param date
     *            java.util.Date类型
     * @param format
     *            如果为null或""，默认为DATE格式
     * @return 无法成功转换则返回null
     */
    public static String date2String(Date date, String format) {
        String result = null;
        if (date == null) {
            return result;
        }
        if (StringUtils.isEmpty(format)) {
            format = FORMAT_YYYY_MM_DD;
        }
        try {
            result = formatDate(date, format);
        } catch (Exception ex) {
            logger.warn("日期转换为字符串错误，日期：" + date.toString() + "， 格式：" + format);
        }

        return result;
    }



    public static String workDays(String strStartDate, String strEndDate) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        Calendar cl1 = Calendar.getInstance();
        Calendar cl2 = Calendar.getInstance();

        try {
            cl1.setTime(df.parse(strStartDate));
            cl2.setTime(df.parse(strEndDate));

        } catch (ParseException e) {
            logger.warn("日期格式非法");
            e.printStackTrace();
        }

        Long time =  cl2.getTimeInMillis() - cl1.getTimeInMillis();

        int count = 0;
        while (cl1.compareTo(cl2) <= 0) {
            if (cl1.get(Calendar.DAY_OF_WEEK) != 7 && cl1.get(Calendar.DAY_OF_WEEK) != 1)
                count++;
            cl1.add(Calendar.DAY_OF_MONTH, 1);
        }


        int nDays =  nDaysBetweenTwoDate(strStartDate, strEndDate);
        int minute = (int) (time/(1000 * 60));

        int differenceHours = (minute - 60 * 24 * nDays)/60;

        return count - 1 + "天" + Math.abs(differenceHours) + "小时";

    }


    // 计算两个日期相隔的天数
    public static int nDaysBetweenTwoDate(String firstString, String secondString) {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date firstDate = null;
        Date secondDate = null;
        try {
            firstDate = df.parse(firstString);
            secondDate = df.parse(secondString);
        } catch (Exception e) {
            // 日期型字符串格式错误
            logger.warn("日期型字符串格式错误");
        }
        int nDay = (int) ((secondDate.getTime() - firstDate.getTime()) / (24 * 60 * 60 * 1000));
        return nDay;
    }



    /**
     * 字符串 转 日期
     *
     * @param dateString
     *            日期字符串
     * @return 日期对象
     */
    public static final Date parseDate_YYYYMMDD(String dateString) {
        return parseDate(dateString, FORMAT_YYYYMMDD);
    }

    public static final Date parseDate_YYYYMMDDHHMM(String dateString) {
        return parseDate(dateString, FORMAT_YYYYMMDDHHMM);
    }

    public static final Date parseDate_YYYYMMDDHHMMSS(String dateString) {
        return parseDate(dateString, FORMAT_YYYYMMDDHHMMSS);
    }

    public static final Date parseDate(String dateString, String pattern) {
        Date date = null;
        if (StringUtils.isEmpty(dateString))
            return null;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            date = sdf.parse(dateString);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
        }
        return date;
    }


    /**
     * 日期对象格式化
     *
     * @param date
     *            日期对象
     * @param pattern
     *            格式化样式
     * @return 字符串日期
     */
    public static final String formatDate(Date date, String pattern) {
        String v = null;
        try {
            if (date == null)
                return null;
            SimpleDateFormat dateFormat = new SimpleDateFormat(pattern);
            v = dateFormat.format(date);
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    /**
     * 日期对象格式化
     *
     * @param date
     *            日期对象
     * @return 字符串日期
     */
    public static final String formatDate_YYYY_MM_DD(Date date) {
        String v = null;
        try {
            if (date == null)
                return null;
            SimpleDateFormat dateFormat = new SimpleDateFormat(FORMAT_YYYY_MM_DD);
            v = dateFormat.format(date);
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    public static final String formatDate_YYYYMMDD(Date date) {
        String v = null;
        try {
            if (date == null)
                return null;
            SimpleDateFormat dateFormat = new SimpleDateFormat(FORMAT_YYYYMMDD);
            v = dateFormat.format(date);
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    /**
     * 判断传入的日期是否是今天
     * <p>
     * 2002-03-28 13:45 和 2002-03-28 返回 true. 2002-03-12 13:45 和 2002-03-28 返回
     * false.
     * </p>
     *
     * @param date
     *            需要判断的日期对象
     * @return 是今天返回true,否则返回false
     */
    public static boolean isToday(Date date) {
        if (date == null)
            return false;
        return DateUtils.isSameDay(date, new Date());
    }


    /**
     * 判断时间是否早于当前时间
     * <p>
     * 例如:今天是2002-03-28 13:45:00, 传入2002-03-27 11:00:00 返回true,传入2002-03-28
     * 11:00:00返回false
     * </p>
     *
     * @param date
     *            需要判断的时间对象
     * @return 如果传入的时间对象小于今天(时分秒都为0),则返回true,否则返回false
     */
    public static boolean isBeforeToday(Date date) {
        if (date == null)
            return false;
        long today = DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH).getTime();
        return date.getTime() < today;
    }


    /**
     * 判断时间是否早于当前时间
     * <p>
     * 例如:今天是2002-03-28 13:45:00, 传入2002-03-27 11:00:00 返回true,传入2002-03-28
     * 11:00:00返回false
     * </p>
     *
     * @param dateString
     *            日期字符串
     * @param sFormat
     *            日期字符串的原始格式,例如yyyy-MM-dd
     * @return
     */
    public static boolean isBeforeToday(String dateString, String sFormat) {
        return isBeforeToday(parseDate(dateString, sFormat));
    }


    /**
     * 日期字符串之间的类型转换.
     * <p>
     * 例如:convertDate("2012-01-02", "yyyy-MM-dd", "yyyy/mm/dd")返回2012/01/02
     * </p>
     *
     * @param source
     *            待处理的日期字符串
     * @param sformat
     *            原来的格式
     * @param dformat
     *            新的格式
     * @return 转换后的日期字符串
     */
    public static String convertDate(String source, String sformat, String dformat) {
        // 参数检查
        if (StringUtils.isEmpty(source) || StringUtils.isEmpty(sformat) || StringUtils.isEmpty(dformat))
            return source;
        // 开始转换
        String newString = formatDate(parseDate(source, sformat), dformat);
        // 如果转换失败返回原始字符串
        return (newString == null) ? source : newString;
    }


    /**
     * 获得当前日期字符串,格式为 yyyyMMdd
     *
     * @return 当前日期字符串(yyyyMMdd)
     */
    public static String getCurrDate_HHMMSS() {
        return formatDate(new Date(), FORMAT_HHMMSS);
    }


    public static String getDate_DD(Date date) {
        return formatDate(date, FORMAT_DD);
    }


    /**
     * 获得当前日期字符串,格式为 yyyyMMdd
     *
     * @return 当前日期字符串(yyyyMMdd)
     */
    public static String getCurrDate_YYYYMMDD() {
        return formatDate(new Date(), FORMAT_YYYYMMDD);
    }


    /**
     * 获得当前日期字符串,格式为 yyyy-MM-dd
     *
     * @return 当前日期字符串(yyyy-MM-dd)
     */
    public static String getCurrDate_YYYY_MM_DD() {
        return formatDate(new Date(), FORMAT_YYYY_MM_DD);
    }


    /**
     * 获得当前日期字符串,格式为 yyyy-MM-dd HH:mm:ss
     *
     * @return 当前日期字符串(yyyy-MM-dd HH:mm:ss)
     */
    public static String getCurrDate_YYYY_MM_DD_HH_MM_SS() {
        return formatDate(new Date(), FORMAT_YYYY_MM_DD_HH_MM_SS);
    }


    /**
     * 获得当前日期字符串,格式为 yyyyMMddHHmmss
     *
     * @return 当前日期字符串(yyyyMMddHHmmss)
     */
    public static String getCurrTime_YYYYMMDDHHMMSS() {
        return formatDate(new Date(), FORMAT_YYYYMMDDHHMMSS);
    }


    public static int getYesterDay() {
        Date yesterday = calDate(new Date(), 0, 0, -1);
        return Integer.parseInt(getDate_DD(yesterday));
    }


    public static String getYesterDay_MMDD() {
        Date yesterday = calDate(new Date(), 0, 0, -1);
        return formatDate(yesterday, FORMAT_MMDD);
    }


    public static String getYesterDay_YYYYMMDD() {
        Date yesterday = calDate(new Date(), 0, 0, -1);
        return formatDate(yesterday, FORMAT_YYYYMMDD);
    }


    public static Date getLastMonthDate(int day) {
        Date lastMonth = calDate(new Date(), 0, -1, 0);
        Calendar cal = Calendar.getInstance();
        cal.setTime(lastMonth);
        if (day == 31) {
            cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
        } else {
            cal.set(Calendar.DATE, day);
        }
        return cal.getTime();
    }


    public static Date getCurMonthDate(int day) {
        Date cur = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(cur);
        if (day == 31) {
            cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
        } else {
            cal.set(Calendar.DATE, day);
        }
        return cal.getTime();
    }


    /**
     * 判断是否是日期
     *
     * @return dateString 日期字符串
     * @param pattern
     *            格式化样式(yyyyMMddHHmmss)
     * @return 是日期返回true, 否则返回false
     */
    public static boolean isDateString(String dateString, String pattern) {
        boolean v = false;
        try {
            if (StringUtils.isNotEmpty(dateString)) {
                SimpleDateFormat dateFormat = new SimpleDateFormat(pattern);
                Date d = dateFormat.parse(dateString);
                if (d != null) {
                    v = true;
                }
            }
        } catch (Exception e) {
            // do nothing
        }
        return v;
    }


    /**
     * 判断是否是日期
     *
     * @return dateString 日期字符串
     * @param pattern
     *            格式化样式(yyyyMMddHHmmss)
     * @return 不是日期返回true, 否则返回false
     */
    public static boolean isNotDateString(String dateString, String pattern) {
        return !isDateString(dateString, pattern);
    }


    /**
     * 计算日期
     *
     * @param date
     * @param yearNum
     * @param monthNum
     * @param dateNum
     * @return
     */
    public static String calDate(String date, int yearNum, int monthNum, int dateNum) {
        String result = "";
        try {
            SimpleDateFormat sd = new SimpleDateFormat(FORMAT_YYYYMMDD);
            Calendar cal = Calendar.getInstance();
            cal.setTime(sd.parse(date));
            cal.add(Calendar.MONTH, monthNum);
            cal.add(Calendar.YEAR, yearNum);
            cal.add(Calendar.DATE, dateNum);
            result = sd.format(cal.getTime());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }


    public static Date calDate(Date date, int yearNum, int monthNum, int dateNum) {
        Date result = null;
        try {
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            cal.add(Calendar.MONTH, monthNum);
            cal.add(Calendar.YEAR, yearNum);
            cal.add(Calendar.DATE, dateNum);
            result = cal.getTime();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    public static Date addDate(Date date,int dateNum){
    	return calDate(date, 0, 0, dateNum);
    }


    /**
     * 增加或减少指定数量的时间
     *
     * @param date   时间
     * @param field  计算域
     * @param amount 数值
     */
    public static Date add(Date date, int field, int amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(field, amount);
        return calendar.getTime();
    }


    /**
     * 增加或减少指定天数
     *
     * @param date
     * @param amount
     * @return
     */
    public static Date addDay(Date date, int amount) {
        return add(date, Calendar.DAY_OF_YEAR, amount);
    }

    public static Date addMin(Date date, int min) {
        Date result = null;
        try {
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            cal.add(Calendar.MINUTE,min);
            result = cal.getTime();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * 获得指定时间当天 00:00:00 的Date对象
     *
     * @param date 指定时间
     * @return 结果
     */
    public static Date dayStart(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND,0);
        return calendar.getTime();
    }

    /**
     * 获得指定时间当天 23:59:59 的Date对象
     *
     * @param date 指定时间
     * @return 结果
     */
    public static Date dayEnd(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND,999);
        return calendar.getTime();
    }


    public static Date getDate(long time){
    	return new Date(time);
    }

	/**
	 * 判断是否是明天
	 *
	 * @param date
	 * @return
	 */
	public static boolean isTommorow(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(new Date());
		c.set(Calendar.DATE, c.get(Calendar.DATE) + 1);
		Date yesterday = c.getTime();
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		return format.format(yesterday).equals(format.format(date));
	}

	/**
	 * 判断是否是昨天
	 *
	 * @param date
	 * @return
	 */
	public static boolean isYesterday(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(new Date());
		c.set(Calendar.DATE, c.get(Calendar.DATE) - 1);
		Date yesterday = c.getTime();
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		return format.format(yesterday).equals(format.format(date));
	}

	public static Date timeStamp2Date(Integer timestampString){
		Long timestamp = Long.parseLong(timestampString.toString())*1000;
		Date date = new Date(timestamp);
		return date;
	}

	public static boolean isOverlap(Date leftStartDate, Date leftEndDate,Date rightStartDate, Date rightEndDate) {
        return
            ((leftStartDate.getTime() >= rightStartDate.getTime())
                    && leftStartDate.getTime() < rightEndDate.getTime())
            ||
            ((leftStartDate.getTime() > rightStartDate.getTime())
                    && leftStartDate.getTime() <= rightEndDate.getTime())
            ||
            ((rightStartDate.getTime() >= leftStartDate.getTime())
                    && rightStartDate.getTime() < leftEndDate.getTime())
            ||
            ((rightStartDate.getTime() > leftStartDate.getTime())
                    && rightStartDate.getTime() <= leftEndDate.getTime());
    }

    public static Date parseUTCDate(String date, String pattern){
        try{
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setTimeZone(TimeZone.getTimeZone("UTC"));
            Date resultDate = sdf.parse(date);
            return resultDate;
        }catch (Throwable t){
            //do nothing
        }
        return null;
    }

    public static Date parseUTCDate(String date){
        return parseUTCDate(date,FORMAT_UTC);
    }
}
