package com.thebeastshop.common.spring;

import com.thebeastshop.common.converter.AfterCopyProperties;
import com.thebeastshop.common.converter.BeanConverterHandler;
import com.thebeastshop.common.converter.ConverterSupport;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.core.Ordered;
import org.springframework.core.PriorityOrdered;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-08-22 18:06
 */
public class BeanConverterManager implements BeanPostProcessor, PriorityOrdered {
    private static final Logger log = LoggerFactory.getLogger(BeanConverterManager.class);

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        processMethods(bean);
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    private void processMethods(final Object bean) {
        Class clazz = bean.getClass();
        Method[] methods = clazz.getDeclaredMethods();
        for (final Method method : methods) {
            AfterCopyProperties ann = method.getAnnotation(AfterCopyProperties.class);
            if (ann == null) continue;
            Class[] paramTypes = method.getParameterTypes();
            if (paramTypes.length != 2) {
                throw new RuntimeException("自定义类型转换方法参数数量必须为2个！");
            }
            String description = ann.value();
            String logDesc = "";
            if (StringUtils.isNotBlank(description)) {
                logDesc = " [" + description + "] ";
            }
            log.info("[自定义类型转换] 扫描到自定义类型转换方法 " + logDesc +"[" + clazz.getName() + "." + method.getName() + "]");
            ReflectionUtils.makeAccessible(method);
            ConverterSupport.from(paramTypes[0]).to(paramTypes[1]).afterCopy(new BeanConverterHandler() {
                @Override
                public void copyFields(Object sourceObj, Object targetObj) {
                    try {
                        method.invoke(bean, sourceObj, targetObj);
                    } catch (IllegalAccessException e) {
                        throw new RuntimeException(e);
                    } catch (InvocationTargetException e) {
                        throw new RuntimeException(e);
                    }
                }
            });
        }
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }
}
