package com.thebeastshop.common.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.thebeastshop.common.cat.RestCat;
import org.apache.commons.lang3.StringUtils;
import org.unidal.helper.Joiners;

import com.dianping.cat.Cat;
import com.dianping.cat.CatConstants;
import com.dianping.cat.configuration.client.entity.Server;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.DefaultMessageManager;
import com.dianping.cat.message.internal.DefaultTransaction;

/**
 * 野兽派CAT监控系统Servlet拦截器 </BR>
 * 主要实现了CAT监控对于HTTP请求的打点处理
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-05-23 16:51
 */
public class BeastCatFilter implements Filter {

    private List<BeastCatFilter.Handler> m_handlers = new ArrayList<BeastCatFilter.Handler>();

    private final static Map<String, String> m_patterns = new LinkedHashMap<>();

    @Override
    public void destroy() {
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException,
            ServletException {
		try {
			BeastCatFilter.Context ctx = new BeastCatFilter.Context((HttpServletRequest) request, (HttpServletResponse) response, chain, m_handlers);

			ctx.handle();
		} catch (Exception e) {
			System.out.println("BeastCatFilter异常：" + e);
		}
    }

    protected String getOriginalUrl(ServletRequest request) {
        return ((HttpServletRequest) request).getRequestURI();
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        m_handlers.add(BeastCatFilter.CatHandler.ENVIRONMENT);
        m_handlers.add(BeastCatFilter.CatHandler.LOG_SPAN);
        m_handlers.add(BeastCatFilter.CatHandler.LOG_CLIENT_PAYLOAD);
        m_handlers.add(BeastCatFilter.CatHandler.ID_SETUP);

        String patterns = filterConfig.getInitParameter("cat-uri-pattern");
        if (StringUtils.isNotBlank(patterns)) {
            String[] params = patterns.split(",");
            for (int i = 0, len = params.length; i < len; i++) {
                String nameValue = params[i];
                String[] args = nameValue.split("=");
                if (args != null && args.length > 1) {
                    String name = args[0];
                    String pattern = args[1];
                    m_patterns.put(name, pattern);
                }
            }
        }
    }


    private static enum CatHandler implements BeastCatFilter.Handler {
        ENVIRONMENT {

            @Override
            public void handle(BeastCatFilter.Context ctx) throws IOException, ServletException {
                HttpServletRequest req = ctx.getRequest();
                boolean top = !Cat.getManager().hasContext();

                ctx.setTop(top);

                if (top) {
                    ctx.setType(CatConstants.TYPE_URL);

                    setTraceMode(req);
                } else {
                    ctx.setType(CatConstants.TYPE_URL_FORWARD);
                }

                ctx.handle();
            }

            protected void setTraceMode(HttpServletRequest req) {
                String traceMode = "X-CAT-TRACE-MODE";
                String headMode = req.getHeader(traceMode);

                if ("true".equals(headMode)) {
                    Cat.getManager().setTraceMode(true);
                }
            }
        },

        ID_SETUP {
            private String m_servers;

            private String getCatServer() {
                if (m_servers == null) {
                    DefaultMessageManager manager = (DefaultMessageManager) Cat.getManager();
                    List<Server> servers = manager.getConfigManager().getServers();

                    m_servers = Joiners.by(',').join(servers, new Joiners.IBuilder<Server>() {
                        @Override
                        public String asString(Server server) {
                            String ip = server.getIp();
                            Integer httpPort = server.getHttpPort();

                            return ip + ":" + httpPort;
                        }
                    });
                }

                return m_servers;
            }

            @Override
            public void handle(BeastCatFilter.Context ctx) throws IOException, ServletException {
                boolean isTraceMode = Cat.getManager().isTraceMode();
                HttpServletResponse res = ctx.getResponse();

                if (isTraceMode) {
                    String id = Cat.getCurrentMessageId();

                    res.setHeader("X-CAT-ROOT-ID", id);
                    res.setHeader("X-CAT-SERVER", getCatServer());
                }

                ctx.handle();
            }
        },

        LOG_CLIENT_PAYLOAD {
            @Override
            public void handle(BeastCatFilter.Context ctx) throws IOException, ServletException {
                HttpServletRequest req = ctx.getRequest();
                String type = ctx.getType();

                if (ctx.isTop()) {
                    logRequestClientInfo(req, type);
                    logRequestPayload(req, type);
                } else {
                    logRequestPayload(req, type);
                }

                ctx.handle();
            }

            protected void logRequestClientInfo(HttpServletRequest req, String type) {
                StringBuilder sb = new StringBuilder(1024);
                String ip = "";
                String ipForwarded = req.getHeader("x-forwarded-for");

                if (ipForwarded == null) {
                    ip = req.getRemoteAddr();
                } else {
                    ip = ipForwarded;
                }

                sb.append("IPS=").append(ip);
                sb.append("&VirtualIP=").append(req.getRemoteAddr());
                sb.append("&Server=").append(req.getServerName());
                sb.append("&Referer=").append(req.getHeader("referer"));
                sb.append("&Agent=").append(req.getHeader("user-agent"));

                StringBuilder sbCat = new StringBuilder();
                String headerRoot = req.getHeader(RestCat.HEADER_ROOT);
                String headerParent = req.getHeader(RestCat.HEADER_PARENT);
                String headerChild = req.getHeader(RestCat.HEADER_CHILD);

                if (StringUtils.isNotEmpty(headerRoot)) {
                    sbCat.append(RestCat.HEADER_ROOT + "=").append(headerRoot);
                }
                if (StringUtils.isNotEmpty(headerParent)) {
                    sbCat.append("&" + RestCat.HEADER_PARENT + "=").append(headerParent);
                }
                if (StringUtils.isNotEmpty(headerChild)) {
                    sbCat.append("&" + RestCat.HEADER_CHILD + "=").append(headerChild);
                }

                Cat.logEvent(type, type + ".Server", Message.SUCCESS, sb.toString());
                Cat.logEvent(type, type + ".CatInfo", Message.SUCCESS, sbCat.toString());
            }

            protected void logRequestPayload(HttpServletRequest req, String type) {
                StringBuilder sb = new StringBuilder(256);

                sb.append(req.getScheme().toUpperCase()).append('/');
                sb.append(req.getMethod()).append(' ').append(req.getRequestURI());

                String qs = req.getQueryString();

                if (qs != null) {
                    sb.append('?').append(qs);
                }

                Cat.logEvent(type, type + ".Method", Message.SUCCESS, sb.toString());
            }
        },

        LOG_SPAN {

            public static final char SPLIT = '/';

            private void customizeStatus(Transaction t, HttpServletRequest req) {
                Object catStatus = req.getAttribute(CatConstants.CAT_STATE);

                if (catStatus != null) {
                    t.setStatus(catStatus.toString());
                } else {
                    t.setStatus(Message.SUCCESS);
                }
            }

            private void customizeUri(Transaction t, HttpServletRequest req) {
                if (t instanceof DefaultTransaction) {
                    Object catPageType = req.getAttribute(CatConstants.CAT_PAGE_TYPE);

                    if (catPageType instanceof String) {
                        ((DefaultTransaction) t).setType(catPageType.toString());
                    }

                    Object catPageUri = req.getAttribute(CatConstants.CAT_PAGE_URI);

                    if (catPageUri instanceof String) {
                        ((DefaultTransaction) t).setName(catPageUri.toString());
                    }
                }
            }

            private String getRequestURI(HttpServletRequest req) {
                String url = req.getRequestURI();
                int length = url.length();
                StringBuilder sb = new StringBuilder(length);

                for (int index = 0; index < length;) {
                    char c = url.charAt(index);

                    if (c == SPLIT && index < length - 1) {
                        sb.append(c);

                        StringBuilder nextSection = new StringBuilder();
                        boolean isNumber = false;
                        boolean first = true;

                        for (int j = index + 1; j < length; j++) {
                            char next = url.charAt(j);

                            if ((first || isNumber == true) && next != SPLIT) {
                                isNumber = isNumber(next);
                                first = false;
                            }

                            if (next == SPLIT) {
                                String name = matchPattern(nextSection.toString());
                                if (name != null) {
                                    sb.append("{" + name + "}");
                                }
                                else if (isNumber) {
                                    sb.append("{num}");
                                } else {
                                    sb.append(nextSection.toString());
                                }
                                index = j;

                                break;
                            } else if (j == length - 1) {
                                String name = matchPattern(nextSection.toString());
                                if (name != null) {
                                    sb.append("{" + name + "}");
                                }
                                else if (isNumber) {
                                    sb.append("{num}");
                                } else {
                                    nextSection.append(next);
                                    sb.append(nextSection.toString());
                                }
                                index = j + 1;
                                break;
                            } else {
                                nextSection.append(next);
                            }
                        }
                    } else {
                        sb.append(c);
                        index++;
                    }
                }

                return sb.toString();
            }

            @Override
            public void handle(BeastCatFilter.Context ctx) throws IOException, ServletException {
                HttpServletRequest req = ctx.getRequest();

                Transaction t = Cat.newTransaction(ctx.getType(), req.getMethod() + " " + getRequestURI(req));
                Cat.Context context = RestCat.getServerSideContext(req);
                RestCat.createServerCross(req, t);
                Cat.logRemoteCallServer(context);

                try {
                    ctx.handle();
                    customizeStatus(t, req);
                } catch (ServletException e) {
                    t.setStatus(e);
                    Cat.logError(e);
                    throw e;
                } catch (IOException e) {
                    t.setStatus(e);
                    Cat.logError(e);
                    throw e;
                } catch (Throwable e) {
                    t.setStatus(e);
                    Cat.logError(e);
                    throw new RuntimeException(e);
                } finally {
                    customizeUri(t, req);
                    t.complete();
                    RestCat.removeContext();
                }
            }

            private boolean isNumber(char c) {
                return (c >= '0' && c <= '9') || c == '.' || c == '-' || c == ',';
            }

            private String matchPattern(String section) {
                for (String name : m_patterns.keySet()) {
                    String pattern = m_patterns.get(name);
                    if (section.matches(pattern)) {
                        return name;
                    }
                }
                return null;
            }

        };
    }

    protected static class Context {
        private FilterChain m_chain;

        private List<BeastCatFilter.Handler> m_handlers;

        private int m_index;

        private HttpServletRequest m_request;

        private HttpServletResponse m_response;

        private boolean m_top;

        private String m_type;

        public Context(HttpServletRequest request, HttpServletResponse response, FilterChain chain, List<BeastCatFilter.Handler> handlers) {
            m_request = request;
            m_response = response;
            m_chain = chain;
            m_handlers = handlers;
        }

        public HttpServletRequest getRequest() {
            return m_request;
        }

        public HttpServletResponse getResponse() {
            return m_response;
        }

        public String getType() {
            return m_type;
        }

        public void handle() throws IOException, ServletException {
            if (m_index < m_handlers.size()) {
                BeastCatFilter.Handler handler = m_handlers.get(m_index++);

                handler.handle(this);
            } else {
                m_chain.doFilter(m_request, m_response);
            }
        }

        public boolean isTop() {
            return m_top;
        }

        public void setTop(boolean top) {
            m_top = top;
        }

        public void setType(String type) {
            m_type = type;
        }
    }

    protected static interface Handler {
        public void handle(BeastCatFilter.Context ctx) throws IOException, ServletException;
    }

}
