package com.thebeastshop.pegasus.component.coupon.service.impl;

import java.util.Arrays;
import java.util.List;

import com.thebeastshop.pegasus.component.coupon.condition.PendingCouponCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponCodeDao;
import com.thebeastshop.pegasus.component.coupon.domain.CouponContent;
import com.thebeastshop.pegasus.component.coupon.domain.impl.FullCutPrice;
import com.thebeastshop.pegasus.component.coupon.enums.CouponType;
import com.thebeastshop.pegasus.component.coupon.model.CouponSampleEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.component.coupon.condition.CouponSampleCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponSampleDao;
import com.thebeastshop.pegasus.component.coupon.domain.CouponSample;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSampleState;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSendWay;
import com.thebeastshop.pegasus.component.coupon.service.BaseService;
import com.thebeastshop.pegasus.component.coupon.service.CouponSampleService;
import com.thebeastshop.support.AccessWay;
import com.thebeastshop.support.exception.NoSuchResourceException;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.mark.State;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class CouponSampleServiceImpl extends BaseService implements CouponSampleService {
	@Autowired
	private CouponSampleDao couponSampleDao;

	@Autowired
	private CouponCodeDao couponCodeDao;

	private void validate(CouponSample obj) {
		if (obj.getCouponType() == null) {
			throw new WrongArgException("未选择优惠券类型");
		}
		if (StringUtils.isEmpty(obj.getName())) {
			throw new WrongArgException("未填优惠券名称");
		}
		if (StringUtils.isEmpty(obj.getTitle())) {
			throw new WrongArgException("未填优惠券标题");
		}
		if (obj.getBalanceCount() == null) {
			throw new WrongArgException("未填剩余数量");
		}
		if (obj.getMaxCount() == null) {
			throw new WrongArgException("未填生成数量");
		}
		if (obj.getMaxPerMember() == null) {
			throw new WrongArgException("未填用户限领数量");
		}
		if (CollectionUtils.isEmpty(obj.getAccessWays())) {
			throw new WrongArgException("未选择适用平台");
		}
		if (obj.getCouponSendWay() == null) {
			throw new WrongArgException("未选择优惠券派发方式");
		}
		if (obj.getStartTime() == null) {
			throw new WrongArgException("未选择优惠券开始日期");
		}
		if (obj.getExpireTime() == null) {
			throw new WrongArgException("未选择优惠券截止日期");
		}
		if (!obj.getExpireTime().after(obj.getStartTime())) {
			throw new WrongArgException("优惠券截止日期必须在开始日期之后");
		}
		if (obj.getCouponContent() == null) {
			throw new WrongArgException("未填优惠券优惠内容");
		}
		CouponContent content = obj.getCouponContent();
		if (obj.getCouponType().getId() == CouponType.CHEAPEN_LINE.getId()) {
			FullCutPrice fullCutPrice = (FullCutPrice) content;
			if (fullCutPrice.getFullPrice() == null) {
				throw new WrongArgException("未填优惠满足条件");
			}
			if (fullCutPrice.getCutPrice() == null) {
				throw new WrongArgException("未填优惠金额");
			}
			if (fullCutPrice.getCutPrice().compareTo(fullCutPrice.getFullPrice()) >= 0) {
				throw new WrongArgException("优惠金额必须小于满足条件");
			}
		}
		Integer sendWayId = obj.getCouponSendWay().getId();
		if (sendWayId == CouponSendWay.CODE.getId()) {
			Integer ret = couponCodeDao.countOf(obj);
			if (ret != null && ret > 0) {
				throw new WrongArgException("在同一时间段内已有相同优惠口令");
			}
		}
	}

	@Override
	public CouponSample create(CouponSample obj) {
		validate(obj);
		return this.couponSampleDao.create(obj);
	}

	@Override
	public boolean update(CouponSample obj) {
		return this.couponSampleDao.update(obj);
	}

	@Override
	public boolean delete(Long id) {
		return false;
	}

	@Override
	public CouponSample getById(final long id) {
		CouponSample obj = this.couponSampleDao.getById(id);
		if (obj == null) {
			throw new WrongArgException("优惠券模板不存在", "id", id);
		}
		return obj;
	}

	@Override
	public List<State> getStates() {
		return Lists.<State> newArrayList(CouponSampleState.values());
	}

	@Override
	public List<CouponSendWay> getCouponSendWays() {
		return Arrays.asList(CouponSendWay.values());
	}

	@Override
	public List<AccessWay> getAccessWays() {
		return Arrays.asList(AccessWay.values());
	}

	@Override
	public List<CouponSample> getByCondition(CouponSampleCondition condition) {
		return this.couponSampleDao.getByCondition(condition);
	}

	@Override
	public List<CouponSample> getFirstPendingCouponSample(PendingCouponCondition cond) {
		Integer type = cond.getType();
		if(null == type) {
			return null;
		}
		if(1 == type) {
			//返回已经过第一级审批的数据集合
			return couponSampleDao.getFirstApprovaledCouponSample();
		}else if (0 == type) {
			//返回未经过第一级审批的数据集合
			return couponSampleDao.getFirstPendingCouponSample();
		}
		return null;
	}

	@Override
	public List<CouponSample> getSecondPendingCouponSample(PendingCouponCondition cond) {
		Integer type = cond.getType();
		if(null == type) {
			return null;
		}
		if(1 == type) {
			//返回已经过第二级审批的数据集合
			return couponSampleDao.getSecondApprovaledCouponSample();
		}else if(0 == type) {
			//返回未经过第二级审批的数据集合
			return couponSampleDao.getSecondPendingCouponSample();
		}
		return null;
	}

	@Override
	public Integer submitApproval(Long id) {
		return couponSampleDao.submitApproval(id);
	}

	@Override
	public CouponSample checkById(long id) {
		final CouponSample one = getById(id);
		if (one == null) {
			throw new NoSuchResourceException("优惠券样板", id);
		}
		return one;
	}

}
