package com.thebeastshop.pegasus.component.coupon.service.impl;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.thebeastshop.pegasus.component.coupon.condition.CouponCodeExchangeCondition;
import com.thebeastshop.pegasus.component.coupon.dao.CouponCodeExchangeDao;
import com.thebeastshop.pegasus.component.coupon.domain.Coupon;
import com.thebeastshop.pegasus.component.coupon.domain.CouponCode;
import com.thebeastshop.pegasus.component.coupon.domain.CouponCodeExchange;
import com.thebeastshop.pegasus.component.coupon.domain.CouponSample;
import com.thebeastshop.pegasus.component.coupon.enums.CouponSampleState;
import com.thebeastshop.pegasus.component.coupon.service.BaseService;
import com.thebeastshop.pegasus.component.coupon.service.CouponCodeExchangeService;
import com.thebeastshop.pegasus.component.coupon.service.CouponCodeService;
import com.thebeastshop.pegasus.component.coupon.service.CouponSampleService;
import com.thebeastshop.pegasus.component.coupon.service.CouponService;
import com.thebeastshop.support.exception.WrongArgException;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class CouponCodeExchangeServiceImpl extends BaseService implements CouponCodeExchangeService {

	@Autowired
	private CouponSampleService couponSampleService;

	@Autowired
	private CouponService couponService;

	@Autowired
	private CouponCodeService couponCodeService;

	@Autowired
	private CouponCodeExchangeDao dao;

	@Transactional
	@Override
	public CouponCodeExchange exchange(final CouponCodeExchange exchange) {
		final String code = exchange.getCode();
		if (StringUtils.isBlank(code)) {
			throw new WrongArgException("优惠券口令为空", "code", code);
		}
		// 校验优惠券口令是否有效
		CouponCode couponCode = couponCodeService.getByCode(code);
		if (couponCode == null) {
			throw new WrongArgException("优惠券口令无效", "code", code);
		}
		final CouponSample couponSample = this.couponSampleService.getById(couponCode.getCouponSampleId());
		// 校验优惠券模板是否可领取
		if (!CouponSampleState.ONLINE.equals(couponSample.getState()) || couponSample.isExpired()) {
			throw new WrongArgException("优惠券已失效", "state", couponSample.getState());
		}
		// 校验优惠券是否超过用户限领数量
		CouponCodeExchangeCondition condition = new CouponCodeExchangeCondition();
		condition.setOwnerId(exchange.getOwnerId());
		condition.setCouponSampleId(couponCode.getCouponSampleId());
		List<CouponCodeExchange> exchanges = this.getByCondition(condition);
		if (exchanges.size() >= couponSample.getMaxPerMember()) {
			throw new WrongArgException("优惠券超过限领数量", "code", code);
		}
		// 发送优惠券
		Coupon coupon = couponService.send(couponSample, exchange.getOwnerId());
		// 创建优惠码兑换记录
		CouponCodeExchange createObj = new CouponCodeExchange();
		createObj.setCode(code);
		createObj.setCouponId(coupon.getId());
		createObj.setCouponSampleId(couponSample.getId());
		createObj.setOwnerId(exchange.getOwnerId());
		return this.create(createObj);
	}

	@Override
	public List<CouponCodeExchange> getByCondition(CouponCodeExchangeCondition condition) {
		return this.dao.getByCondition(condition);
	}

	@Override
	public CouponCodeExchange create(CouponCodeExchange exchange) {
		return this.dao.create(exchange);
	}

}
