package com.thebeastshop.es.dao.core;

import com.thebeastshop.es.annotation.EsDaoConfig;
import com.thebeastshop.es.spring.ElasticsearchClientFactory;
import com.thebeastshop.scm.es.PsUpdateVO;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.converters.BigDecimalConverter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.rest.RestStatus;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.AnnotationUtils;
import pers.richard.ormybatis.domain.core.AbstractDomain;

import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.util.*;



public class AbstractEsDao<T extends PsUpdateVO> {
	protected final Logger log = LoggerFactory.getLogger(getClass());


    protected String index;
    protected String type;
    protected String id;

	static {
		BigDecimalConverter bc = new BigDecimalConverter(BigDecimal.ZERO);
		ConvertUtils.register(bc, BigDecimal.class);
	}
    
    protected RestHighLevelClient client = null;
    
    public AbstractEsDao(ElasticsearchClientFactory elasticsearchClientFactory){
    	Annotation annotation = AnnotationUtils.findAnnotation(this.getClass(), EsDaoConfig.class);
    	this.index = (String)AnnotationUtils.getValue(annotation, "index");
    	this.id = (String)AnnotationUtils.getValue(annotation, "id");
    	if (StringUtils.isBlank(index)) {
    		this.index = elasticsearchClientFactory.getIndex();
    	}
    	this.type = (String)AnnotationUtils.getValue(annotation, "type");
		client = elasticsearchClientFactory.createClient();
    }

    public T byId(String id) {
    	if (StringUtils.isBlank(id)) {
			return null;
		}
		GetRequest getRequest = new GetRequest(index,type, id);
//		getRequest.realtime(false);
		GetResponse responseGet = null;
		try {
			responseGet = client.get(getRequest);
		}catch(Exception e){
		   log.error("error id:",id);
           log.error("es error:",e);
		   return null;
		}
    	Map<String, Object> sourceMap = responseGet.getSourceAsMap();
		if (MapUtils.isEmpty(sourceMap)) {
			return null;
		}
    	return this.toBean(sourceMap);
    }
    
    public IndexResponse insertOrUpdateOld(T vo) {
    	Map<String, Object> sourceMap = this.toMap(vo);
    	String id = MapUtils.getString(sourceMap, this.id);
    	IndexRequest indexRequest = this.getIndexRequest(id, sourceMap);
    	//return client.index(indexRequest).actionGet();
		return null;
    }

    public BulkResponse insertOrUpdate(Collection<T> list){
    	if (CollectionUtils.isEmpty(list)) {
			return null;
		}
    	BulkRequest bulkRequest = new BulkRequest();
    	for (T t : list) {
    		Map<String, Object> sourceMap = this.toMap(t);
    		String id = MapUtils.getString(sourceMap, this.id);
    		if (this.byId(id) == null) {
    			IndexRequest indexRequest = this.getIndexRequest(id, sourceMap);
    			bulkRequest.add(indexRequest);
			} else {
				UpdateRequest updateRequest = this.getUpdateRequest(id, sourceMap);
				bulkRequest.add(updateRequest);
			}
    	}
		BulkResponse bulkResponse = null;
    	try {
			bulkResponse = client.bulk(bulkRequest);
			if (bulkResponse.hasFailures()) {
				log.error("es_fail:" + bulkResponse.buildFailureMessage());
			}
		}catch(Exception e){
			log.error("es_error:", e);
		}
		return bulkResponse;
    }
    
    public UpdateResponse update(T vo){
    	Map<String, Object> sourceMap = this.toMap(vo);
    	String id = MapUtils.getString(sourceMap, this.id);
    	UpdateRequest updateRequest = this.getUpdateRequest(id, sourceMap);
		UpdateResponse updateResponse =  null;
		try {
			updateResponse = client.update(updateRequest);
			if (!updateResponse.status().equals(RestStatus.OK)) {
				log.error("es_fail:" + updateResponse.status().toString());
			}
		}
		catch(Exception e){
			log.error("es_error:", e);
		}
		return updateResponse;
    }

    public BulkResponse update(Collection<T> list){
    	if (CollectionUtils.isEmpty(list)) {
			return null;
		}
		BulkRequest bulkRequest = new BulkRequest();
    	for (T t : list) {
    		Map<String, Object> sourceMap = this.toMap(t);
    		String id = MapUtils.getString(sourceMap, this.id);
    		UpdateRequest ur = this.getUpdateRequest(id, sourceMap);
    		bulkRequest.add(ur);
    	}
		BulkResponse bulkResponse = null;
    	try {
			bulkResponse = client.bulk(bulkRequest);
			if (bulkResponse.hasFailures()) {
				log.error("es_fail:" + bulkResponse.buildFailureMessage());
			}
		}
		catch(Exception e){
			log.error("es_error:", e);
		}
    	return bulkResponse;
    }
    
    public DeleteResponse delete(String id){
    	DeleteRequest deleteRequest = this.getDeleteRequest(id);
		DeleteResponse deleteResponse = null;
		try {
			deleteResponse = client.delete(deleteRequest);
			if (!deleteResponse.status().equals(RestStatus.OK)) {
				log.error("es_fail:" + deleteResponse.status().toString());
			}
		}
		catch(Exception e){
			log.error("es_error:", e);
		}
    	return deleteResponse;
    }
    
    public BulkResponse delete(List<String> list){
    	if (CollectionUtils.isEmpty(list)) {
			return null;
		}
    	BulkRequest bulkRequest = new BulkRequest();
    	for (String id : list) {
    		DeleteRequest deleteRequest = this.getDeleteRequest(id);
    		bulkRequest.add(deleteRequest);
    	}
		BulkResponse bulkResponse = null;
    	try {
			bulkResponse = client.bulk(bulkRequest);
			if (bulkResponse.hasFailures()) {
				log.error("es_fail:" + bulkResponse.buildFailureMessage());
			}
		}
		catch(Exception e){
			log.error("es_error:", e);
		}
    	return bulkResponse;
    }
    
    
    public T toBean(Map<String, Object> sourceMap) {
    	if (MapUtils.isEmpty(sourceMap)) {
			return null;
		}
    	T vo = null;
        try {
            vo = this.getClz().newInstance();
            BeanUtils.populate(vo, sourceMap);
        } catch (Exception e) {
            log.error(null, e);
        }
        return vo;
    }
    
    public Map<String, Object> toMap(T t) {
		Map<String, Object> map = new HashMap<String, Object>();
		try {
			PropertyDescriptor[] propertyDescriptors = Introspector.getBeanInfo(this.getClz()).getPropertyDescriptors();
			for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
				String propertyName = propertyDescriptor.getName();
				if ("class".equals(propertyName)) {
					continue;
				}
				Method readMethod = propertyDescriptor.getReadMethod();
				Object result = readMethod.invoke(t, new Object[0]);
				if (result == null) {
					continue;
				}
				if (result instanceof Collection) {
					if (CollectionUtils.isEmpty((Collection)result)) {
						continue;
					}
				}
				if (result instanceof Map) {
					if (MapUtils.isEmpty((Map)result)) {
						continue;
					}
				}
				map.put(propertyName, result);
			}
		} catch (Exception e) {
			log.error(null, e);
		}
		return map;
    }
    
    protected UpdateRequest getUpdateRequest(String id, Map sourceMap){
    	DateTime now = DateTime.now();
    	sourceMap.put("updateTime", now.toString("yyyy-MM-dd HH:mm:ss"));
    	sourceMap.put("updateLong", now.getMillis());
    	return new UpdateRequest(index, type, id).doc(sourceMap);
    }

    protected IndexRequest getIndexRequest(String id, Map sourceMap){
   	DateTime now = DateTime.now();
   	sourceMap.put("updateTime", now.toString("yyyy-MM-dd HH:mm:ss"));
   	sourceMap.put("updateLong", now.getMillis());
    	return new IndexRequest(index, type, id).source(sourceMap);	
    }
    
    protected DeleteRequest getDeleteRequest(String id){
    	return new DeleteRequest(index, type, id);	
    }
    

    public List toListBean(Object listMapObj, Class clz){
    	if (listMapObj == null) {
    		return Collections.EMPTY_LIST;
    	}
    	List result = new ArrayList();
    	if (listMapObj instanceof List) {
    		List<Map> listMap = (List) listMapObj;
    		for (Map map : listMap) {
    			try {
    				Object o = clz.newInstance();
    				BeanUtils.populate(o, map);
    				result.add(o);
    			} catch (Exception e) {
    				log.error(null, e);
    			}
    		}
    	}
    	return result;
    }
    
    public List toListBean(Map sourceMap, String key, Class clz){
    	Collection list = (Collection) MapUtils.getObject(sourceMap, key, Collections.EMPTY_LIST);
    	if (CollectionUtils.isEmpty(list)) {
			return Collections.EMPTY_LIST;
		}
		List listBean = new ArrayList();
		if (list instanceof List) {
			List<Map> listMap = (List) list;
			for (Map map : listMap) {
				try {
					Object o = clz.newInstance();
					BeanUtils.populate(o, map);
					listBean.add(o);
				} catch (Exception e) {
					log.error(null, e);
				}
			}
		}
		
		return listBean;
	}
    
    public List<Map> toListMap(List listBean){
		if (listBean == null) return null;
		List<Map> result = new ArrayList();
		if (CollectionUtils.isNotEmpty(listBean)) {
			for (Object obj : listBean) {
				if (obj instanceof AbstractDomain) {
					AbstractDomain abstractDomain = (AbstractDomain) obj;
					result.add(abstractDomain.toMap());
				}
			}
		}
		return result;
	}
    
    private Class<T> getClz(){
		 Type type = this.getClass().getGenericSuperclass();
        Type[] p = ((ParameterizedType) type).getActualTypeArguments();
        Class<T> clz = (Class<T>) p[0];
        return clz;
   }
}