package com.thebeastshop.es.dao;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.thebeastshop.scm.es.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.update.UpdateRequest;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;

import com.thebeastshop.es.annotation.EsDaoConfig;
import com.thebeastshop.es.dao.core.AbstractEsDao;
import com.thebeastshop.es.spring.ElasticsearchFactoryBean;

import javax.annotation.Resource;


@EsDaoConfig(id = "id", type = "pubprod")
public class ProductEsDao extends AbstractEsDao<PsProductVO> {

	@Resource
	private SpvEsDao spvEsDao;

	@Autowired
	public ProductEsDao(ElasticsearchFactoryBean elasticsearchFactoryBean) {
		super(elasticsearchFactoryBean);
	}
	
	public void batchSave(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			Map<String, Object> sourceMap = this.toMap(vo);
			//必须更新字段,为null则更新为空
			sourceMap.put("smallIconList", vo.getSmallIconList());
			sourceMap.put("largeIconList", vo.getLargeIconList());
			sourceMap.put("frontCategories",vo.getFrontCategories());
			try {
				sourceMap.put("psTags", this.toListMap(vo.getPsTags()));
			} catch (Exception ex) {
				log.error(ex.toString());
			}
//			sourceMap.put("stocks", vo.getStocks());
			sourceMap.put("brandList", vo.getBrandList());
			//不更新的字段
//			sourceMap.remove("chnSaleChnCodes");
//			sourceMap.remove("onShelfChnCodes");
//			sourceMap.remove("offShelfChnCodes");
//			sourceMap.remove("selfDeliveries");
//			sourceMap.remove("dynmContents");
//			sourceMap.remove("chnCanSaleSKu");
//			sourceMap.remove("chnCanSeeSKu");
			sourceMap.remove("channelProdInfo");
//			sourceMap.remove("spvList");
			String id = MapUtils.getString(sourceMap, this.id);
			if (this.byId(id) == null) {
				IndexRequest indexRequest = this.getIndexRequest(id, sourceMap);
				bulkRequest.add(indexRequest);
			} else {
				UpdateRequest updateRequest = this.getUpdateRequest(id, sourceMap);
				bulkRequest.add(updateRequest);
			}
		}
		bulkRequest.execute().actionGet();
	}
	
	public void batchUpdateCalSaleProd(List<PsProductVO> list){
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			//必须更新字段,为null则更新为空
			sourceMap.put("chnSaleChnCodes", vo.getChnSaleChnCodes());
			sourceMap.put("onShelfChnCodes", vo.getOnShelfChnCodes());
			sourceMap.put("offShelfChnCodes", vo.getOffShelfChnCodes());
			sourceMap.put("chnCanSaleSKu", this.toListMap(vo.getChnCanSaleSKu()));
			sourceMap.put("chnCanSeeSKu", this.toListMap(vo.getChnCanSeeSKu()));
			if (vo.getListPrice() != null) {
				sourceMap.put("listPrice", vo.getListPrice().doubleValue());
			}
			if (vo.getListPriceSkuCode() != null) {
				sourceMap.put("listPriceSkuCode", vo.getListPriceSkuCode());
			}
			if (vo.getListPriceSpvId() != null) {
				sourceMap.put("listPriceSpvId", vo.getListPriceSpvId());
			}
			sourceMap.put("listPriceList", vo.getListPriceList());
			if (vo.getStocks() != null) {
				sourceMap.put("stocks", vo.getStocks());
			}
			if (vo.getPrimaryUpdateTime() != null) {
				sourceMap.put("primaryUpdateTime", vo.getPrimaryUpdateTime());
			}
			if (vo.getPrimaryUpdateLong() != null) {
				sourceMap.put("primaryUpdateLong", vo.getPrimaryUpdateLong());
			}
			List<PsChnProdTimeCodeInfo> PsChnProdTimeCodeInfo = vo.getChannelProdInfo();
			if (CollectionUtils.isNotEmpty(PsChnProdTimeCodeInfo)) {
				sourceMap.put("channelProdInfo", this.toListMap(PsChnProdTimeCodeInfo));
			}
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}

	public void batchUpdateSku(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("skuCodes", vo.getSkuCodes());
			sourceMap.put("categoryId", vo.getCategoryId());
			sourceMap.put("categoryCode", vo.getCategoryCode());
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}

	public void batchUpdateSelfDeliveries(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("selfDeliveries", this.toListMap(vo.getSelfDeliveries()));
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId(), sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}

	public void batchUpdateDymnContent(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("dynmContents", this.toListMap(vo.getDynmContents()));
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}

	public void batchUpdateBrand(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("brandList", vo.getBrandList());
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();		
	}
	
	public void batchUpdateLabel(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("labels", vo.getLabels());
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();	
	}
	
	public void batchUpdateAttribute(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("attributeList", this.toListMap(vo.getAttributeList()));
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}
	
	public void batchUpdateSpv(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			List<Map> spvListMap = new ArrayList<>();
			for (PsSpvVO spvVO : vo.getSpvList()) {
				spvListMap.add(spvEsDao.toMap(spvVO));
			}
			if (vo.getListPrice() != null) {
				sourceMap.put("listPrice", vo.getListPrice().doubleValue());
			}
			if (vo.getListPriceSkuCode() != null) {
				sourceMap.put("listPriceSkuCode", vo.getListPriceSkuCode());
			}
			if (vo.getListPriceSpvId() != null) {
				sourceMap.put("listPriceSpvId", vo.getListPriceSpvId());
			}
			sourceMap.put("listPriceList", vo.getListPriceList());
			sourceMap.put("spvList", spvListMap);
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}




	public void batchUpdateStocks(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				return;
			}
			Map<String, Object> sourceMap = new HashMap();
			sourceMap.put("stocks", vo.getStocks());
			if (vo.getListPrice() != null) {
				sourceMap.put("listPrice", vo.getListPrice().doubleValue());
			}
			if (vo.getListPriceSkuCode() != null) {
				sourceMap.put("listPriceSkuCode", vo.getListPriceSkuCode());
			}
			if (vo.getListPriceSpvId() != null) {
				sourceMap.put("listPriceSpvId", vo.getListPriceSpvId());
			}
			sourceMap.put("listPriceList", vo.getListPriceList());
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}


	public void batchUpdateCampaign(List<PsProductVO> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		BulkRequestBuilder bulkRequest = client.prepareBulk();
		for (PsProductVO vo : list) {
			if (this.byId(vo.getId() + "") == null) {
				log.warn("productId:{} ES没有查到", vo.getId());
				continue;
			}
			Map<String, Object> sourceMap = new HashMap();
			if (vo.getCampaignList() == null) continue;
			List<Map> campaignList = toListMap(vo.getCampaignList());
			sourceMap.put("campaignList", campaignList);
			UpdateRequest updateRequest = this.getUpdateRequest(vo.getId() + "", sourceMap);
			bulkRequest.add(updateRequest);
		}
		bulkRequest.execute().actionGet();
	}


	public void updateSaleStartDate(String productId, Date saleStartDate) {
		Map sourceMap = new HashMap();
		sourceMap.put("saleStartDate", saleStartDate);
		client.prepareBulk().add(this.getUpdateRequest(productId, sourceMap)).execute();
	}
	
	@Override
	public PsProductVO toBean(Map<String, Object> sourceMap) {
		if (MapUtils.isEmpty(sourceMap)) {
			return null;
		}
		PsProductVO productEsVO = new PsProductVO();
		productEsVO.setId(MapUtils.getString(sourceMap, "id"));
		productEsVO.setCode(MapUtils.getString(sourceMap, "code"));
		String listPrice = MapUtils.getString(sourceMap, "listPrice");
		if (StringUtils.isNotBlank(listPrice)) {
			productEsVO.setListPrice(new BigDecimal(listPrice));
		}
		productEsVO.setAllowVipDiscount(MapUtils.getInteger(sourceMap, "allowVipDiscount"));//是否支持VIP折扣，1是0否
		productEsVO.setAllowNoteCard(MapUtils.getInteger(sourceMap, "allowNoteCard"));//是否支持代写卡片，0否1是
		productEsVO.setPrepareDays(MapUtils.getInteger(sourceMap, "prepareDays"));// 备货天数，默认0
		productEsVO.setSelectDeliveryDays(MapUtils.getInteger(sourceMap, "selectDeliveryDays"));//可选择配送日期天数，0:尽快（不可选），n:n天内(默认n=7)
		productEsVO.setPicList(MapUtils.getString(sourceMap, "picList"));//图片（列表）
		productEsVO.setPicDetail(MapUtils.getString(sourceMap, "picDetail"));//图片（详情）
		productEsVO.setPicThumb(MapUtils.getString(sourceMap, "picThumb"));//图片（缩略）
	    productEsVO.setPicDetailMul(MapUtils.getString(sourceMap, "picDetailMul"));//图片,手机端,多张以’，‘ 分割
	    productEsVO.setExpressType(MapUtils.getInteger(sourceMap, "expressType", 3));// 快递类型，1店送2联邦3顺丰
	    productEsVO.setName(MapUtils.getString(sourceMap, "name"));//产品名
	    productEsVO.setNameCn(MapUtils.getString(sourceMap, "nameCn"));// 产品中文名
	    productEsVO.setSkuCategoryDesc(MapUtils.getString(sourceMap, "skuCategoryDesc"));//SKU分类描述
	    productEsVO.setCategoryId(MapUtils.getLongValue(sourceMap, "categoryId"));//prod category , last update sku's category
	    productEsVO.setCategoryCode(MapUtils.getString(sourceMap, "categoryCode", ""));//prod category , last update sku's category
	    productEsVO.setSeq(MapUtils.getInteger(sourceMap, "seq"));
	    productEsVO.setSort1(MapUtils.getInteger(sourceMap, "sort1"));
	    productEsVO.setSort2(MapUtils.getInteger(sourceMap, "sort2"));
	    productEsVO.setDescription(MapUtils.getString(sourceMap, "description"));//一句话描述
		productEsVO.setCrossBorderFlag(MapUtils.getInteger(sourceMap, "crossBorderFlag"));
		String saleStartDateString = MapUtils.getString(sourceMap, "saleStartDate");
	    if (StringUtils.isNotBlank(saleStartDateString)) {
	    	Date saleStartDate = DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").parseDateTime(saleStartDateString).plusHours(8).toDate();
	    	productEsVO.setSaleStartDate(saleStartDate);
		}
	    productEsVO.setUpdateTime(MapUtils.getString(sourceMap, "updateTime"));
	    productEsVO.setUpdateLong(MapUtils.getString(sourceMap, "updateLong"));
	    
		productEsVO.setSkuCodes((List<String>)sourceMap.get("skuCodes"));
		productEsVO.setSmallIconList((List<String>)sourceMap.get("smallIconList"));
		productEsVO.setLargeIconList((List<String>)sourceMap.get("largeIconList"));
		productEsVO.setChnSaleChnCodes((List<String>)sourceMap.get("chnSaleChnCodes"));
		productEsVO.setOnShelfChnCodes((List<String>)sourceMap.get("onShelfChnCodes"));
		productEsVO.setOffShelfChnCodes((List<String>)sourceMap.get("offShelfChnCodes"));
		
		productEsVO.setFrontCategories((List<Map>)sourceMap.get("frontCategories"));
		productEsVO.setStocks((List<Map>)sourceMap.get("stocks"));
		List<PsTagVO> psTags = this.toListBean(sourceMap, "psTags", PsTagVO.class);
		productEsVO.setPsTags(psTags);
		List<PsDeliveryVO> selfDeliveries = this.toListBean(sourceMap, "selfDeliveries", PsDeliveryVO.class);
		productEsVO.setSelfDeliveries(selfDeliveries);
		List<PsDynmContentVO> dynmContents = this.toListBean(sourceMap, "dynmContents", PsDynmContentVO.class);
		productEsVO.setDynmContents(dynmContents);
		productEsVO.setChnCanSaleSKu(this.toListBean(sourceMap, "chnCanSaleSKu", PsChnProdSaleSku.class));
		productEsVO.setChnCanSeeSKu(this.toListBean(sourceMap, "chnCanSeeSKu", PsChnProdSaleSku.class));
		productEsVO.setBrandList(this.toListBean(sourceMap, "brandList", PsBrandVO.class));
		productEsVO.setLabels((List<Map>)sourceMap.get("labels"));
		productEsVO.setAttributeList(this.toListBean(sourceMap, "attributeList", PsAttributeVO.class));
		productEsVO.setSpvList(this.toListBean(sourceMap, "spvList", PsSpvVO.class));

		Object channelProdInfoObj =  sourceMap.get("channelProdInfo");
		if (channelProdInfoObj != null) {
			List<Map> listMap = (List<Map>)channelProdInfoObj;
			if (CollectionUtils.isNotEmpty(listMap)) {
				List<PsChnProdTimeCodeInfo> channelProdInfo = new ArrayList<>();
				for (Map map : listMap) {
					String channelCode = MapUtils.getString(map, "channelCode");
					String creatTimeString = MapUtils.getString(map, "createTime");
					if (StringUtils.isAnyBlank(channelCode, creatTimeString)) {
						Date creatTime = DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").parseDateTime(creatTimeString).plusHours(8).toDate();
						channelProdInfo.add(new PsChnProdTimeCodeInfo(channelCode, creatTime));
					}
				}
				productEsVO.setChannelProdInfo(channelProdInfo);
			}
		}
		return productEsVO;
	}
	
	@Override
	public Map<String, Object> toMap(PsProductVO vo) {
		Map<String, Object> sourceMap = new LinkedHashMap<String, Object>();
		sourceMap.put("id", vo.getId());
		sourceMap.put("prodId", vo.getId());
		sourceMap.put("code", vo.getCode());
		sourceMap.put("combined", vo.getCombined() == null ? 0 : vo.getCombined());
		BigDecimal listPrice = vo.getListPrice();
		if (listPrice != null) {
			sourceMap.put("listPrice", listPrice.doubleValue());
		}
		if (vo.getListPriceSkuCode() != null) {
			sourceMap.put("listPriceSkuCode", vo.getListPriceSkuCode());
		}
		if (vo.getListPriceSpvId() != null) {
			sourceMap.put("listPriceSpvId", vo.getListPriceSpvId());
		}
		sourceMap.put("listPriceList", vo.getListPriceList());
		sourceMap.put("allowVipDiscount", vo.getAllowVipDiscount());
		sourceMap.put("allowNoteCard", vo.getAllowNoteCard());
		sourceMap.put("prepareDays", vo.getPrepareDays());
		sourceMap.put("selectDeliveryDays", vo.getSelectDeliveryDays());
		sourceMap.put("picList", vo.getPicList());//图片（列表）
		sourceMap.put("picDetail",vo.getPicDetail());//图片（详情）
		sourceMap.put("picThumb",vo.getPicThumb());//图片（缩略）
		sourceMap.put("picDetailMul", vo.getPicDetailMul());
		sourceMap.put("expressType", vo.getExpressType());
		sourceMap.put("name", vo.getName());
		sourceMap.put("nameCn", vo.getNameCn());// 产品中文名
		if (vo.getSkuCategoryDesc() != null) {
			sourceMap.put("skuCategoryDesc", vo.getSkuCategoryDesc());//SKU分类描述
		}
		sourceMap.put("categoryId", vo.getCategoryId());
		sourceMap.put("categoryCode", vo.getCategoryCode());
		sourceMap.put("seq",vo.getSeq());
		sourceMap.put("sort1", vo.getSort1());
		sourceMap.put("sort2", vo.getSort2());
		sourceMap.put("description", vo.getDescription());
		if (vo.getPrimaryUpdateTime() != null) {
			sourceMap.put("primaryUpdateTime", vo.getPrimaryUpdateTime());
		}
		if (vo.getPrimaryUpdateLong() != null) {
			sourceMap.put("primaryUpdateLong", vo.getPrimaryUpdateLong());
		}
		
		Date saleStartDate = vo.getSaleStartDate();
		if (saleStartDate != null) {
			sourceMap.put("saleStartDate", saleStartDate);
		}
		sourceMap.put("crossBorderFlag", vo.getCrossBorderFlag());
		
		sourceMap.put("skuCodes", vo.getSkuCodes());
		sourceMap.put("smallIconList", vo.getSmallIconList());
		sourceMap.put("largeIconList", vo.getLargeIconList());
		sourceMap.put("chnSaleChnCodes", vo.getChnSaleChnCodes());
		sourceMap.put("onShelfChnCodes", vo.getOnShelfChnCodes());
		sourceMap.put("offShelfChnCodes", vo.getOffShelfChnCodes());
		
		sourceMap.put("frontCategories", vo.getFrontCategories());
		sourceMap.put("stocks", vo.getStocks());
		sourceMap.put("psTags", this.toListMap(vo.getPsTags()));
		sourceMap.put("selfDeliveries", this.toListMap(vo.getSelfDeliveries()));//配送信息 区县  (自送)
		sourceMap.put("dynmContents", this.toListMap(vo.getDynmContents()));
		sourceMap.put("chnCanSaleSKu", this.toListMap(vo.getChnCanSaleSKu()));
		sourceMap.put("chnCanSeeSKu", this.toListMap(vo.getChnCanSeeSKu()));
		sourceMap.put("brandList", this.toListMap(vo.getBrandList()));
		sourceMap.put("channelProdInfo", this.toListMap(vo.getChannelProdInfo()));
		sourceMap.put("labels", vo.getLabels());
		sourceMap.put("attributeList", this.toListMap(vo.getAttributeList()));
		sourceMap.put("campaignList", this.toListMap(vo.getCampaignList()));
		sourceMap.put("spvList", this.toListMap(vo.getSpvList()));
		return sourceMap;
	}


}
