package com.thebeastshop.trans.enums;

import com.thebeastshop.common.utils.DateUtil;
import com.thebeastshop.trans.vo.refund.RefundInfoVO;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 售后订单状态
 */
public enum TsRefundStepEnum {

    /**
     * 提交申请
     */
    APPLY("提交申请") {
        @Override
        public boolean canCancelRefund() {
            return true;
        }

        /**
         * 获取操作描述
         *
         * @return
         */
        @Override
        public String getDesc() {
            return "买家发起申请";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.APPLY);
            if (null == refundStepContentParam.failType) {
                refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("你已提交申请，请等待商家处理", "", REFUND_CONTENT_BLACK));
                if (StringUtils.isNotBlank(refundStepContentParam.acceptLimitCountDownTimeStr)) {
                    refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("还剩%s，超时系统自动同意", refundStepContentParam.acceptLimitCountDownTimeStr), "", REFUND_CONTENT_GRAY));
                } else {
                    refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("商家将在%s个工作日内处理", refundStepContentParam.acceptWorkDayNum), "", REFUND_CONTENT_GRAY));
                }
            } else {
                refundStepInfo.setFailStep(Boolean.TRUE);
                refundStepInfo.setFailType(refundStepContentParam.failType);
                if (RefundStepFailType.SELL_REJECT == refundStepContentParam.failType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("商家已拒绝,如有疑问请联系客服", "", REFUND_CONTENT_YELLOW));
                    refundStepInfo.setRejectContent(new RefundInfoVO.RefundContentText(String.format("拒绝原因：%s", refundStepContentParam.rejectReason), "", REFUND_CONTENT_YELLOW));
                    if (StringUtils.isNotBlank(refundStepContentParam.sellRejectCloseCountDownTimeStr)) {
                        refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("还剩%s，申请将自动关闭", refundStepContentParam.sellRejectCloseCountDownTimeStr), "", REFUND_CONTENT_GRAY));
                    }
                } else if (RefundStepFailType.CLOSE == refundStepContentParam.failType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("申请已关闭", "", REFUND_CONTENT_BLACK));
                    refundStepInfo.setRejectContent(new RefundInfoVO.RefundContentText(String.format("关闭原因：%s", refundStepContentParam.closeReason), "", REFUND_CONTENT_BLACK));
                    refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(DateUtil.formatDate(refundStepContentParam.closeTime, DateUtil.FORMAT_YYYY_MM_DD_HH_MM_SS), "", REFUND_CONTENT_GRAY));
                }
            }
            return refundStepInfo;
        }

    }, ALLOW("卖家同意") {
        @Override
        public boolean canCancelRefund() {
            return true;
        }

        /**
         * 获取操作描述
         *
         * @return
         */
        @Override
        public String getDesc() {
            return "商家同意本次售后申请";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.ALLOW);
            if (null == refundStepContentParam.failType) {
                refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("商家已同意售后申请\n请在时限内填写快递单号并寄回商品", "", REFUND_CONTENT_BLACK));
                if (StringUtils.isNotBlank(refundStepContentParam.deliveryCloseCountDownTimeStr)) {
                    refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("还剩%s，超时将自动关闭", refundStepContentParam.deliveryCloseCountDownTimeStr), "", REFUND_CONTENT_BLACK));
                }
            } else {
                refundStepInfo.setFailStep(Boolean.TRUE);
                refundStepInfo.setFailType(refundStepContentParam.failType);
                if (RefundStepFailType.SELL_REJECT == refundStepContentParam.failType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("商家已拒绝,如有疑问请联系客服", "", REFUND_CONTENT_YELLOW));
                    refundStepInfo.setRejectContent(new RefundInfoVO.RefundContentText(String.format("拒绝原因：%s", refundStepContentParam.rejectReason), "", REFUND_CONTENT_YELLOW));
                    if (StringUtils.isNotBlank(refundStepContentParam.sellRejectCloseCountDownTimeStr)) {
                        refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("还剩%s，申请将自动关闭", refundStepContentParam.sellRejectCloseCountDownTimeStr), "", REFUND_CONTENT_GRAY));
                    }
                } else if (RefundStepFailType.CLOSE == refundStepContentParam.failType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("申请已关闭", "", REFUND_CONTENT_BLACK));
                    refundStepInfo.setRejectContent(new RefundInfoVO.RefundContentText(String.format("关闭原因：%s", refundStepContentParam.closeReason), "", REFUND_CONTENT_BLACK));
                    refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(DateUtil.formatDate(refundStepContentParam.closeTime, DateUtil.FORMAT_YYYY_MM_DD_HH_MM_SS), "", REFUND_CONTENT_GRAY));
                }
            }
            return refundStepInfo;
        }
    }, RETURN("买家寄回") {

        @Override
        public boolean canCancelRefund() {
            return false;
        }

        /**
         * 获取操作描述
         *
         * @return
         */
        @Override
        public String getDesc() {
            return "买家寄回商品";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.RETURN);
            if (null == refundStepContentParam.failType) {
                if (TsRefundTypeEnum.EXCHANGE == refundStepContentParam.refundType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("请等待商家收货并发出新商品", "", REFUND_CONTENT_BLACK));
                } else {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("请等待商家收货并退款", "", REFUND_CONTENT_BLACK));
                }
            } else {
                refundStepInfo.setFailStep(Boolean.TRUE);
                refundStepInfo.setFailType(refundStepContentParam.failType);
                if (RefundStepFailType.SELL_REJECT == refundStepContentParam.failType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("商家已拒绝，如有疑问请联系客服", "", REFUND_CONTENT_YELLOW));
                    refundStepInfo.setRejectContent(new RefundInfoVO.RefundContentText(String.format("拒绝原因：%s", refundStepContentParam.rejectReason), "", REFUND_CONTENT_YELLOW));
                    if (StringUtils.isNotBlank(refundStepContentParam.sellRejectCloseCountDownTimeStr)) {
                        refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("还剩%s，申请将自动关闭", refundStepContentParam.sellRejectCloseCountDownTimeStr), "", REFUND_CONTENT_GRAY));
                    }
                } else if (RefundStepFailType.CLOSE == refundStepContentParam.failType) {
                    refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("申请已关闭", "", REFUND_CONTENT_BLACK));
                    refundStepInfo.setRejectContent(new RefundInfoVO.RefundContentText(String.format("关闭原因：%s", refundStepContentParam.closeReason), "", REFUND_CONTENT_BLACK));
                    refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(DateUtil.formatDate(refundStepContentParam.closeTime, DateUtil.FORMAT_YYYY_MM_DD_HH_MM_SS), "", REFUND_CONTENT_GRAY));
                }
            }
            return refundStepInfo;
        }
    }, REFUND("商家退款") {

        @Override
        public boolean canCancelRefund() {
            return false;
        }

        /**
         * 获取操作描述
         *
         * @return
         */
        @Override
        public String getDesc() {
            return "商家操作退款";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.REFUND);
            if (null == refundStepContentParam.failType) {
                refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("商家已操作退款", "", REFUND_CONTENT_BLACK));
                refundStepInfo.setDesc(new RefundInfoVO.RefundContentText(String.format("退款将在%s个工作日返回", refundStepContentParam.refundWorkDayNum), "", REFUND_CONTENT_GRAY));
            }
            return refundStepInfo;
        }
    }, REFUND_SUCCESS("退款成功") {
        @Override
        public boolean canCancelRefund() {
            return false;
        }

        /**
         * 获取操作描述
         *
         * @return
         */
        @Override
        public String getDesc() {
            return "退款成功";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.REFUND_SUCCESS);
            if (null == refundStepContentParam.failType) {
                refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("退款成功", refundStepContentParam.getRefundAmt().setScale(2, RoundingMode.DOWN).toString(), REFUND_CONTENT_YELLOW));
                refundStepInfo.setDesc(new RefundInfoVO.RefundContentText("退款时间", DateUtil.formatDate(refundStepContentParam.refundTime, DateUtil.FORMAT_YYYY_MM_DD_HH_MM_SS), REFUND_CONTENT_BLACK));
                List<RefundInfoVO.RefundContentText> paymentRefundAmts = new ArrayList<>(refundStepContentParam.paymentRefundAmt.size());
                for (RefundStepContentParam.RefundPaymentAmt refundPaymentAmt : refundStepContentParam.paymentRefundAmt) {
                    paymentRefundAmts.add(new RefundInfoVO.RefundContentText("退回" + refundPaymentAmt.paymentType, "¥" + refundPaymentAmt.paymentRefundAmt.setScale(2, RoundingMode.DOWN), REFUND_CONTENT_BLACK));
                }
                refundStepInfo.setPaymentRefundAmts(paymentRefundAmts);
            }
            return refundStepInfo;
        }
    }, EXCHANGE_SEND("商家发货") {

        @Override
        public boolean canCancelRefund() {
            return false;
        }

        @Override
        public String getDesc() {
            return "商家发货";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.EXCHANGE_SEND);
            if (null == refundStepContentParam.failType) {
                refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("新商品已发货，请及时关注物流信息", "", REFUND_CONTENT_BLACK));
            }
            return refundStepInfo;
        }

    }, EXCHANGE_SUCCESS("换货完成") {

        @Override
        public boolean canCancelRefund() {
            return false;
        }

        @Override
        public String getDesc() {
            return "换货完成";
        }

        @Override
        public RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam) {
            RefundInfoVO.RefundStepInfo refundStepInfo = new RefundInfoVO.RefundStepInfo();
            refundStepInfo.setRefundStep(TsRefundStepEnum.EXCHANGE_SEND);
            if (null == refundStepContentParam.failType) {
                refundStepInfo.setTitle(new RefundInfoVO.RefundContentText("换货完成", "", REFUND_CONTENT_BLACK));
                refundStepInfo.setDesc(new RefundInfoVO.RefundContentText("换货服务已完成，如有疑问请联系客服", "", REFUND_CONTENT_GRAY));
                refundStepInfo.setDateText(new RefundInfoVO.RefundContentText(DateUtil.formatDate(refundStepContentParam.getCloseTime(), DateUtil.FORMAT_YYYY_MM_DD_HH_MM_SS), "", REFUND_CONTENT_GRAY));
            }
            return refundStepInfo;
        }
    }

    ;
    /**
     * 文案黄色色值
     */
    static final String REFUND_CONTENT_YELLOW = "#DD4F14";

    /**
     * 文案黑色色值
     */
    static final String REFUND_CONTENT_BLACK = "#000000";

    /**
     * 文案灰色色值
     */
    static final String REFUND_CONTENT_GRAY = "#9A9A9A";
    private final String name;

    TsRefundStepEnum(final String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }


    /**
     * 是否可以取消退款
     *
     * @return
     */
    public abstract boolean canCancelRefund();


    /**
     * 获取当前状态的文案
     */
    public abstract RefundInfoVO.RefundStepInfo getContent(RefundStepContentParam refundStepContentParam);

    /**
     * 获取操作描述
     *
     * @return
     */
    public abstract String getDesc();

    /**
     * 售后的阶段
     *
     * @param refundStatus
     * @return
     */
    public static TsRefundStepEnum getByRefundStatus(Integer refundStatus) {
        if (ReturnRequestState.STATUS_CREATE.equals(refundStatus)
                || ReturnRequestState.STATUS_WAITING_CHECK.equals(refundStatus)) {
            return TsRefundStepEnum.APPLY;
        }
        if (ReturnRequestState.STATUS_BUYER_DELIVER.equals(refundStatus)) {
            return TsRefundStepEnum.ALLOW;
        }
        if (ReturnRequestState.STATUS_WAITING_RECEIVE.equals(refundStatus)
                || ReturnRequestState.STATUS_WAITING_CHECK_AFTER_RECEIVE.equals(refundStatus)
                || ReturnRequestState.STATUS_SELLEL_DELIVERY.equals(refundStatus)) {
            return TsRefundStepEnum.RETURN;
        }
        if (ReturnRequestState.STATUS_WAITING_REFUND.equals(refundStatus)) {
            return TsRefundStepEnum.REFUND;
        }
        if (ReturnRequestState.STATUS_FINISHED.equals(refundStatus)) {
            return TsRefundStepEnum.REFUND_SUCCESS;
        }
        return null;
    }

    /**
     * 售后申请状态
     */
    public static class ReturnRequestState {

        /**
         * 发起
         */
        public static final Integer STATUS_CREATE = 1;

        /**
         * 待审核(待处理,退款待处理)
         */
        public static final Integer STATUS_WAITING_CHECK = 2;

        /**
         * 待入库
         */
        public static final Integer STATUS_WAITING_RECEIVE = 3;

        /**
         * 待审核入库情况
         */
        public static final Integer STATUS_WAITING_CHECK_AFTER_RECEIVE = 4;

        /**
         * 待退款
         */
        public static final Integer STATUS_WAITING_REFUND = 5;

        /**
         * 卖家拒绝
         */
        public static final Integer STATUS_BUYER_DELIVER = 7;

        /**
         * 待商家收货
         */
        public static final Integer STATUS_SELLEL_DELIVERY = 8;

        /**
         * 完成
         */
        public static final Integer STATUS_FINISHED = 9;

    }



    /**
     * 售后步骤文案入参
     */
    public static class RefundStepContentParam {

        /**
         * 售后类型
         */
        private TsRefundTypeEnum refundType;

        /**
         * 当前步骤失败类型
         */
        private RefundStepFailType failType;

        /**
         * 同意倒计时
         */
        private String acceptLimitCountDownTimeStr;

        /**
         * 处理申请工作日时间
         */
        private String acceptWorkDayNum;

        /**
         * 商家拒绝后的订单关闭倒计时
         */
        private String sellRejectCloseCountDownTimeStr;

        /**
         * 填写物流单号关闭时间
         */
        private String deliveryCloseCountDownTimeStr;

        /**
         * 退款限时
         */
        private String refundWorkDayNum;

        /**
         * 退款金额
         */
        private BigDecimal refundAmt;

        /**
         * 退款时间
         */
        private BigDecimal refundDate;

        /**
         * 支付渠道退款金额
         */
        private List<RefundPaymentAmt> paymentRefundAmt;

        /**
         * 退款时间
         */
        private Date refundTime;

        /**
         * 退款失败时间
         */
        private Date refundFailTime;

        /**
         * 售后单关闭时间
         */
        private Date closeTime;

        /**
         * 拒绝售后理由
         */
        private String rejectReason;

        /**
         * 关闭原因
         */
        private String closeReason;

        public String getAcceptWordDayNum() {
            return acceptWorkDayNum;
        }

        public void setAcceptWorkDayNum(String acceptWorkDayNum) {
            this.acceptWorkDayNum = acceptWorkDayNum;
        }

        public TsRefundTypeEnum getRefundType() {
            return refundType;
        }

        public void setRefundType(TsRefundTypeEnum refundType) {
            this.refundType = refundType;
        }

        public RefundStepFailType getFailType() {
            return failType;
        }

        public void setFailType(RefundStepFailType failType) {
            this.failType = failType;
        }

        public BigDecimal getRefundAmt() {
            return refundAmt;
        }

        public void setRefundAmt(BigDecimal refundAmt) {
            this.refundAmt = refundAmt;
        }

        public BigDecimal getRefundDate() {
            return refundDate;
        }

        public void setRefundDate(BigDecimal refundDate) {
            this.refundDate = refundDate;
        }

        public List<RefundPaymentAmt> getPaymentRefundAmt() {
            return paymentRefundAmt;
        }

        public void setPaymentRefundAmt(List<RefundPaymentAmt> paymentRefundAmt) {
            this.paymentRefundAmt = paymentRefundAmt;
        }

        public String getRejectReason() {
            return rejectReason;
        }

        public void setRejectReason(String rejectReason) {
            this.rejectReason = rejectReason;
        }

        public String getCloseReason() {
            return closeReason;
        }

        public void setCloseReason(String closeReason) {
            this.closeReason = closeReason;
        }

        public String getAcceptLimitCountDownTimeStr() {
            return acceptLimitCountDownTimeStr;
        }

        public void setAcceptLimitCountDownTimeStr(String acceptLimitCountDownTimeStr) {
            this.acceptLimitCountDownTimeStr = acceptLimitCountDownTimeStr;
        }

        public String getSellRejectCloseCountDownTimeStr() {
            return sellRejectCloseCountDownTimeStr;
        }

        public void setSellRejectCloseCountDownTimeStr(String sellRejectCloseCountDownTimeStr) {
            this.sellRejectCloseCountDownTimeStr = sellRejectCloseCountDownTimeStr;
        }

        public String getDeliveryCloseCountDownTimeStr() {
            return deliveryCloseCountDownTimeStr;
        }

        public void setDeliveryCloseCountDownTimeStr(String deliveryCloseCountDownTimeStr) {
            this.deliveryCloseCountDownTimeStr = deliveryCloseCountDownTimeStr;
        }

        public String getRefundWorkDayNum() {
            return refundWorkDayNum;
        }

        public void setRefundWorkDayNum(String refundWorkDayNum) {
            this.refundWorkDayNum = refundWorkDayNum;
        }

        public Date getRefundTime() {
            return refundTime;
        }

        public void setRefundTime(Date refundTime) {
            this.refundTime = refundTime;
        }

        public Date getRefundFailTime() {
            return refundFailTime;
        }

        public void setRefundFailTime(Date refundFailTime) {
            this.refundFailTime = refundFailTime;
        }

        public Date getCloseTime() {
            return closeTime;
        }

        public void setCloseTime(Date closeTime) {
            this.closeTime = closeTime;
        }

        public static class RefundPaymentAmt {

            /**
             * 退款的支付渠道
             */
            private String paymentType;

            /**
             * 支付渠道退款金额
             */
            private BigDecimal paymentRefundAmt;

            public RefundPaymentAmt() {
            }

            public RefundPaymentAmt(String paymentType, BigDecimal paymentRefundAmt) {
                this.paymentType = paymentType;
                this.paymentRefundAmt = paymentRefundAmt;
            }

            public String getPaymentType() {
                return paymentType;
            }

            public void setPaymentType(String paymentType) {
                this.paymentType = paymentType;
            }

            public BigDecimal getPaymentRefundAmt() {
                return paymentRefundAmt;
            }

            public void setPaymentRefundAmt(BigDecimal paymentRefundAmt) {
                this.paymentRefundAmt = paymentRefundAmt;
            }
        }
    }


    /**
     * 售后失败原因
     */
    public enum RefundStepFailType {

        /**
         * 关闭
         */
        CLOSE,

        /**
         * 卖家拒绝
         */
        SELL_REJECT,

    }


    /**
     * 售后状态
     */
    public enum RefundSimpleStatusType {

        /**
         * 申请中
         */
        APPLY("申请中"),

        /**
         * 退款成功
         */
        SUCCESS("退款成功"),

        /**
         * 售后关闭
         */
        CLOSE("售后关闭");

        public String desc;

        RefundSimpleStatusType(String desc) {
            this.desc = desc;
        }

        public String getDesc() {
            return desc;
        }
    }
}
