package com.thebeastshop.salesorder.service;

import java.util.Date;
import java.util.List;
import java.util.Map;

import com.thebeastshop.common.PageQueryResp;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.salesorder.dto.SoOrderCond;
import com.thebeastshop.salesorder.vo.*;
import com.thebeastshop.salesorder.vo.pub.*;
import com.thebeastshop.salesorder.vo.simple.SoPsSimpleOrderVO;

/**
 * 订单服务【面向前端】
 *
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-02-24 13:56
 */
public interface SoPsOrderService {

    /**
     * 订单列表
     *
     * @return List<OpSalesOrder>
     */
    PageQueryResp<SoPsOrderVO> list(Long id, String memberCode, String code, Integer currpage, int pagenum);

    /**
     * 订单列表
     *
     * @param cond
     * @param currpage
     * @param pagenum
     * @return
     */
    PageQueryResp<SoPsOrderVO> list(SoOrderCond cond, Integer currpage, Integer pagenum);

    /**
     * 已发货订单列表
     *
     * @param memberCode
     * @param currpage
     * @param pagenum
     * @return
     */
    PageQueryResp<SoPsOrderVO> deliveringOrderList(String memberCode, Integer currpage, Integer pagenum);

    /**
     * 根据memberCode查询用户全渠道订单
     *
     * @param memberCode
     * @param currpage
     * @param pagenum
     * @return
     */
    PageQueryResp<SoPsOrderVO> list(String memberCode, Integer currpage, Integer pagenum);

    /**
     * 根据memberCode查询用户全渠道订单
     *
     * @param code
     * @param currpage
     * @param pagenum
     * @return
     */
    List<SoPsOrderVO> ordersByCode(String code, Integer currpage, Integer pagenum, Date startDate, Date endDate);

    /**
     * 根据code，查询渠道订单数量
     *
     * @param code
     * @return
     */
    Integer getOrderCountByCode(String code, Date startDate, Date endDate);

    /**
     * 根据membercode，查询用户全渠道订单数量
     *
     * @param memberCode
     * @return
     */
    Integer getOrderCount(String memberCode);

    /**
     * 根据membercode，查询用户已发货订单数量
     *
     * @param memberCode
     * @return
     */
    Integer getDeliveringOrderCount(String memberCode);

    /**
     * 通用 根据订单号 查询订单详情
     *
     * @param orderCode
     * @return
     */
    SoPsOrderVO detail(String memberCode, String orderCode);

    /**
     * 通用 根据订单号，查询简单订单详情
     *
     * @param orderCode
     * @return
     */
    SoPsSimpleOrderVO simpleDetail(String orderCode);

    /**
     * 通用 根据订单号，查询订单
     *
     * @param orderCode
     * @return
     */
    SoPsOrderVO getOrderByIdOrCode(Long id, String orderCode);

    /**
     * 查询记录数
     *
     * @param id
     * @param memberCode
     * @param code
     * @return
     */
    Integer getCount(Long id, String memberCode, String code);

    /**
     * 添加记录数
     *
     * @param cond
     * @return
     */
    Integer getCount(SoOrderCond cond);

    /**
     * 创建订单
     *
     * @return
     */
    SoOrderResponseVO create(SoOrderVO soOrderVO, AccessWayEnum accessWay);

    /**
     * 订单详情
     *
     * @param orderId
     * @return OpSalesOrder
     */
    SoPsOrderVO detail(Long orderId);

    /**
     * 根据订单号和渠道编码，查询订单
     *
     * @param orderCode
     * @param channelCode
     * @return
     */
    SoPsOrderVO getOrderByCode(String orderCode, String channelCode);

    /**
     * 根据订单号，查询订单
     *
     * @param orderCode
     * @return
     */
    SoPsOrderVO getOrderByCode(String orderCode);

    /**
     * 根据订单号查询订单是否已取消
     *
     * @param orderCode
     * @return boolean
     */
    Boolean orderSpecilIsCancel(String orderCode);

    /**
     * 根据订单号查询订单状态
     *
     * @param orderCode
     * @return
     */
    Integer orderStatus(String orderCode);

    /**
     * 根据订单ID逻辑删除
     *
     * @param orderId
     * @return
     */
    Integer delete(Long orderId);

    /**
     * 根据会员编号和订单号逻辑删除
     *
     * @param memberCode
     * @param orderCode
     * @return
     */
    Integer delete(String memberCode, String orderCode);

    /**
     * 查询5分钟前的订单数据 查询驿氪所需要的订单数据，用于定时推送订单信息
     *
     * @return boolean
     */
    List<SoShoppingTicketsVO> getOrderInfoForEZR();

    /**
     * 查询当天门店销售基本信息 查询驿氪所需要的门店数据，用于定时推送门店销售信息
     *
     * @return boolean
     */
    List<SoStoreDailyTradingVO> getChanelSaleInfoForEZR();

    /**
     * 查询5分钟前的退货数据 查询驿氪所需要的订单数据，用于定时推送订单信息
     *
     * @return boolean
     */
    List<SoShoppingTicketsVO> getRefundOrderInfoForEZR();

    /**
     * 根据订单Id，查询包裹商品
     *
     * @param orderId
     * @return
     */
    List<SoPsPackageSkuVO> getSkusByOrderId(Long orderId);

    /**
     * 取消订单
     *
     * @param soCode
     * @return
     * @throws Exception
     */
    Boolean cancleSalesOrder(final String soCode) throws Exception;

    /**
     * 判断是否首单 已支付 或 已取消
     *
     * @param memberCode
     * @param channelCode
     * @return
     */
    public boolean ifFirstOrder(String memberCode, String channelCode);

    /**
     * 判断是否首单
     *
     * @param memberCode
     * @param channelCodeList
     * @return
     */
    public boolean ifFirstOrderByChannels(String memberCode, List<String> channelCodeList);

    /**
     * 合并订单
     *
     * @param taskId         合并任务ID
     * @param mainMemberCode 主账号编号
     * @param subMemberCode  子账号编号
     * @return
     */
    Boolean merge(Long taskId, String mainMemberCode, String subMemberCode);

    /**
     * 删除redis
     *
     * @param key
     * @return Boolean
     */
    Boolean removeRedis(String key);

    /**
     * 判断是否有已付款订单
     *
     * @param channelCode
     * @param memberCode
     * @return true有已付款订单 false 么有已付款订单
     * @Author WangYawei
     * @Date 2017-03-15
     */
    Boolean ifPaidOrder(String memberCode, String channelCode);

    /**
     * 判断是否有 已支付且未取消
     *
     * @param memberCode
     * @param channelCode
     * @return
     */
    Boolean ifPaidAndUnCancel(String memberCode, String channelCode);

    /**
     * 判断是否有 实付未取消订单(排除当前订单)
     *
     * @param beastMemberCode
     * @return
     */
    @Deprecated
    Boolean ifActuallyPayAndExclude(String beastMemberCode, String orderCode);

    /**
     * 查询 实付未取消订单-仅订单数据
     *
     * @param beastMemberCode
     * @return
     */
    List<SoPsOrderVO> findActuallyPayUnCancelList(String beastMemberCode);

    /**
     * 根据订单号、渠道编码，查询订单
     *
     * @param orderCode
     * @return
     */
    List<SoPsOrderVO> findOrderByCodes(List<String> orderCode);

    /**
     * 根据渠道code、时间段，查询支付订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodePayment(String channelCode, Date startTime, Date endTime);

    /**
     * 根据渠道code、时间段，查询支付订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodePaymentItem(String channelCode, Date startTime, Date endTime);

    /**
     * 根据渠道code、时间段，查询退款订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodeCancel(String channelCode, Date startTime, Date endTime);

    /**
     * 根据渠道code、时间段，查询退款订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodeCancelItem(String channelCode, Date startTime, Date endTime);

    /**
     * 判断当前订单是否已支付
     *
     * @param orderCode
     * @return
     */
    public boolean isPayByOrderCode(String orderCode);

    /**
     * 根据会员code，查询使用生日花权益的订单
     *
     * @param memberCode
     * @return SoPsOrderVO
     */
    SoPsOrderVO findUserBirthFlowerByMemberCode(String memberCode);

    /**
     * 根据会员code，查询使用生日折扣的订单
     *
     * @param memberCode
     * @return SoPsOrderVO
     */
    SoPsOrderVO findUserBirthDiscountByMemberCode(String memberCode);

    /**
     * 是否有支付订单
     *
     * @param memberCode
     * @param startDate
     * @param endDate
     * @return
     */
    Boolean hasPayOrder(String memberCode, Date startDate, Date endDate);

    /**
     * 根据渠道code、时间段，查询已支付订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodePaymentAndItem(String channelCode, Date startTime, Date endTime);

    /**
     * 根据渠道code、时间段，查询退款订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodeCancelAndItem(String channelCode, Date startTime, Date endTime);

    /**
     * 取消订单
     *
     * @param soCode
     * @param cancelReason
     * @return
     */
    ResultVO cancleSalesOrder(String soCode, String cancelReason);

    /**
     * 根据订单号查询订单上报信息
     *
     * @param orderCode
     * @return
     */
    SoPsOrderReportVO findOrderByOrderCode(String orderCode);

    /**
     * 根据渠道code、时间段查，询退款订单
     *
     * @param channelCode
     * @param startTime
     * @param endTime
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByChannelCodeReturn(String channelCode, Date startTime, Date endTime);

    /**
     * 根据订单号查询退款订单
     *
     * @param orderCode
     * @return List<SoPsOrderVO>
     */
    List<SoPsOrderReportVO> findOrderByOrderCodeReturn(String orderCode);

    /**
     * 根据退货单号查询退货单
     *
     * @param returnCode
     * @return
     */
    SoPsOrderReportVO findOrderByReturnCode(String returnCode);

    /**
     * 根据订单Id逻辑删除
     *
     * @param id
     * @param reason
     * @return
     */
    Integer deleteOrder(Long id, String reason);

    /**
     * 取得配方定制商品的配送日期
     *
     * @param skuVO
     * @return 配送日期
     */
    Date getCustomizePlanedDeliveryDate(SoPackageSkuVO skuVO);

    /**
     * 根据包裹ID查询订单包裹
     *
     * @param packageId
     * @return
     */
    SoPsPackageVO findSoPsPackageById(Long packageId);

    /**
     * 是否首单
     *
     * @param memberCode
     * @param number
     * @return
     */
    boolean ifFirstOrderByMemberCode(String memberCode, int number);

    /**
     * ipos打印小票订单查询接口(深圳e码头店提模式)
     *
     * @param soCode,channelCode
     * @return SoHaiTaoTicketVO
     */
    List<SoHaiTaoTicketVO> getOrderBySoCodeOrChannelCode(String soCode, String channelCode);

    /**
     * 根据订单号打印小票纪录打印次数（深圳e码头店提模式)
     *
     * @param soCode
     */
    void recordEbayTicketPrintNumBySoId(String soCode);

    /**
     * 第三方订单号
     *
     * @param orderCode
     * @return
     */
    List<String> thirdpartyOrders(String orderCode);

    /**
     * 创建订单号
     *
     * @return
     */
    String createSalesOrderCode(String channelCode, Integer salesOrderType);

    /**
     * 查询商品权益订单号
     *
     * @param memberCode
     * @param interestGoodsId
     * @return
     */
    String findOrderCodeByInterestGoods(String memberCode, Integer interestGoodsId);

    /**
     * 查询packgeSku的扩展信息
     *
     * @param packageSkuIds 需要查询的package sku 订单行id
     * @return
     */
    List<SoPsPackageSkuInnerVO> findPackageSkuInnerInfos(List<Integer> packageSkuIds);

    /**
     * package sku Id查询order
     * @return
     */
    SoPsOrderVO findSimpleSoOrderByPackageSkuId(Long packageSkuId);

    /**
     * 通用 根据订单Id 查询订单详情
     * 使用code的逻辑
     * @param memberCode
     * @return
     */
    SoPsOrderVO detailBySalesOrderId(Long salesOrderId, String memberCode);

    /**
     * packageSkuId 查询packageSku
     */
    SoPsPackageSkuVO getSimplePackageSkuById(Long packageSkuId);

    /**
     * 查询订单id的所用的礼品卡
     */
    Map<Integer,List<String>> getUsedGiftCardNoListBySoIds(List<Integer> salesOrderIds);

    /**
     * 订单取消返回sku限量-龚俊花神礼盒(临时接口)
     * @param salesOrderId
     * @return
     */
    void refundSkuStockLimit(Long memberId, Long salesOrderId);

}
