package com.thebeastshop.kit.flowable;

import com.thebeastshop.kit.flowable.Exceptions.FlowableException;
import com.thebeastshop.kit.flowable.bean.FlowGroup;
import com.thebeastshop.kit.flowable.bean.FlowUser;
import com.thebeastshop.kit.flowable.bean.NextTask;
import org.apache.commons.collections4.MapUtils;
import org.flowable.engine.*;
import org.flowable.engine.history.HistoricActivityInstance;
import org.flowable.engine.repository.Deployment;
import org.flowable.engine.repository.ProcessDefinition;
import org.flowable.engine.runtime.ProcessInstance;
import org.flowable.idm.api.Group;
import org.flowable.idm.api.User;
import org.flowable.idm.engine.impl.persistence.entity.GroupEntityImpl;
import org.flowable.idm.engine.impl.persistence.entity.UserEntityImpl;
import org.flowable.task.api.Task;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

/**
 * @author Bryan.Zhang
 * @Date 2019-06-06
 */
public class FlowableEngine {

    private final Logger log = LoggerFactory.getLogger(getClass());

    private ProcessEngine processEngine;

    public FlowableEngine(ProcessEngine processEngine) {
        this.processEngine = processEngine;
    }

    public void deploy(String resource,String deployName,String category){
        try{
            RepositoryService repositoryService = processEngine.getRepositoryService();
            Deployment deployment = repositoryService.createDeployment().addClasspathResource(resource)
                    .name(deployName)
                    .category(category)
                    .deploy();
            ProcessDefinition processDefinition = repositoryService.createProcessDefinitionQuery()
                    .deploymentId(deployment.getId()).singleResult();
            log.info("[FLOWABLE]流程资源[{}]已经被成功部署! 流程名称:[{}],流程ID:[{}],流程KEY:[{}]",
                    resource,
                    processDefinition.getName(),
                    processDefinition.getId(),
                    processDefinition.getKey());
        }catch (Throwable t){
            String errMsg = "flowable process deploy error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public List<NextTask> startProcess(String processId,String bizKey,String applyUser){
        try{
            // 设置流程创建人
            IdentityService identityService = processEngine.getIdentityService();
            identityService.setAuthenticatedUserId(applyUser);

            // 启动流程
            RuntimeService runtimeService = processEngine.getRuntimeService();
            ProcessInstance processInstance = runtimeService.startProcessInstanceByKey(processId, bizKey);
            log.info("[FLOWABLE]流程实例ID = {}",processInstance.getId());
            log.info("[FLOWABLE]正在活动的流程节点ID = {}",processInstance.getActivityId());
            log.info("[FLOWABLE]流程定义ID = {}",processInstance.getProcessDefinitionId());
            log.info("[FLOWABLE]部署ID = {}",processInstance.getDeploymentId());
            log.info("[FLOWABLE]业务KEY = {}",processInstance.getBusinessKey());
            return queryActiveTask(processInstance.getId());
        }catch (Throwable t){
            String errMsg = "flowable process start error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public void completeTask(Task task, Map<String, Object> vars){
        try{
            TaskService taskService = processEngine.getTaskService();
            RuntimeService runtimeService = processEngine.getRuntimeService();
            ProcessInstance processInstance = runtimeService.createProcessInstanceQuery().processInstanceId(task.getProcessInstanceId()).singleResult();
            log.info("[FLOWABLE]任务名称 = {}",task.getName());
            log.info("[FLOWABLE]流程实例ID = {}",processInstance.getId());
            log.info("[FLOWABLE]正在活动的流程节点ID = {}",processInstance.getActivityId());
            log.info("[FLOWABLE]流程定义ID = {}",processInstance.getProcessDefinitionId());

            if(vars == null || MapUtils.isEmpty(vars)){
                taskService.complete(task.getId());
            }else{
                taskService.complete(task.getId(),vars);
            }
        }catch (Throwable t){
            String errMsg = "flowable task complete error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public void completeTask(String taskId, Map<String, Object> vars){
        TaskService taskService = processEngine.getTaskService();
        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
        completeTask(task,vars);
    }

    public void completeTask(String taskId){
        completeTask(taskId,new HashMap<>());
    }

    public List<NextTask> completeTask(String processInstanceId, String assignment, Map<String, Object> vars){
        TaskService taskService = processEngine.getTaskService();
        Task task = taskService.createTaskQuery().processInstanceId(processInstanceId)
                .taskAssignee(assignment).singleResult();
        completeTask(task,vars);
        return queryActiveTask(processInstanceId);
    }

    public List<NextTask> completeTask(String processInstanceId, String assignment){
        return completeTask(processInstanceId, assignment, new HashMap<>());
    }

    public void terminateProcess(String processInstanceId,String terminateReason){
        try{
            RuntimeService runtimeService = processEngine.getRuntimeService();
            runtimeService.deleteProcessInstance(processInstanceId,terminateReason);
        }catch (Throwable t){
            String errMsg = "flowable process terminate error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public List<Task> queryTask(String assignment, String processInstanceId, int firstResult, int maxResults){
        try{
            TaskService taskService = processEngine.getTaskService();
            List<Task> taskList = taskService.createTaskQuery().taskAssignee(assignment)
                    .processInstanceId(processInstanceId).orderByTaskCreateTime().desc().listPage(firstResult,maxResults);
            return taskList;
        }catch (Throwable t){
            String errMsg = "flowable task query error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public List<Task> queryTask(String assignment, int firstResult, int maxResults){
        try{
            TaskService taskService = processEngine.getTaskService();
            List<Task> taskList = taskService.createTaskQuery().taskAssignee(assignment).orderByTaskCreateTime().desc().listPage(firstResult,maxResults);
            return taskList;
        }catch (Throwable t){
            String errMsg = "flowable task query error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public List<NextTask> queryActiveTask(String processInstanceId){
        try{
            TaskService taskService = processEngine.getTaskService();
            List<Task> taskList = taskService.createTaskQuery().processInstanceId(processInstanceId).active().orderByTaskCreateTime().desc().list();
            return transform(taskList);
        }catch (Throwable t){
            String errMsg = "flowable task query error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public List<ProcessInstance> queryActiveProcessInstance(int firstResult, int maxResults){
        RuntimeService runtimeService = processEngine.getRuntimeService();
        List<ProcessInstance> processInstanceList = runtimeService.createProcessInstanceQuery().active().orderByStartTime().desc().listPage(firstResult,maxResults);
        return processInstanceList;
    }

    public List<ProcessInstance> queryAllProcessInstance(int firstResult, int maxResults){
        RuntimeService runtimeService = processEngine.getRuntimeService();
        List<ProcessInstance> processInstanceList = runtimeService.createProcessInstanceQuery().orderByStartTime().desc().listPage(firstResult,maxResults);
        return processInstanceList;
    }

    public ProcessInstance queryProcessInstanceByBizKey(String bizKey){
        RuntimeService runtimeService = processEngine.getRuntimeService();
        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery().processInstanceBusinessKey(bizKey).singleResult();
        return processInstance;
    }

    public List<HistoricActivityInstance> queryHisAct(String processInstanceId){
        List<HistoricActivityInstance> list = processEngine.getHistoryService()
                .createHistoricActivityInstanceQuery().processInstanceId(processInstanceId)
                .orderByHistoricActivityInstanceStartTime().asc().list();
        return list;
    }

    public void createUser(FlowUser flowUser){
        try{
            IdentityService identityService = processEngine.getIdentityService();
            User tuser = identityService.newUser(flowUser.getUserId());
            identityService.saveUser(tuser);

            User user = new UserEntityImpl();
            user.setId(flowUser.getUserId());
            user.setEmail(flowUser.getEmail());
            user.setFirstName(flowUser.getFirstName());
            user.setLastName(flowUser.getLastName());
            identityService.saveUser(user);

            user.setPassword(flowUser.getPassword());
            identityService.updateUserPassword(user);
        }catch (Throwable t){
            String errMsg = "flowable user create error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public void createGroup(FlowGroup flowGroup){
        try{
            IdentityService identityService = processEngine.getIdentityService();
            Group tgroup = identityService.newGroup(flowGroup.getGroupId());
            identityService.saveGroup(tgroup);

            Group group = new GroupEntityImpl();
            group.setId(flowGroup.getGroupId());
            group.setName(flowGroup.getGroupName());
            identityService.saveGroup(group);
        }catch (Throwable t){
            String errMsg = "flowable group create error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public void createMembership(String userId, String groupId){
        try{
            processEngine.getIdentityService().createMembership(userId, groupId);
        }catch (Throwable t){
            String errMsg = "flowable merbership create error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public void deleteUser(String userId){
        try{
            IdentityService identityService = processEngine.getIdentityService();
            identityService.deleteUser(userId);
        }catch (Throwable t){
            String errMsg = "flowable user delete error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public FlowUser queryUser(String userId){
        try{
            IdentityService identityService = processEngine.getIdentityService();
            User user = identityService.createUserQuery().userId(userId).singleResult();
            if(user == null){
                return null;
            }
            FlowUser flowUser = transformUser(user);

            List<Group> groupList = identityService.createGroupQuery().groupMember(flowUser.getUserId()).list();
            List<FlowGroup> flowGroupList = new ArrayList<>();
            groupList.forEach(group -> flowGroupList.add(new FlowGroup(group.getId(),group.getName())));
            flowUser.setGroupList(flowGroupList);

            return flowUser;
        }catch (Throwable t){
            String errMsg = "flowable query user error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public FlowGroup queryGroup(String groupId){
        try{
            Group group = processEngine.getIdentityService().createGroupQuery().groupId(groupId).singleResult();
            if(group == null){
                return null;
            }
            FlowGroup flowGroup = new FlowGroup(group.getId(), group.getName());
            return flowGroup;
        }catch (Throwable t){
            String errMsg = "flowable query group error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    public List<FlowUser> queryGroupUser(String groupId){
        try{
            IdentityService identityService = processEngine.getIdentityService();
            List<User> userList = identityService.createUserQuery().memberOfGroup(groupId).list();
            List<FlowUser> flowUserList = new ArrayList<>();
            userList.forEach(user -> flowUserList.add(transformUser(user)));
            return flowUserList;
        }catch (Throwable t){
            String errMsg = "flowable query group error";
            log.error(errMsg,t);
            throw new FlowableException(errMsg,t);
        }
    }

    private List<NextTask> transform(List<Task> taskList){
        List<NextTask> nextTaskList = new ArrayList<>();
        taskList.forEach(task -> {
            NextTask nextTask = new NextTask();
            nextTask.setTaskId(task.getId());
            nextTask.setTaskName(task.getName());
            nextTask.setProcessInstanceId(task.getProcessInstanceId());
            nextTask.setAssignee(task.getAssignee());
            nextTask.setTaskDefinationId(task.getTaskDefinitionId());
            nextTask.setTaskDefinationKey(task.getTaskDefinitionKey());
            nextTaskList.add(nextTask);
        });
        return nextTaskList;
    }

    private FlowUser transformUser(User user){
        FlowUser flowUser = new FlowUser();
        flowUser.setUserId(user.getId());
        flowUser.setFirstName(user.getFirstName());
        flowUser.setLastName(user.getLastName());
        flowUser.setEmail(user.getEmail());
        flowUser.setPassword(user.getPassword());
        return flowUser;
    }
}
