package com.thebeast.kit.alitool;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;
import com.aliyun.oss.model.ObjectMetadata;
import com.thebeastshop.common.enums.FileTypeEnum;
import com.thebeastshop.common.utils.DateUtil;
import com.thebeastshop.common.utils.StringUtil;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

/**
 * @author Paul-xiong
 * @date 2017年7月25日
 * @description 文件上传操作类
 */
public class FileUploadClient {

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	/**
	 * 基础路径
	 */
	private static final String BASE_PATH = "file/";

	private OSSClient ossClient;

	private String ossEndPoint;

	private String ossAccessId;

	private String ossAccessKey;

	private String ossBucketName;

	private String baseUrl;

	private final String KEY_IMG = "image/";

	/**
	 * 获得oss的文件key
	 *
	 * @param fileUrl
	 * @return
	 */
	private String getKey(String fileUrl) {
		return fileUrl.replace(baseUrl, "");
	}

	/**
	 * 处理额外路径
	 *
	 * @param extraPath
	 * @return
	 */
	private String buildExtraPath(String extraPath) {
		if (StringUtils.isNotBlank(extraPath)) {
			extraPath = extraPath.indexOf("/") == 0 ? extraPath.substring(1) : extraPath;
			extraPath += extraPath.lastIndexOf("/") == -1 ? "/" : "";
		} else {
			extraPath = "";
		}
		return extraPath;
	}

	/**
	 * 获取ossClient
	 *
	 * @return
	 */
	private OSSClient getOssClient() {
		if (ossClient == null) {
			ossClient = new OSSClient(ossEndPoint, ossAccessId, ossAccessKey);
		}
		return ossClient;
	}

	// --------------------getter setter----------------------

	public String getOssEndPoint() {
		return ossEndPoint;
	}

	public void setOssEndPoint(String ossEndPoint) {
		this.ossEndPoint = ossEndPoint;
	}

	public String getOssAccessId() {
		return ossAccessId;
	}

	public void setOssAccessId(String ossAccessId) {
		this.ossAccessId = ossAccessId;
	}

	public String getOssAccessKey() {
		return ossAccessKey;
	}

	public void setOssAccessKey(String ossAccessKey) {
		this.ossAccessKey = ossAccessKey;
	}

	public String getOssBucketName() {
		return ossBucketName;
	}

	public void setOssBucketName(String ossBucketName) {
		this.ossBucketName = ossBucketName;
	}

	// -------------------------------------------------------

	public String getBaseUrl() {
		return baseUrl;
	}

	public void setBaseUrl(String baseUrl) {
		this.baseUrl = baseUrl;
	}

	/**
	 * 上传文件
	 *
	 * @param localFilePath
	 *            本地文件路径
	 * @param extraPath
	 *            上传的文件自定义路径
	 * @param fileType
	 *            文件类型
	 * @return
	 */
	public FileUploadResp uploadFile(String localFilePath, String extraPath, FileTypeEnum fileType) {
		File file = new File(localFilePath);
		return this.uploadFile(file, extraPath, fileType);
	}
	public String uploadImg(String suffix, InputStream content) {
		String fileName = DateUtil.formatDate(new Date(), "yyyyMMddHHmmss")
				+ StringUtil.getRandomString(6, StringUtil.randomInts);

		return uploadImg(fileName, suffix, content);
	}
	public String uploadImg(String fileName, String suffix, InputStream content) {
		String key = KEY_IMG + fileName + "." + suffix;

		// 创建上传Object的Metadata
		ObjectMetadata meta = new ObjectMetadata();

		// 必须设置ContentLength
		//        meta.setContentLength(file.length());

		MimeType mimeType = MimeType.findByName(suffix);
		if (mimeType != null) {
			meta.setContentType(mimeType.getContentType());
		}
		getOssClient().putObject(ossBucketName, key, content);
		return key;
	}
	/**
	 * 上传文件
	 *
	 * @param file
	 *            文件类
	 * @param extraPath
	 *            上传的文件自定义路径
	 * @param fileType
	 *            文件类型
	 * @return
	 */
	public FileUploadResp uploadFile(File file, String extraPath, FileTypeEnum fileType) {
		FileUploadResp info = null;
		try {
			info = this.uploadFile(new FileInputStream(file), extraPath, fileType);
		} catch (Exception e) {
			logger.error("上传文件失败: {}", e);
		}
		return info;
	}

	/**
	 * 上传文件
	 *
	 * @param inputStream
	 *            文件流
	 * @param extraPath
	 *            上传的文件自定义路径
	 * @param fileType
	 *            文件类型
	 * @return
	 */
	public FileUploadResp uploadFile(InputStream inputStream, String extraPath, FileTypeEnum fileType) {
		FileUploadResp info = null;
		try {
			info = this.uploadFile(inputStream, extraPath, fileType, "");
		} catch (Exception e) {
			logger.error("上传文件异常: {}", e);
		}
		return info;
	}

	public FileUploadResp uploadFile(File file, String extraPath, FileTypeEnum fileType, String fileName) {
		FileUploadResp info = null;
		try {
			info = this.uploadFile(new FileInputStream(file), extraPath, fileType, fileName);
		} catch (Exception e) {
			logger.error("上传文件异常: {}", e);
		}
		return info;
	}

	public FileUploadResp uploadFile(InputStream inputStream, String extraPath, FileTypeEnum fileType, String fileName) {
		FileUploadResp info = null;
		try {
			if (StringUtils.isBlank(fileName)) {
				fileName = UUID.randomUUID().toString().replace("-", "");
			}
			String key = BASE_PATH + this.buildExtraPath(extraPath) + fileName + "." + fileType.getName();
			getOssClient().putObject(ossBucketName, key, inputStream);
			info = new FileUploadResp(inputStream, baseUrl + key);
		} catch (Exception e) {
			logger.error("上传文件异常: {}", e);
		}
		return info;
	}

	/**
	 * 获取文件列表
	 *
	 * @param extraPath
	 *            上传的文件自定义路径
	 * @return
	 */
	public List<FileUploadResp> listFiles(String extraPath) {
		List<FileUploadResp> list = null;
		try {
			ObjectListing listing = getOssClient().listObjects(ossBucketName, BASE_PATH + this.buildExtraPath(extraPath));
			list = new LinkedList<FileUploadResp>();
			for (OSSObjectSummary objectSummary : listing.getObjectSummaries()) {
				list.add(this.getFile(baseUrl + objectSummary.getKey()));
			}
		} catch (Exception e) {
			logger.error("获取文件列表异常: {}", e);
		}
		return list;
	}

	/**
	 * 获取上传文件
	 *
	 * @param fileUrl
	 *            文件路径
	 * @return
	 */
	public FileUploadResp getFile(String fileUrl) {
		FileUploadResp info = null;
		try {
			OSSObject ossObject = getOssClient().getObject(ossBucketName, this.getKey(fileUrl));
			info = new FileUploadResp(ossObject.getObjectContent(), fileUrl);
		} catch (Exception e) {
			logger.error("获取上传文件异常: {}", e);
		}
		return info;
	}

	/**
	 * 删除上传文件
	 *
	 * @param fileUrl
	 *            文件路径
	 */
	public void deleteFile(String fileUrl) {
		try {
			String key = this.getKey(fileUrl);
			getOssClient().deleteObject(ossBucketName, key);
		} catch (Exception e) {
			logger.error("删除上传文件异常: {}", e);
		}
	}

}
