package com.thebeastshop.common.prop;

import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigChangeListener;
import com.ctrip.framework.apollo.ConfigService;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfig;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.thebeastshop.common.prop.PropConstants;
import com.thebeastshop.common.prop.annotation.DynamicPropAccessor;
import com.thebeastshop.common.prop.annotation.DynamicPropValue;
import com.thebeastshop.common.prop.annotation.PropOnChange;
import com.thebeastshop.common.prop.annotation.PropOnChangeAccessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.core.Ordered;
import org.springframework.core.PriorityOrdered;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

/**
 * @author gongjun[jun.gong@thebeastshop.com]
 * @since 2017-07-07 17:17
 */
public class PropAnnotationProcessor implements BeanPostProcessor, PriorityOrdered {

    private static final Logger log = LoggerFactory.getLogger(PropAnnotationProcessor.class);


    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        Class clazz = bean.getClass();
        processFields(bean, clazz.getDeclaredFields());
        processMethods(bean, clazz.getDeclaredMethods());
        processPatternProcessor(bean);
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    private void processFields(Object bean, Field[] declaredFields) {
        for (Field field : declaredFields) {
            processDynamicPropValueAnn(bean, field);
        }
    }


    private void processDynamicPropValueAnn(Object bean, Field field) {
        DynamicPropValue annotation = AnnotationUtils.getAnnotation(field, DynamicPropValue.class);
        if (annotation == null) {
            return;
        }

        String propName = annotation.value();

        ReflectionUtils.makeAccessible(field);

        DynamicPropAccessor propAccessor = new DynamicPropAccessor(propName, bean, field);
        PropConstants.registerPropAccessor(bean.getClass().getName() + "@" + bean.hashCode(), propName, propAccessor);
        propAccessor.refreshValue();
    }


    private void processMethods(final Object bean, Method[] declaredMethods) {
        for (final Method method : declaredMethods) {
            PropOnChange annotation = AnnotationUtils.findAnnotation(method, PropOnChange.class);
            if (annotation == null) {
                continue;
            }

            Class<?>[] parameterTypes = method.getParameterTypes();
            Preconditions.checkArgument(parameterTypes.length == 1,
                    "Invalid number of parameters: %s for method: %s, should be 1", parameterTypes.length, method);
            Preconditions.checkArgument(PropChange.class.isAssignableFrom(parameterTypes[0]),
                    "Invalid parameter type: %s for method: %s, should be ConfigChangeEvent", parameterTypes[0], method);

            String propName = annotation.value();

            log.info("%%%% 扫描到@PropOnChange注解 PropName: " + propName
                    + " Class: " + bean.getClass().getName()
                    + " Method: " + method.getName());

            ReflectionUtils.makeAccessible(method);

            String beanId = bean.getClass().getName() + "@" + bean.hashCode();

            PropConstants.addPropChangeCallback(beanId, propName, new PropChangeCallback() {
                @Override
                public void onChange(PropChange propChange) {
                    ReflectionUtils.invokeMethod(method, bean, propChange);
                }
            });
        }
    }

    private void processPatternProcessor(Object bean) {
        if (bean instanceof PropPatternProcessor) {
            PropConstants.addPatternProcessor((PropPatternProcessor) bean);
        }
    }

}
