package com.thebeastshop.common.kafka;

import java.lang.reflect.ParameterizedType;
import java.nio.ByteBuffer;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.kafka.listener.AcknowledgingMessageListener;
import org.springframework.kafka.support.Acknowledgment;

import com.thebeastshop.common.ByteHandler;

/**
 * Kafka Json解析监听器抽象类
 * @param <V>
 */
public abstract class KafkaRecordConsumerListener<V extends ByteHandler> implements AcknowledgingMessageListener<String, ByteBuffer> {

	private final Logger log = LoggerFactory.getLogger(getClass());

	/**
	 * 重载GenericMessageListener.onMessage方法</BR>
	 * 当消费者端监听到有新的消息进来会调用该方法</BR>
	 * 然后会调用子类实现的processMessage抽象方法
	 * @param record
     */
	@Override
	public void onMessage(ConsumerRecord<String, ByteBuffer> record, Acknowledgment acknowledgment) {
		ByteBuffer data = record.value();
		log.info("receive message, type:{}, value:{}", record.value().getClass().getSimpleName(), record.value().toString());
		try {
			String key = record.key();
			Class<?> clazz = getClass(key);
			@SuppressWarnings("unchecked")
			V v = (V) clazz.newInstance();
			v.mergeFrom(data);
			this.processMessage(key, v);
		} catch (Exception e) {
			log.error("消费者端处理消息异常：", e);
		} finally {
			acknowledgment.acknowledge();
		}
	}
	
	/**
	 * 获取消息值对应Class类
	 * 
	 * @param key
	 * @return
	 */
	public Class<?> getClass(String key) {
		@SuppressWarnings("unchecked")
		Class<V> clazz = (Class<V>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
		return clazz;
	}

	/**
	 * 消息处理逻辑所在的方法，需要子类实现
	 * @param key
	 * @param v
     */
	public abstract void processMessage(String key,V v);
	
}
