package com.thebeastshop.common.lock;

import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.redisson.Redisson;
import org.redisson.api.RBatch;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import com.thebeastshop.common.prop.PropConstants;

/**
 * @author bryan.zhang
 * @date 2015-1-21
 * @description 基于Redis分布式锁的实现
 */
public class RedisDistributLock {

	private final Log log = LogFactory.getLog(this.getClass());

	private int defaultSingleExpireTime = 3;
	
	private int defaultBatchExpireTime = 6;

	private RedissonClient redissonClient;
	
	/**
	 * 获取锁  如果锁可用   立即返回true，  否则返回false
	 */
	public boolean tryLock(String key) {
		return tryLock(key, 0L, null);
	}

	/**
	 * 锁在给定的等待时间内空闲，则获取锁成功 返回true， 否则返回false
	 */
	public boolean tryLock(String key, long timeout, TimeUnit unit) {
		boolean result = false;
		RLock lock = redissonClient.getLock(key);
		try{
			result = lock.tryLock(timeout, defaultSingleExpireTime, unit);
		}catch(Exception e){
			e.printStackTrace();
			result = false;
		}
		return result;
	}
	
	
	public boolean tryLock(List<String> keyList, long timeout, TimeUnit unit) {
		boolean result = false;
		RLock lock = null;
		try{
			for(String key : keyList){
				lock = redissonClient.getLock(key);
				result = lock.tryLock(timeout, defaultSingleExpireTime, unit);
				if(!result){
					return false;
				}
			}
		}catch(Exception e){
			e.printStackTrace();
			result = false;
		}
		return true;
	}

	/**
	 * 如果锁空闲立即返回   获取失败 一直等待
	 */
	public void lock(String key) {
		RLock lock = redissonClient.getLock(key);
		try{
			lock.lock();
		}catch(Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * 释放锁
	 */
	public void unLock(String key) {
		RLock lock = redissonClient.getLock(key);
		try{
			lock.unlock();
		}catch(Exception e){
			e.printStackTrace();
		}
	}
	
	public void unLock(List<String> keyList) {
		RLock lock = null;
		try{
			for(String key : keyList){
				lock = redissonClient.getLock(key);
				lock.unlock();
			}
		}catch(Exception e){
			e.printStackTrace();
		}
	}
	
	/**
	 * 判断这个key是不是已经被锁住了
	 */
	public boolean isLock(String key){
		boolean result = false;
		RLock lock = redissonClient.getLock(key);
		try{
			result = lock.isLocked();
		}catch(Exception e){
			e.printStackTrace();
		}
		return result;
	}
	
	/**
	 * 判断这个key是不是被当前线程占用
	 */
	public boolean isHeldByCurrThread(String key){
		boolean result = false;
		RLock lock = redissonClient.getLock(key);
		try{
			result = lock.isHeldByCurrentThread();
		}catch(Exception e){
			e.printStackTrace();
		}
		return result;
	}

	public int getDefaultSingleExpireTime() {
		return defaultSingleExpireTime;
	}

	public void setDefaultSingleExpireTime(int defaultSingleExpireTime) {
		this.defaultSingleExpireTime = defaultSingleExpireTime;
	}

	public int getDefaultBatchExpireTime() {
		return defaultBatchExpireTime;
	}

	public void setDefaultBatchExpireTime(int defaultBatchExpireTime) {
		this.defaultBatchExpireTime = defaultBatchExpireTime;
	}

	public RedissonClient getRedissonClient() {
		return redissonClient;
	}

	public void setRedissonClient(RedissonClient redissonClient) {
		this.redissonClient = redissonClient;
	}
}