/*
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.service.purchase.service.impl;

import com.thebeastshop.pegasus.service.purchase.cond.PrdcRecipeCond;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcRecipeMapper;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcRecipeMaterialMapper;
import com.thebeastshop.pegasus.service.purchase.exception.PurchaseException;
import com.thebeastshop.pegasus.service.purchase.exception.PurchaseExceptionErrorCode;
import com.thebeastshop.pegasus.service.purchase.model.PrdcRecipe;
import com.thebeastshop.pegasus.service.purchase.model.PrdcRecipeExample;
import com.thebeastshop.pegasus.service.purchase.model.PrdcRecipeMaterial;
import com.thebeastshop.pegasus.service.purchase.model.PrdcRecipeMaterialExample;
import com.thebeastshop.pegasus.service.purchase.service.PcsSkuService;
import com.thebeastshop.pegasus.service.purchase.service.PrdcRecipeService;
import com.thebeastshop.pegasus.service.purchase.vo.PcsSkuVO;
import com.thebeastshop.pegasus.service.purchase.vo.PrdcRecipeMaterialVO;
import com.thebeastshop.pegasus.service.purchase.vo.PrdcRecipeVO;
import com.thebeastshop.pegasus.util.comm.BeanUtil;
import com.thebeastshop.pegasus.util.comm.DateUtil;
import com.thebeastshop.pegasus.util.comm.EmptyUtil;
import com.thebeastshop.pegasus.util.comm.NullUtil;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Eric.Lou
 * @version $Id: PrdcRecipeServiceImpl.java, v 0.1 2015-12-30 17:11
 */
@Service("prdcRecipeService")
public class PrdcRecipeServiceImpl implements PrdcRecipeService {

    @Autowired
    private PcsSkuService pcsSkuService;

    @Autowired
    private PrdcRecipeMapper prdcRecipeMapper;

    @Autowired
    private PrdcRecipeMaterialMapper prdcRecipeMaterialMapper;



    @Override
    public PrdcRecipeVO findRecipeVOById(Long id) {
        PrdcRecipeCond cond = new PrdcRecipeCond();
        cond.setRecipeId(id);
        cond.setCurrpage(1);
        List<PrdcRecipeVO> prdcRecipeVOList = findRecipeVOByCond(cond, true);
        if(CollectionUtils.isEmpty(prdcRecipeVOList)){
            return null;
        }else{
            return prdcRecipeVOList.get(0);
        }
    }
    
    private List<PrdcRecipeMaterialVO> findRecipeMaterialVOByRecipeIds(List<Long> recipeIds){
    	return prdcRecipeMaterialMapper.findRecipeMaterialVOByRecipeIds(recipeIds);
    }

    @Override
    public List<PrdcRecipeMaterialVO> findRecipeMaterialVOByRecipeId(Long recipeId) {
        PrdcRecipeMaterialExample example = new PrdcRecipeMaterialExample();
        example.createCriteria().andRecipeIdEqualTo(recipeId);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = prdcRecipeMaterialMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcRecipeMaterialList)){
            return Collections.emptyList();
        }else{
            List<PrdcRecipeMaterialVO> prdcRecipeMaterialVOList = BeanUtil.buildListFrom(prdcRecipeMaterialList, PrdcRecipeMaterialVO.class);
            for (PrdcRecipeMaterialVO prdcRecipeMaterialVO : prdcRecipeMaterialVOList) {
                PcsSkuVO skuVO = pcsSkuService.findByCode(prdcRecipeMaterialVO.getSkuCode());
                if(EmptyUtil.isNotEmpty(skuVO)) {
                	prdcRecipeMaterialVO.setSkuName(skuVO.getNameCn()==null?"":skuVO.getNameCn());
                }
            }
            return prdcRecipeMaterialVOList;
        }
    }

    @Override
    public List<PrdcRecipeVO> findRecipeVOByCond(PrdcRecipeCond cond, Boolean cascade) {
        List<PrdcRecipeVO> prdcRecipeList = prdcRecipeMapper.findRecipeVOByCond(cond);
        if(CollectionUtils.isEmpty(prdcRecipeList)){
            return Collections.emptyList();
        }else{
            List<PrdcRecipeVO> prdcRecipeVOList = prdcRecipeList;
            if(cascade){
            	List<Long> ids = new ArrayList<>();
                for(PrdcRecipeVO recipe : prdcRecipeVOList){
                	ids.add(recipe.getId());
                }
            	List<PrdcRecipeMaterialVO> recipeMaterialList = findRecipeMaterialVOByRecipeIds(ids);
                Map<Long,List<PrdcRecipeMaterialVO>> recipeMap = new HashMap<>();
                for(PrdcRecipeMaterialVO m : recipeMaterialList){
                	List<PrdcRecipeMaterialVO> tmp = recipeMap.get(m.getRecipeId());
                	if(NullUtil.isNull(tmp)){
                		tmp = new ArrayList<>();
                		recipeMap.put(m.getRecipeId(), tmp);
                	}
                	tmp.add(m);
                }
                for (PrdcRecipeVO prdcRecipeVO : prdcRecipeVOList) {
                	List<PrdcRecipeMaterialVO> mList = recipeMap.get(prdcRecipeVO.getId());
                	if(NullUtil.isNull(mList)){
                		mList = new ArrayList<>();
                	}
                	prdcRecipeVO.setRecipeMaterialVOList(mList);
                }
            }
            
            return prdcRecipeVOList;
        }
    }

    @Override
    @Transactional
    public Long create(PrdcRecipeVO vo) {;
        PrdcRecipe prdcRecipe = BeanUtil.buildFrom(vo, PrdcRecipe.class);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = BeanUtil.buildListFrom(vo.getRecipeMaterialVOList(), PrdcRecipeMaterial.class);
        createRecipe(prdcRecipe);
        saveRecipeMaterial(prdcRecipe,prdcRecipeMaterialList);
        return prdcRecipe.getId();
    }

    @Override
    @Transactional
    public Boolean update(PrdcRecipeVO vo) {
        PrdcRecipe prdcRecipe = BeanUtil.buildFrom(vo, PrdcRecipe.class);
        List<PrdcRecipeMaterial> prdcRecipeMaterialList = BeanUtil.buildListFrom(vo.getRecipeMaterialVOList(), PrdcRecipeMaterial.class);
        updateRecipe(prdcRecipe);
        saveRecipeMaterial(prdcRecipe,prdcRecipeMaterialList);
        return true;
    }

    @Override
    public Boolean deleteRecipeMaterial(Long recipeId, Long recipeMaterialId) {
        if(recipeId==null||recipeMaterialId==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
        }
        PrdcRecipeVO prdcRecipeVO = findRecipeVOById(recipeId);
        if(prdcRecipeVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"匹配不到配方");
        }
        Boolean found = false;
        for (PrdcRecipeMaterialVO prdcRecipeMaterialVO : prdcRecipeVO.getRecipeMaterialVOList()) {
            if(!prdcRecipeMaterialVO.getId().equals(recipeMaterialId)){
                continue;
            }
            found = true;
        }
        if(found){
           if(prdcRecipeMaterialMapper.deleteByPrimaryKey(recipeMaterialId)==0){
               throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_DELETE_DB,"删除配方行失败");
           }
        }else{
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"配方不包含该行数据");
        }
        return true;
    }

    /**
     * 校验配方
     * @param isCreat
     * @param rcd
     */
    private void validateRecipe(Boolean isCreat,PrdcRecipe rcd){
        PrdcRecipeCond con = new PrdcRecipeCond();
        con.setRecipeSkuCode(rcd.getSkuCode());
        List<PrdcRecipeVO> list = prdcRecipeMapper.findRecipeVOByCond(con);
        if(isCreat){
            if(rcd.getId()!=null){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
            }
            if(list != null && list.size() > 0){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"skuCode不能重复");
            }
        }else{
            if(rcd.getId()==null){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
            }
            if(list.size() > 1){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"skuCode不能重复");
            }else if(list.size() == 1){
                PrdcRecipeVO prdcRecipe = list.get(0);
                if(!prdcRecipe.getId().equals(rcd.getId())){
                    throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"skuCode不能重复");
                }
            }
        }
        PcsSkuVO skuVO = pcsSkuService.findByCode(rcd.getSkuCode());
        if(skuVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"skuCode匹配不到sku");
        }
    }

    /**
     * 创建配方
     * @param rcd
     */
    @Transactional
    private void createRecipe(PrdcRecipe rcd){
        validateRecipe(true,rcd);
        rcd.setCreateTime(DateUtil.getNow());
        if(prdcRecipeMapper.insert(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"数据库插入失败");
        }
    }

    /**
     * 更新配方
     * @param rcd
     */
    @Transactional
    private void updateRecipe(PrdcRecipe rcd){
        validateRecipe(false,rcd);
        if(prdcRecipeMapper.updateByPrimaryKeySelective(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
        }
    }

    /**
     * 保存配方行
     * @param prdcRecipe
     * @param rcdList
     */
    private void saveRecipeMaterial(PrdcRecipe prdcRecipe,List<PrdcRecipeMaterial> rcdList){
        for (PrdcRecipeMaterial rcd : rcdList) {
            if(rcd.getId()==null){
                createRecipeMaterial(prdcRecipe,rcd);
            }else{
                updateRecipeMaterial(prdcRecipe,rcd);
            }
        }
    }

    /**
     * 校验配方行
     * @param rcd
     */
    private void validateRecipeMaterial(PrdcRecipe prdcRecipe,PrdcRecipeMaterial rcd){
        PcsSkuVO skuVO = pcsSkuService.findByCode(rcd.getSkuCode());
        if(skuVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材skuCode匹配不到sku");
        }
        if(EmptyUtil.isEmpty(rcd.getQuantity())){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材sku数量不能为空");
        }
        if(rcd.getQuantity().compareTo(new Float(0))<=0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材sku数量不能小于等于0");
        }
        if(rcd.getSkuCode().equals(prdcRecipe.getSkuCode())){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材SKU包含成品SKU");
        }
    }

    /**
     * 创建配方行
     * @param prdcRecipe
     * @param rcd
     */
    @Transactional
    private void createRecipeMaterial(PrdcRecipe prdcRecipe,PrdcRecipeMaterial rcd){
        validateRecipeMaterial(prdcRecipe,rcd);
        rcd.setRecipeId(prdcRecipe.getId());
        if(prdcRecipeMaterialMapper.insert(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"数据库插入失败");
        }
    }

    /**
     * 更新配方行
     * @param prdcRecipe
     * @param rcd
     */
    @Transactional
    private void updateRecipeMaterial(PrdcRecipe prdcRecipe,PrdcRecipeMaterial rcd){
        validateRecipeMaterial(prdcRecipe,rcd);
        if(prdcRecipeMaterialMapper.updateByPrimaryKeySelective(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
        }
    }


}
