/**
 * Copyright (C), 上海布鲁爱电子商务有限公司
 */
package com.thebeastshop.pegasus.service.purchase.service.impl;

import com.thebeastshop.pegasus.service.purchase.cond.PrdcJobCond;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcJobLineMapper;
import com.thebeastshop.pegasus.service.purchase.dao.PrdcJobMapper;
import com.thebeastshop.pegasus.service.purchase.exception.PurchaseException;
import com.thebeastshop.pegasus.service.purchase.exception.PurchaseExceptionErrorCode;
import com.thebeastshop.pegasus.service.purchase.model.PrdcJob;
import com.thebeastshop.pegasus.service.purchase.model.PrdcJobExample;
import com.thebeastshop.pegasus.service.purchase.model.PrdcJobLine;
import com.thebeastshop.pegasus.service.purchase.model.PrdcJobLineExample;
import com.thebeastshop.pegasus.service.purchase.model.PrdcJobLineExample.Criteria;
import com.thebeastshop.pegasus.service.purchase.service.PcsSkuService;
import com.thebeastshop.pegasus.service.purchase.service.PrdcJobService;
import com.thebeastshop.pegasus.service.purchase.service.PrdcRecipeService;
import com.thebeastshop.pegasus.service.purchase.util.PrdcJobLineUtil;
import com.thebeastshop.pegasus.service.purchase.vo.*;
import com.thebeastshop.pegasus.service.warehouse.PegasusWarehouseInnerServiceFacade;
import com.thebeastshop.pegasus.service.warehouse.PegasusWarehouseServiceFacade;
import com.thebeastshop.pegasus.service.warehouse.PegasusWmsServiceFacade;
import com.thebeastshop.pegasus.service.warehouse.model.WhCommand;
import com.thebeastshop.pegasus.service.warehouse.model.WhCommandSku;
import com.thebeastshop.pegasus.service.warehouse.model.WhInvOccupy;
import com.thebeastshop.pegasus.service.warehouse.model.WhWarehouse;
import com.thebeastshop.pegasus.service.warehouse.service.WhInvService;
import com.thebeastshop.pegasus.service.warehouse.vo.WhInvVO;
import com.thebeastshop.pegasus.service.warehouse.vo.WhReleaseOccupationVO;
import com.thebeastshop.pegasus.service.warehouse.vo.WhWmsPrdcJobTaskVO;
import com.thebeastshop.pegasus.service.warehouse.vo.WhWmsSkuStockVO;
import com.thebeastshop.pegasus.util.comm.*;
import org.apache.commons.collections.CollectionUtils;
import org.aspectj.util.CollectionUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.util.*;

/**
 * @author Eric.Lou
 * @version $Id: PrdcJobServiceImpl.java, v 0.1 2016-01-04 上午11:18
 */
@Service("prdcJobService")
public class PrdcJobServiceImpl implements PrdcJobService {

    @Autowired
    private PcsSkuService                        pcsSkuService;

    @Autowired
    private PrdcRecipeService                    prdcRecipeService;

    @Autowired
    private PrdcJobMapper                        prdcJobMapper;

    @Autowired
    private PrdcJobLineMapper                    prdcJobLineMapper;

    private PegasusWarehouseInnerServiceFacade   facadeWi;

    private PegasusWarehouseServiceFacade        facadeWs;

    private PegasusWmsServiceFacade              facadeWms;


    @PostConstruct
    private void init() {
        facadeWi = PegasusWarehouseInnerServiceFacade.getInstance();
        facadeWs = PegasusWarehouseServiceFacade.getInstance();
        facadeWms = PegasusWmsServiceFacade.getInstance();
    }


    @Override
    public PrdcJobVO findJobVOById(Long id) {
        PrdcJobCond cond = new PrdcJobCond();
        cond.setJobId(id);
        cond.setCurrpage(1);
        List<PrdcJobVO> prdcJobVOList = findJobVOByCond(cond, true);
        if(CollectionUtils.isEmpty(prdcJobVOList)){
            return null;
        }else{
            return prdcJobVOList.get(0);
        }
    }

    @Override
    public PrdcJobVO findJobVOByCode(String code) {
        PrdcJobExample example = new PrdcJobExample();
        example.createCriteria().andCodeEqualTo(code);
        List<PrdcJob> jobs = prdcJobMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(jobs)) return null;
        List<PrdcJobVO> jobVOs = BeanUtil.buildListFrom(jobs,PrdcJobVO.class);
        PrdcJobVO jobVO = jobVOs.get(0);
        jobVO.setPrdcJobLineVOList(findJobLineVOByJobId(jobVO.getId()));
        return jobVO;
    }

    @Override
    public List<PrdcJobVO> findJobVOByRecipeId(Long recipeId, Boolean cascade) {
        PrdcJobCond cond = new PrdcJobCond();
        cond.setRecipeId(recipeId);
        cond.setCurrpage(null);
        List<PrdcJobVO> prdcJobVOList = findJobVOByCond(cond, cascade);
        if(CollectionUtils.isEmpty(prdcJobVOList)){
            return Collections.emptyList();
        }else{
            return prdcJobVOList;
        }
    }

    @Override
    public List<PrdcJobLineVO> findJobLineVOByJobId(Long jobId) {
        PrdcJobLineExample example = new PrdcJobLineExample();
        example.createCriteria().andJobIdEqualTo(jobId);
        List<PrdcJobLine> prdcJobLineList = prdcJobLineMapper.selectByExample(example);
        if(CollectionUtils.isEmpty(prdcJobLineList)){
            return Collections.emptyList();
        }else{
            List<PrdcJobLineVO> prdcJobLineVOList = BeanUtil.buildListFrom(prdcJobLineList, PrdcJobLineVO.class);
            for (PrdcJobLineVO prdcJobLineVO : prdcJobLineVOList) {
                PcsSkuVO skuVO = pcsSkuService.findByCode(prdcJobLineVO.getSkuCode());
                prdcJobLineVO.setSkuName(skuVO.getNameCn());
            }
            return prdcJobLineVOList;
        }
    }

    @Override
    public List<PrdcJobVO> findJobVOByCond(PrdcJobCond cond, Boolean cascade) {
        List<PrdcJob> prdcJobList = prdcJobMapper.findJobVOByCond(cond);
        if(CollectionUtils.isEmpty(prdcJobList)){
            return Collections.emptyList();
        }else{
            List<PrdcJobVO> prdcJobVOList = BeanUtil.buildListFrom(prdcJobList, PrdcJobVO.class);
            for (PrdcJobVO prdcJobVO : prdcJobVOList) {
                PcsSkuVO skuVO = pcsSkuService.findByCode(prdcJobVO.getSkuCode());
                prdcJobVO.setSkuName(skuVO == null?"":skuVO.getNameCn());
                if(cascade){
                    PrdcRecipeVO prdcRecipeVO = prdcRecipeService.findRecipeVOById(prdcJobVO.getRecipeId());
                    prdcJobVO.setRecipeName(prdcRecipeVO==null?"":prdcRecipeVO.getRecipeName());
                    prdcJobVO.setPrdcJobLineVOList(findJobLineVOByJobId(prdcJobVO.getId()));
                }
            }
            return prdcJobVOList;
        }
    }

    @Override
    @Transactional
    public Long create(PrdcJobVO vo) {
        vo.setProcessingAmount(0);
        vo.setFinishedDamagedQuantity(0);
        vo.setFinishedQuantity(0);
        PrdcJob prdcJob = BeanUtil.buildFrom(vo, PrdcJob.class);
        List<PrdcJobLine> prdcJobLineList = BeanUtil.buildListFrom(vo.getPrdcJobLineVOList(), PrdcJobLine.class);
        if(vo.getJobType() == PrdcJobVO.JOB_TYPE_ASSEMBLE){
            for(PrdcJobLine jobLine : prdcJobLineList){
                WhInvVO whInvVO = facadeWs.findCanUseQttBySkuCodeAndWarehouseCode(jobLine.getSkuCode(),vo.getWarehouseCode());
                if(jobLine.getQuantity()>whInvVO.getCanUseInv()) throw new PurchaseException("","["+jobLine.getSkuCode()+"]可用库存不足");
            }
        }
        else if(vo.getJobType() == PrdcJobVO.JOB_TYPE_SPLIT){
            WhInvVO whInvVO = facadeWs.findCanUseQttBySkuCodeAndWarehouseCode(vo.getSkuCode(),vo.getWarehouseCode());
            if(vo.getEstQuantity()>whInvVO.getCanUseInv()) throw new PurchaseException("","["+vo.getSkuCode()+"]可用库存不足");
        }
        createJob(prdcJob);
        saveJobLine(prdcJob,prdcJobLineList);
        //占用库存
        List<WhInvOccupy> whOccupyList = buildWhOccupy
                (prdcJob,prdcJobLineList, vo.getWarehouseCode(), WhInvOccupy.TYPE_PRODUCE_OUT);
        facadeWi.getInstance().occupy(whOccupyList);
        return prdcJob.getId();
    }

    @Override
    @Transactional
    public List<Long> batchCreate(List<PrdcJobVO> list) {
    	List<Long>  ids = new ArrayList<>();
    	if(EmptyUtil.isNotEmpty(list)) {
    		for(PrdcJobVO vo : list) {
    			ids.add(this.create(vo));
    		}
    	}
    	return ids;
    }
    @Override
    @Transactional
    public Boolean returnModify(PrdcJobVO vo) {
        PrdcJob prdcJob = BeanUtil.buildFrom(vo, PrdcJob.class);
        List<PrdcJobLine> prdcJobLineList_0 = BeanUtil.buildListFrom(findJobLineVOByJobId(vo.getId()), PrdcJobLine.class);
        // 创建释放
        List<WhReleaseOccupationVO> whReleaseList = buildWhRelease(prdcJob,prdcJobLineList_0, WhInvOccupy.TYPE_PRODUCE_OUT);
        List<PrdcJobLine> prdcJobLineList = BeanUtil.buildListFrom(vo.getPrdcJobLineVOList(), PrdcJobLine.class);
        returnModifyJob(prdcJob);
        changeJobLine(prdcJob, prdcJobLineList);
        //创建新占用
        List<WhInvOccupy> whOccupyList = buildWhOccupy
                (prdcJob,prdcJobLineList, vo.getWarehouseCode(), WhInvOccupy.TYPE_PRODUCE_OUT);
//        facadeWi.getInstance().occupy(whOccupyList);
        facadeWi.occupyAfterRelease(whOccupyList,whReleaseList);

        return true;
    }

    @Override
    public Boolean deleteJobLine(Long jobId, Long jobLineId) {
        if(jobId==null||jobLineId==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
        }
        PrdcJobVO prdcJobVO = findJobVOById(jobId);
        if(prdcJobVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"匹配不到任务");
        }
        Boolean found = false;
        for (PrdcJobLineVO prdcJobLineVO : prdcJobVO.getPrdcJobLineVOList()) {
            if(!prdcJobLineVO.getId().equals(jobLineId)){
                continue;
            }
            found = true;
        }
        if(found){
            if(prdcJobLineMapper.deleteByPrimaryKey(jobLineId)==0){
                throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_DELETE_DB,"删除任务行失败");
            }
        }else{
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"任务不包含该行数据");
        }
        return true;
    }

    @Override
    public Boolean cancelJobForStatusReturnModify(Long jobId) {
        if(jobId!=null && jobId>0){
            PrdcJobVO prdcJobVO = findJobVOById(jobId);
            if(prdcJobVO!=null){
                if(prdcJobVO.getPlanStatus().equals(PrdcJob.PLAN_STATUS_RETURN_MODIFY)){
                    PrdcJob update = new PrdcJob();
                    update.setId(jobId);
                    update.setPlanStatus(PrdcJob.PLAN_STATUS_CANCEL);
                    if(prdcJobMapper.updateByPrimaryKeySelective(update)==0){
                        throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
                    }
                }else{
                    throw new PurchaseException(PurchaseExceptionErrorCode.STATUS_NOT_EXPECTED,"the status is invalid ");
                }
            }else{
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"job is not found ");
            }
        }else{
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"the id is invalid");
        }
        return true;
    }

    @Override
    @Transactional
    public Boolean startProduction(PrdcJobVO jobVO) {
        if (jobVO == null)
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
        PrdcJob prdcJob = BeanUtil.buildFrom(jobVO,PrdcJob.class);
        int update = prdcJobMapper.updateByPrimaryKey(prdcJob);
        if (update<=0)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
//        facadeWi.getInstance().occupy(whInvOccupyList);
        return true;
    }

    @Override
    @Transactional
    public Boolean releaseJobLineOccupyAndUpdate(PrdcJobVO jobVO,List<WhReleaseOccupationVO> whReleaseOccupationVOList){
        if (jobVO == null)
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
        PrdcJob prdcJob = BeanUtil.buildFrom(jobVO,PrdcJob.class);
        int update = prdcJobMapper.updateByPrimaryKey(prdcJob);
        if (update<=0)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
        facadeWi.releaseOccupation(whReleaseOccupationVOList);
        return true;
    }

    @Override
    @Transactional
    public Boolean createCommandsAfterReleaseThenFinishAndUpdate(PrdcJobVO jobVO,List<WhCommand> whCommands,
                                                                 List<WhReleaseOccupationVO> whReleaseOccupationVOList) throws Exception{
        if (jobVO == null)
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"param abnormal!");
        if (jobVO.getPlanStatus() == PrdcJob.PLAN_STATUS_FINISHED || jobVO.getPlanStatus() == PrdcJob.WMS_PLAN_STATUS_FINISHED || jobVO.getPlanStatus() == PrdcJob.PLAN_STATUS_CANCEL)
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"status abnormal!");
        PrdcJob prdcJob = BeanUtil.buildFrom(jobVO,PrdcJob.class);
        prdcJob.setPlanStatus(PrdcJob.WMS_PLAN_STATUS_FINISHED);
        prdcJob.setFinishTime(new Date());
        int updateJob = prdcJobMapper.updateByPrimaryKey(prdcJob);
        if (updateJob<=0)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"update data failed!");
        for(PrdcJobLine jobLine : jobVO.getPrdcJobLineVOList()){
            int updateJobLine = prdcJobLineMapper.updateByPrimaryKey(jobLine);
            if (updateJobLine<=0)
                throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"update data failed!");
        }
        //释放后创建指令并完成后更新job
        List<String> commands =  facadeWi.createCommandsAfterReleaseThenFinish(whCommands, whReleaseOccupationVOList);
        if(EmptyUtil.isEmpty(commands))
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"param abnormal!");
        return true;
    }



    @Override
    @Transactional
    public Boolean updateJobVO(PrdcJobVO jobVO) {
        if (jobVO == null)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"param abnormal!");
        PrdcJob prdcJob = BeanUtil.buildFrom(jobVO,PrdcJob.class);
        int update = prdcJobMapper.updateByPrimaryKey(prdcJob);
        if (update<=0)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"update data failed!");
        return true;
    }

    /**
     * 校验任务
     * @param isCreat
     * @param rcd
     */
    private void validateJob(Boolean isCreat,PrdcJob rcd){
        if(isCreat){
            if(rcd.getId()!=null){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"param abnormal");
            }
        }else{
            if(rcd.getId()==null){
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"param abnormal");
            }
        }

        PrdcRecipeVO prdcRecipeVO = prdcRecipeService.findRecipeVOById(rcd.getRecipeId());
        if(prdcRecipeVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"recipeId匹配不到配方");
        }
        if(!prdcRecipeVO.getSkuCode().equals(rcd.getSkuCode())){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"配方成品skuCode与任务成品skuCode不一致");
        }
        PcsSkuVO skuVO = pcsSkuService.findByCode(rcd.getSkuCode());
        if(skuVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"skuCode匹配不到sku");
        }
    }

    /**
     * 创建任务
     * @param rcd
     */
    @Transactional
    private void createJob(PrdcJob rcd){
        validateJob(true,rcd);
        rcd.setCode("");
        rcd.setPlanStatus(PrdcJob.WMS_STATUS_RETURN_MODIFY);
        rcd.setCreateTime(DateUtil.getNow());
        if(prdcJobMapper.insert(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"insert data failed!");
        }
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("id", rcd.getId());
        String code = CodeGenerator.getInstance().generate("PRDC_JOB_CODE", params);
        if(EmptyUtil.isEmpty(code)){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_GENERATE_CODE,"任务编号生成失败");
        }
        rcd.setCode(code);
        if(prdcJobMapper.updateByPrimaryKey(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"update data failed!");
        }
    }

    /**
     * 退回修改任务
     * @param rcd
     */
    @Transactional
    private void returnModifyJob(PrdcJob rcd){
        if(rcd.getPlanStatus()!=PrdcJob.WMS_STATUS_RETURN_MODIFY)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"status abnormal!");
        validateJob(false,rcd);
        PrdcJob update = new PrdcJob();
        update.setRecipeId(rcd.getRecipeId());
        update.setId(rcd.getId());
        update.setSkuCode(rcd.getSkuCode());
        update.setWarehouseCode(rcd.getWarehouseCode());
        update.setEstQuantity(rcd.getEstQuantity());
        update.setEstDoneDate(rcd.getEstDoneDate());
        update.setRemark(rcd.getRemark());
        update.setJobType(rcd.getJobType());
        if(prdcJobMapper.updateByPrimaryKeySelective(update)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"update data failed!");
        }
    }

    /**
     * 保存任务行
     * @param prdcJob
     * @param rcdList
     */
    private void saveJobLine(PrdcJob prdcJob,List<PrdcJobLine> rcdList){
        List<Integer> nums = new ArrayList<Integer>();

        //算最大编号
        nums.add(0);
        for (PrdcJobLine line : rcdList) {
            if(line.getCode()!=null && !line.getCode().equals("")){
                String suffixCode = line.getCode().substring(line.getCode().indexOf(prdcJob.getCode()) + prdcJob.getCode().length());
                nums.add(Integer.valueOf(suffixCode));
            }
        }
        Collections.sort(nums);
        Integer maxNum = nums.get(0);

        for(int i=0 ; i<rcdList.size() ; i++){
            PrdcJobLine rcd = rcdList.get(i);
            if(rcd.getId()==null){
                rcd.setCode(PrdcJobLineUtil.getJobLineCode(prdcJob.getCode(),++maxNum));
                createJobLine(prdcJob,rcd);
            }
//            else{
//                updateJobLine(prdcJob,rcd,maxNum);
//            }
        }

    }


    /**
     * 修改任务行
     * @param prdcJob
     * @param rcdList
     */
    @Transactional
    private void changeJobLine(PrdcJob prdcJob,List<PrdcJobLine> rcdList){
        List<Integer> nums = new ArrayList<Integer>();
        PrdcJobLineExample example = new PrdcJobLineExample();
        example.createCriteria().andJobIdEqualTo(prdcJob.getId());
        prdcJobLineMapper.deleteByExample(example);
        //算最大编号
        nums.add(0);
        for (PrdcJobLine line : rcdList) {
            if(line.getCode()!=null && !line.getCode().equals("")){
                String suffixCode = line.getCode().substring(line.getCode().indexOf(prdcJob.getCode()) + prdcJob.getCode().length());
                nums.add(Integer.valueOf(suffixCode));
            }
        }
        Collections.sort(nums);
        Integer maxNum = nums.get(0);
        for(int i=0 ; i<rcdList.size() ; i++){
            PrdcJobLine rcd = rcdList.get(i);
            rcd.setCode(PrdcJobLineUtil.getJobLineCode(prdcJob.getCode(),++maxNum));
            createJobLine(prdcJob,rcd);
        }

    }

    /**
     * 校验任务行
     * @param rcd
     */
    private void validateJobLine(PrdcJob prdcJob,PrdcJobLine rcd){
        PcsSkuVO skuVO = pcsSkuService.findByCode(rcd.getSkuCode());
        if(skuVO==null){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材skuCode匹配不到sku");
        }
        if(rcd.getQuantity().compareTo(new Float(0))<=0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材sku数量不能小于等于0");
        }
        if(rcd.getSkuCode().equals(prdcJob.getSkuCode())){
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"耗材SKU包含成品SKU");
        }
    }

    /**
     * 创建任务行
     * @param prdcJob
     * @param rcd
     */
    @Transactional
    private void createJobLine(PrdcJob prdcJob,PrdcJobLine rcd){
        validateJobLine(prdcJob,rcd);
        rcd.setJobId(prdcJob.getId());
        if(prdcJobLineMapper.insert(rcd)==0){
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_INSERT_DB,"insert data failed!");
        }
    }

//    /**
//     * 更新任务行
//     * @param prdcJob
//     * @param rcd
//     */
//    @Transactional
//    private void updateJobLine(PrdcJob prdcJob,PrdcJobLine rcd,Integer maxNum){
//        validateJobLine(prdcJob,rcd);
//        rcd.setCode(PrdcJobLineUtil.getJobLineCode(prdcJob.getCode(),++maxNum));
//        createJobLine(prdcJob,rcd);
//    }

    @Override
    public List<WhCommand> buildPrdcCommands(PrdcJobVO jobVO) {
        if (CollectionUtils.isEmpty(jobVO.getPrdcJobLineVOList())) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造释放库存失败,prdcJobLineVOList");
        }
        List<WhCommand> whCommandList = new ArrayList<>();
        WhCommand whCommandJob = new WhCommand();
        whCommandJob.setWarehouseCode(jobVO.getWarehouseCode());
        whCommandJob.setReferenceCode(jobVO.getCode());
        if(jobVO.getJobType() == PrdcJob.JOB_TYPE_ASSEMBLE){
            whCommandJob.setInOutType(WhCommand.TYPE_PRODUCE_IN);
        }
        else if(jobVO.getJobType() == PrdcJob.JOB_TYPE_SPLIT){
            whCommandJob.setInOutType(WhCommand.TYPE_PRODUCE_OUT);
        }
        List<WhCommandSku> jobSkuList = new ArrayList<>();
        WhCommandSku jobSku = new WhCommandSku();
        jobSku.setSkuCode(jobVO.getSkuCode());
        jobSku.setPlanedQuantity(jobVO.getEstQuantity());
        jobSku.setQuantity(jobVO.getFinishedQuantity());
        jobSku.setDamagedQuantity(jobVO.getFinishedDamagedQuantity());
        jobSkuList.add(jobSku);
        whCommandJob.setWhCommandSkuList(jobSkuList);
        whCommandList.add(whCommandJob);
        for (PrdcJobLineVO jobLineVO : jobVO.getPrdcJobLineVOList()) {
            WhCommand whCommandJobLine = new WhCommand();
            List<WhCommandSku> jobLineSkuList = new ArrayList<>();
            whCommandJobLine.setWarehouseCode(jobVO.getWarehouseCode());
            whCommandJobLine.setReferenceCode(jobLineVO.getCode());
            //组装
            if(jobVO.getJobType() == PrdcJob.JOB_TYPE_ASSEMBLE) {
                whCommandJobLine.setInOutType(WhCommand.TYPE_PRODUCE_OUT);
            }
            //拆分,辅耗材置损耗
            else if(jobVO.getJobType() == PrdcJob.JOB_TYPE_SPLIT) {
                if (jobLineVO.getMaterialType() == PrdcJobLineVO.MATERIAL_TYPE_AUXILIARY) continue;
                whCommandJobLine.setInOutType(WhCommand.TYPE_PRODUCE_IN);
            }
            //数量小于等于0,报错
            if (jobLineVO.getQuantity().compareTo(new Float(0)) <= 0) {
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造仓库指令失败,数量必须大于0");
            }
            WhCommandSku jobLineSku = new WhCommandSku();
            jobLineSku.setSkuCode(jobLineVO.getSkuCode());
            jobLineSku.setPlanedQuantity(jobLineVO.getQuantity().intValue());
            jobLineSku.setQuantity(jobLineVO.getRealQuantity().intValue());
            jobLineSku.setDamagedQuantity(jobLineVO.getDamageQuantity().intValue());
            jobLineSkuList.add(jobLineSku);
            whCommandJobLine.setWhCommandSkuList(jobLineSkuList);
            whCommandList.add(whCommandJobLine);
        }

        return whCommandList;
    }

    @Override
    public List<WhReleaseOccupationVO> buildWhRelease(PrdcJob prdcJob,List<PrdcJobLine> prdcJobLineList, Integer occupyType) {
        List<WhReleaseOccupationVO> releaseList = new ArrayList();
        if (CollectionUtils.isEmpty(prdcJobLineList)) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造释放库存的LIST失败,opSoPackageSkuList不能为空");
        }

        PrdcJob prdcJob_0 = findJobVOById(prdcJob.getId());
        if(prdcJob_0.getJobType()== PrdcJob.JOB_TYPE_ASSEMBLE) {
            for (PrdcJobLine rdcJobLine : prdcJobLineList) {
                //数量小于等于0,报错
                if (rdcJobLine.getQuantity().compareTo(new Float(0)) <= 0) {
                    throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造释放库存的LIST失败,数量必须大于0");
                }
                WhReleaseOccupationVO whReleaseOccupationVO = new WhReleaseOccupationVO();
                whReleaseOccupationVO.setOccupyType(occupyType);
                whReleaseOccupationVO.setReferenceCode(rdcJobLine.getCode());
                releaseList.add(whReleaseOccupationVO);
            }
        }
        else if(prdcJob_0.getJobType()== PrdcJob.JOB_TYPE_SPLIT) {
            WhReleaseOccupationVO whReleaseOccupationVO = new WhReleaseOccupationVO();
            whReleaseOccupationVO.setOccupyType(occupyType);
            whReleaseOccupationVO.setReferenceCode(prdcJob_0.getCode());
            releaseList.add(whReleaseOccupationVO);
        }
        else{
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "error,type abnormal!");
        }

        return releaseList;
    }

    @Override
    public List<WhInvOccupy> buildWhOccupy(PrdcJob prdcJob,List<PrdcJobLine> prdcJobLineList, String warehouseCode, Integer occupyType) {
        List<WhInvOccupy> occupyList = new ArrayList<>();

        if (CollectionUtils.isEmpty(prdcJobLineList)) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "param list cannot be null");
        }
        if(prdcJob.getJobType()== PrdcJob.JOB_TYPE_ASSEMBLE){
            for (PrdcJobLine prdcJobLine : prdcJobLineList) {
                //数量小于等于0,报错
                if (NumberUtil.isNullOrZero(prdcJobLine.getQuantity())) {
                    throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "the quantity cannot be 0 or null");
                }
                WhInvOccupy whInvOccupy = new WhInvOccupy();
                whInvOccupy.setWarehouseCode(warehouseCode);
                whInvOccupy.setOccupyType(occupyType);
                whInvOccupy.setReferenceCode(prdcJobLine.getCode());
                whInvOccupy.setSkuCode(prdcJobLine.getSkuCode());
                whInvOccupy.setQuantity(prdcJobLine.getQuantity().intValue());
                occupyList.add(whInvOccupy);
            }
        }
        else if(prdcJob.getJobType()== PrdcJob.JOB_TYPE_SPLIT){
            //数量小于等于0,报错
            if (NumberUtil.isNullOrZero(prdcJob.getEstQuantity())) {
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "the quantity cannot be 0 or null!");
            }
            WhInvOccupy whInvOccupy = new WhInvOccupy();
            whInvOccupy.setWarehouseCode(warehouseCode);
            whInvOccupy.setOccupyType(occupyType);
            whInvOccupy.setReferenceCode(prdcJob.getCode());
            whInvOccupy.setSkuCode(prdcJob.getSkuCode());
            whInvOccupy.setQuantity(prdcJob.getEstQuantity());
            occupyList.add(whInvOccupy);
        }
        else{
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "error,status abnormal!");
        }

        return occupyList;
    }



//    @Override
//    public List<WhInvOccupy> buildWhOccupy(List<PrdcJob> prdcJobList, String warehouseCode, Integer occupyType) {
//        List<WhInvOccupy> occupyList = new ArrayList<>();
//        if (CollectionUtils.isEmpty(prdcJobLineList)) {
//            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造占用库存的LIST失败,opSoPackageSkuList不能为空");
//        }
//        for (PrdcJobLine prdcJobLine : prdcJobLineList) {
//            //数量小于等于0,报错
//            if (prdcJobLine.getQuantity().compareTo(new Float(0)) <= 0) {
//                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造占用库存的LIST失败,数量必须大于0");
//            }
//            WhInvOccupy whInvOccupy = new WhInvOccupy();
//            whInvOccupy.setWarehouseCode(warehouseCode);
//            whInvOccupy.setOccupyType(occupyType);
//            whInvOccupy.setReferenceCode(prdcJobLine.getCode());
//            whInvOccupy.setSkuCode(prdcJobLine.getSkuCode());
//            whInvOccupy.setQuantity(prdcJobLine.getQuantity().intValue());
//            occupyList.add(whInvOccupy);
//        }
//        return occupyList;
//    }

    @Override
    public List<Map<String, String>> getWarehouseList(List<String> warehouseCodeList) {
        List<Map<String, String>> warehouseNameList = new ArrayList<>();
        for (String warehouseCode : warehouseCodeList) {
            Map<String, String> option;
            option = new HashMap();
            option.put("code", warehouseCode);
            option.put("name", getWarehouseName(warehouseCode));
            warehouseNameList.add(option);
        }
        return warehouseNameList;
    }

    @Override
    public String getWarehouseName(String warehouseCode) {
        String name = "";
        if (NullUtil.isNotNull(warehouseCode)) {
            WhWarehouse wh = facadeWs.findWarehouseByCode(warehouseCode);
            if (NullUtil.isNotNull(wh) && EmptyUtil.isNotEmpty(wh.getName())) {
                name = wh.getName();
            }
        }
        return name;
    }

    @Override
    public int getPrdcJobMaxTask(WhWmsPrdcJobTaskVO vo) {
        return facadeWms.getPrdcJobMaxTask(vo);
    }



    @Override
    public boolean updatePrdcJobWhenTaskFinish(WhWmsPrdcJobTaskVO vo) {
        List<String> prdcJobCodes = new ArrayList<String>();
        if(vo!=null){
            prdcJobCodes.add(vo.getPrdcJobCode());
        }
        List<PrdcJob> prdcJobs = prdcJobMapper.getPrdcJobProcessingOrPartFinished(prdcJobCodes);
        if(prdcJobs!=null && !prdcJobs.isEmpty()){
            prdcJobCodes.clear();
            for(PrdcJob job : prdcJobs){
                prdcJobCodes.add(job.getCode());
            }
            List<Map<String,Object>> wmsPrdcJobTaskInfos = facadeWms.getWmsPrdcJobTaskFinishedInfo(prdcJobCodes);
            List<Map<String,Object>> wmsPrdcJobTaskProcesInfos = facadeWms.getWmsPrdcJobTaskProcessingInfo(prdcJobCodes);
            List<Map<String,Object>> wmsPrdcJobTaskFinishedRealAmount = facadeWms.getWmsPrdcJobTaskFinishedRealAmount(prdcJobCodes);
            if(wmsPrdcJobTaskInfos!=null && !wmsPrdcJobTaskInfos.isEmpty()){
                for(PrdcJob job : prdcJobs){
                    Map<String,Object> info = getTaskInfoByPrdcJobCode(wmsPrdcJobTaskInfos,job.getCode());
                    Map<String,Object> processingInfo = getTaskInfoByPrdcJobCode(wmsPrdcJobTaskProcesInfos,job.getCode());
                    List<Map<String,Object>> jobLineRealAmounts = getJobInfoByPrdcJobCode(wmsPrdcJobTaskFinishedRealAmount,job.getCode());
                    {

                        if(info!=null && processingInfo==null){//所有子任务都已完成
                            PrdcJob update = new PrdcJob();
                            update.setId(job.getId());
                            if(PrdcJob.JOB_TYPE_ASSEMBLE.equals(job.getJobType())){//组装
                                Integer finishedGoodsAmount = object2Int(info.get("finishedGoodsAmount"));
                                Integer finishedDefectiveAmount = object2Int(info.get("finishedDefectiveAmount"));
                                Integer amount = object2Int(info.get("amount"));
                                if(job.getEstQuantity().compareTo(amount)==0){//所有产品都已组装
                                    update.setPlanStatus(PrdcJob.WMS_PLAN_STATUS_FINISHED);
                                    update.setFinishTime(DateUtil.getNow());
                                }else if(job.getEstQuantity().compareTo(amount)>0 && amount > 0 ){//部分产品组装
                                    update.setPlanStatus(PrdcJob.WMS_PLAN_STATUS_PARTFINISHED);
                                }else{
                                	update.setPlanStatus(PrdcJob.WMS_STATUS_RETURN_MODIFY);
                                }
                                update.setFinishedQuantity(finishedGoodsAmount);
                                update.setFinishedDamagedQuantity(finishedDefectiveAmount);
                                update.setProcessingAmount(0);

                            }else{//拆分
                                //Integer finishedGoodsAmount = object2Int(info.get("finishedGoodsAmount"));
                                Integer amount = object2Int(info.get("amount"));
                                if(job.getEstQuantity().compareTo(amount)==0){//所有产品都已拆分
                                    update.setPlanStatus(PrdcJob.WMS_PLAN_STATUS_FINISHED);
                                    update.setFinishTime(DateUtil.getNow());
                                }else if(job.getEstQuantity().compareTo(amount)>0 && amount > 0){//部分产品拆分
                                    update.setPlanStatus(PrdcJob.WMS_PLAN_STATUS_PARTFINISHED);
                                }else{
                                	update.setPlanStatus(PrdcJob.WMS_STATUS_RETURN_MODIFY);
                                }
                                update.setFinishedQuantity(amount);
                                update.setProcessingAmount(0);
                            }
                            updatePrdcJobCheckCancle(update);
                            // 更新prdcJobLine realQuantity
                            for (Map<String,Object> realInfo : jobLineRealAmounts){
                            	PrdcJobLine prdcJobLine = makePrdcJobLine(realInfo,update);
                            	if (prdcJobLine != null){
                            		prdcJobLineMapper.updateByExampleSelective(prdcJobLine, makePrdcJobLineExample(prdcJobLine));
                            	}
                            }
                        }else if(info !=null && processingInfo!=null){//存在未完成的任务
                            PrdcJob update = new PrdcJob();
                            update.setId(job.getId());
                            update.setPlanStatus(PrdcJob.WMS_PLAN_STATUS_PROCESSING);
                            if(PrdcJob.JOB_TYPE_ASSEMBLE.equals(job.getJobType())){//组装
                                Integer finishedGoodsAmount = object2Int(info.get("finishedGoodsAmount"));
                                Integer finishedDefectiveAmount = object2Int(info.get("finishedDefectiveAmount"));
                                update.setFinishedQuantity(finishedGoodsAmount);
                                update.setFinishedDamagedQuantity(finishedDefectiveAmount);
                            }else{//拆分
                                Integer finishedGoodsAmount = object2Int(info.get("finishedGoodsAmount"));
                                update.setFinishedQuantity(finishedGoodsAmount);
                            }
                            Integer processingAmount = object2Int(processingInfo.get("amount"));
                            update.setProcessingAmount(processingAmount);
                            updatePrdcJobCheckCancle(update);
                        }else{
                            //throw new PurchaseException(PurchaseExceptionErrorCode.STATUS_NOT_EXPECTED, "error,status abnormal!");
                        }


                    }

                }
            }
        }


        return true;
    }
    /**
     * 更新未取消加工单信息,若已取消则不更新状态,其它字段更新
     * */
    private void updatePrdcJobCheckCancle(PrdcJob update){
    	PrdcJobExample example = new PrdcJobExample();
        example.createCriteria().andIdEqualTo(update.getId()).andPlanStatusNotEqualTo(PrdcJob.PLAN_STATUS_CANCEL);
        int size = prdcJobMapper.updateByExampleSelective(update, example);
        if(size == 0){
        	update.setPlanStatus(null);
        	prdcJobMapper.updateByPrimaryKeySelective(update);
        }
    }

    private PrdcJobLineExample makePrdcJobLineExample(PrdcJobLine prdcJobLine) {
    	if (prdcJobLine == null){
    		return null;
    	}
    	PrdcJobLineExample exam = new PrdcJobLineExample();
    	Criteria crit = exam.createCriteria();
    	crit.andSkuCodeEqualTo(prdcJobLine.getSkuCode());
    	crit.andJobIdEqualTo(prdcJobLine.getJobId());
		return exam;
	}


	private PrdcJobLine makePrdcJobLine(Map<String, Object> jobLineRealAmount, PrdcJob update) {
		if (jobLineRealAmount == null){
			return null;
		}
		PrdcJobLine prdcJobLine = new PrdcJobLine();
		prdcJobLine.setJobId(update.getId());
		prdcJobLine.setSkuCode(String.valueOf(jobLineRealAmount.get("skuCode")));
		prdcJobLine.setRealQuantity(Float.valueOf(String.valueOf(jobLineRealAmount.get("realAmount"))));
		return prdcJobLine;
	}


	private Integer object2Int(Object obj){
        if(obj instanceof Integer){
            return (Integer)obj;
        }
        return Integer.parseInt(obj.toString());
    }

    private Map<String,Object> getTaskInfoByPrdcJobCode(List<Map<String,Object>> infos , String prdcJobCode){
        if(infos!=null && !infos.isEmpty()){
            for(Map<String,Object> info : infos){
                Object code = info.get("prdcJobCode");
                if(prdcJobCode.equals(code)){
                    return info;
                }
            }
        }
        return null;
    }
    
    private List<Map<String,Object>> getJobInfoByPrdcJobCode(List<Map<String,Object>> jobFinishedRealAmount , String prdcJobCode){
        List<Map<String,Object>> map = new ArrayList<Map<String,Object>>();
    	if(jobFinishedRealAmount!=null && !jobFinishedRealAmount.isEmpty()){
            for(Map<String,Object> info : jobFinishedRealAmount){
                Object code = info.get("prdcJobCode");
                if(prdcJobCode.equals(code)){
                	map.add(info);
                }
            }
            return map;
        }
        return null;
    }


    @Override
    public List<Map<String, Object>> getWmsPrdcJobTaskFinishedInfo(
            List<String> prdcJobCodes) {
        return facadeWms.getWmsPrdcJobTaskFinishedInfo(prdcJobCodes);
    }


    @Override
    public List<PrdcJob> getPrdcJobProcessingOrPartFinished(List<String> codes) {
        return prdcJobMapper.getPrdcJobProcessingOrPartFinished(codes);
    }
    @Override
    public List<WhReleaseOccupationVO> buildWhRelease(List<PrdcJobLineVO> prdcJobLineVOList, Integer occupyType) {
        List<WhReleaseOccupationVO> releaseList = new ArrayList();
        if (CollectionUtils.isEmpty(prdcJobLineVOList)) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造释放库存的LIST失败,opSoPackageSkuList不能为空");
        }
        for (PrdcJobLineVO rdcJobLineVO : prdcJobLineVOList) {
            //数量小于等于0,报错
            if (rdcJobLineVO.getQuantity().compareTo((float) 0) <= 0) {
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造释放库存的LIST失败,数量必须大于0");
            }
            WhReleaseOccupationVO whReleaseOccupationVO = new WhReleaseOccupationVO();
            whReleaseOccupationVO.setOccupyType(occupyType);
            whReleaseOccupationVO.setReferenceCode(rdcJobLineVO.getCode());
            releaseList.add(whReleaseOccupationVO);
        }
        return releaseList;
    }

    @Override
    public List<WhInvOccupy> buildWhOccupy(List<PrdcJobLineVO> prdcJobLineVOList, String warehouseCode, Integer occupyType) {
        List<WhInvOccupy> occupyList = new ArrayList<>();
        if (CollectionUtils.isEmpty(prdcJobLineVOList)) {
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造占用库存的LIST失败,opSoPackageSkuList不能为空");
        }
        for (PrdcJobLineVO prdcJobLineVO : prdcJobLineVOList) {
            //数量小于等于0,报错
            if (prdcJobLineVO.getQuantity().compareTo((float)0) <= 0) {
                throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM, "构造占用库存的LIST失败,数量必须大于0");
            }
            WhInvOccupy whInvOccupy = new WhInvOccupy();
            whInvOccupy.setWarehouseCode(warehouseCode);
            whInvOccupy.setOccupyType(occupyType);
            whInvOccupy.setReferenceCode(prdcJobLineVO.getCode());
            whInvOccupy.setSkuCode(prdcJobLineVO.getSkuCode());
            whInvOccupy.setQuantity(((Double)Math.ceil(prdcJobLineVO.getQuantity())).intValue());
            occupyList.add(whInvOccupy);
        }
        return occupyList;
    }
    @Override
    @Transactional
    public Boolean checkOccupyAndUpdate(PrdcJobVO jobVO,List<WhInvOccupy> whInvOccupyList) {
        if (jobVO == null)
            throw new PurchaseException(PurchaseExceptionErrorCode.ILLEGAL_PARAM,"参数异常");
        PrdcJob prdcJob = BeanUtil.buildFrom(jobVO,PrdcJob.class);
        int update = prdcJobMapper.updateByPrimaryKey(prdcJob);
        if (update<=0)
            throw new PurchaseException(PurchaseExceptionErrorCode.ERROR_UPDATE_DB,"数据库更新失败");
        facadeWi.getInstance().occupy(whInvOccupyList);
        return true;
    }


	@Override
	public List<PrdcJobVO> findJobVOByCond(PrdcJobCond cond) {
		return this.prdcJobMapper.findExportJobVOByCond(cond);
	}


	@Override
	public List<PrdcJob> findSkuReportJobVO() {
		return prdcJobMapper.findSkuReportJobVO();
	}
}
