package com.thebeastshop.pegasus.component.redenvelope.business.impl;

import java.util.Date;
import java.util.List;
import java.util.Random;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.dubbo.common.utils.StringUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.component.redenvelope.business.IRedEnvelopeShareService;
import com.thebeastshop.pegasus.component.redenvelope.condition.RedEnvelopeCondition;
import com.thebeastshop.pegasus.component.redenvelope.domain.RedEnvelopeDTO;
import com.thebeastshop.pegasus.component.redenvelope.domain.RedEnvelopeRecordDTO;
import com.thebeastshop.pegasus.component.redenvelope.model.RedEnvelope;
import com.thebeastshop.pegasus.component.redenvelope.model.RedEnvelopePrizeGroup;
import com.thebeastshop.pegasus.component.redenvelope.service.RedEnvelopePrizeGroupService;
import com.thebeastshop.pegasus.component.redenvelope.service.RedEnvelopeRecordService;
import com.thebeastshop.pegasus.component.redenvelope.service.RedEnvelopeService;
import com.thebeastshop.pegasus.component.redenvelope.util.RedEnveloperDivideUtil;
import com.thebeastshop.pegasus.component.support.Res;
import com.thebeastshop.pegasus.util.comm.NumberUtil;
import com.thebeastshop.support.enums.RedEnvelopeTemplateType;
import com.thebeastshop.support.enums.RedEnvelopeType;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.vo.redenvelope.ReceiveDetailsVO;
import com.thebeastshop.support.vo.redenvelope.ReceivedRedEnvelopeVo;
import com.thebeastshop.support.vo.redenvelope.RedEnvelopeConfig;
import com.thebeastshop.support.vo.redenvelope.RedEnvelopeRecordVo;
import com.thebeastshop.support.vo.redenvelope.RedPrize;

/**
 * Created by feilong.gao on 2016/12/20.
 */
@Service
public class IRedEnvelopeShareServiceImpl implements IRedEnvelopeShareService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    public RedEnvelopeService redEnvelopeService;
    @Autowired
    public RedEnvelopePrizeGroupService redEnvelopePrizeGroupService;
    @Autowired
    public RedEnvelopeRecordService redEnvelopeRecordService;

    @Override
    public Res<ReceiveDetailsVO> receiveRedEnvelope(MemberVO member, RedEnvelopeCondition condition) {
        logger.info("领取红包入参：member：{} 领取信息：{}", member, condition);
        Res<ReceiveDetailsVO> result = new Res<>();
        ReceiveDetailsVO details = new ReceiveDetailsVO();
        //1.分享活动是否有效
        RedEnvelopeDTO redEnvelopeDTO = redEnvelopeService.getRedEnveloper(Long.parseLong(condition.getShareId()));
        details.setAvailable(this.checkAavailable(redEnvelopeDTO));

        //2 .查询活动领取记录
        List<RedEnvelopeRecordDTO> list = Lists.newArrayList();
        if (redEnvelopeDTO != null) {
            list = redEnvelopeRecordService.queryRecordByActivityId(condition.getShareId());
            details.setRecords(convertVo4Dto(list));
        } else {
            logger.info("活动id[{}]的红包分享活动为空", condition.getShareId());
            result.setData(details);
            return result;
        }

        Res<RedEnvelopeRecordVo> myReceivedRecord = this.getMyReceivedRecord(list, condition.getUnionId(), condition.getMobile());
        if (null != myReceivedRecord) {
            details.setMyReceived(myReceivedRecord.getData());
            result.setCode(myReceivedRecord.getCode());
            result.setMessage(myReceivedRecord.getMessage());
            details.setMyReceived(myReceivedRecord.getData());
            logger.info("微信会员[{}]为手机号[{}]已经领取红包：{}", condition.getUnionId(), condition.getMobile(), JSON.toJSONString(result));
        } else if (details.isAvailable() && StringUtils.isNotEmpty(condition.getMobile())) {
            //3. 领取红包
            RedEnvelopePrizeGroup prizeGroup = this.getSharePrize(Long.parseLong(condition.getShareId()));

            //4.消减红包库存
            Long prizeId = null;
            if(null!=prizeGroup){
                prizeId = prizeGroup.getId();
            }
            boolean flag = this.updateStock(Long.parseLong(condition.getShareId()),prizeId);
            logger.info("修改红包活动[{}],库存结果：{}", condition.getShareId(), flag);
            if (flag) {
                // 5 添加红包领取记录
                RedEnvelopeRecordDTO saveRecord = new RedEnvelopeRecordDTO();
                saveRecord.setShareActivityId(condition.getShareId());
                if(null == prizeGroup){
                    saveRecord.setReType(redEnvelopeDTO.getReType());
                    saveRecord.setReName(redEnvelopeDTO.getReName());
                    saveRecord.setReValue(redEnvelopeDTO.getReValue());
                    saveRecord.setReDesc(redEnvelopeDTO.getReDesc());
                }else{
                    saveRecord.setReType(prizeGroup.getReType());
                    saveRecord.setReName(prizeGroup.getReName());
                    saveRecord.setReValue(prizeGroup.getReValue());
                    saveRecord.setReDesc(prizeGroup.getReDesc());
                }

                saveRecord.setWechatHeadImg(condition.getAvatar());
                saveRecord.setWechatNickName(condition.getNickName());
                saveRecord.setWechatUnionid(condition.getUnionId());
                saveRecord.setReceiveMemberId(member.getId().toString());
                saveRecord.setReceiveMemberCode(member.getCode());
                saveRecord.setReceiveMobile(condition.getMobile());
                saveRecord = redEnvelopeRecordService.saveRedEnveloperRecord(saveRecord);
                logger.info("添加红包领取记录：{}", JSON.toJSONString(saveRecord));
                RedEnvelopeRecordVo myRecord = convertVo4Dto(saveRecord);
                myRecord.setFirstReceive(true);
                details.setMyReceived(myRecord);
                logger.info("微信会员[{}]为手机号[{}]领取新红包 成功：{}", condition.getUnionId(), condition.getMobile(), JSON.toJSONString(myRecord));
            } else {
                details.setAvailable(false);
                logger.info("微信会员[{}]为手机号[{}]领取新红包 失败", condition.getUnionId(), condition.getMobile());
            }

        }
        result.setData(details);
        logger.info("领取结果：{}", JSON.toJSONString(result));
        return result;
    }


    private RedEnvelopePrizeGroup getSharePrize(Long shareId){
        RedEnvelopePrizeGroup result = null;
        List<RedEnvelopePrizeGroup> list = redEnvelopePrizeGroupService.getPrizeGroupByActiveId(shareId);
        List<RedEnvelopePrizeGroup> prizePool = Lists.newArrayList();
        if(CollectionUtils.isNotEmpty(list)){
            for(RedEnvelopePrizeGroup prize : list){
                if(prize.getReAmount()>prize.getReleasedCount()){
                    for(int i = 0;i<prize.getReAmount()-prize.getReleasedCount() ;i++){
                        prizePool.add(prize);
                    }
                }
            }
        }
        if(CollectionUtils.isNotEmpty(prizePool)){
            int max = prizePool.size();
            int idx =0;
            if(max>1){
                idx = new Random().nextInt(max);
            }
            result = prizePool.get(idx);
        }
        return result;
    }

    @Transactional
    private boolean updateStock(Long shareId,Long prizeId){
        boolean flag = redEnvelopeService.updateStock(shareId);
        logger.info("修改红包活动[{}],库存结果：{}", shareId, flag);
        if(flag && !NumberUtil.isNullOrZero(prizeId)){
            flag = redEnvelopePrizeGroupService.updateStock(prizeId);
            logger.info("修改红包活动奖品[{}],库存结果：{}", prizeId, flag);
        }
        return flag;
    }


    /**
     * 获取领取记录
     *
     * @param records 已领取红包记录
     * @param unionId 微信unionId
     * @param mobile  领取手机号
     * @return
     */
    private Res<RedEnvelopeRecordVo> getMyReceivedRecord(List<RedEnvelopeRecordDTO> records, String unionId, String mobile) {
        RedEnvelopeRecordVo myRecord = new RedEnvelopeRecordVo();
        Res<RedEnvelopeRecordVo> result = null;
        if (CollectionUtils.isNotEmpty(records)) {
            for (RedEnvelopeRecordDTO record : records) {
                if (record.getReceiveMobile().equalsIgnoreCase(mobile) && !record.getWechatUnionid().equalsIgnoreCase(unionId)) {
                    myRecord = convertVo4Dto(record);
                    result = new Res<>(201, "手机已领取，请更换手机号", myRecord);
                    break;
                }
                if (record.getWechatUnionid().equalsIgnoreCase(unionId)) {
                    myRecord = convertVo4Dto(record);
                    result = new Res<>();
                    result.setData(myRecord);
                    break;
                }
            }
        }
        return result;
    }


    /**
     * 校验是否有效
     *
     * @return
     */
    private boolean checkAavailable(RedEnvelopeDTO redEnvelopeDTO) {
        if (redEnvelopeDTO == null) {
            logger.info("红包分享活动不存在");
            return false;
        }
        if (redEnvelopeDTO.getReAmount() <= redEnvelopeDTO.getReleasedCount()) {
            logger.info("红包已经领取完");
            return false;
        }
        Date date = new Date();
        if (redEnvelopeDTO.getStartTime() != null) {
            if (redEnvelopeDTO.getStartTime().getTime() > date.getTime()) {
                logger.info("红包活动没有开始");
                return false;
            }
        }
        if (redEnvelopeDTO.getEndTime() != null) {
            if (redEnvelopeDTO.getEndTime().getTime() < date.getTime()) {
                logger.info("红包活动已经结束");
                return false;
            }
        }
        return true;
    }



    /**
     * 红包记录转换
     *
     * @param sourceObj
     * @return
     */
    private RedEnvelopeRecordVo convertVo4Dto(RedEnvelopeRecordDTO sourceObj) {
        RedEnvelopeRecordVo targetObj = null;
        if (null != sourceObj) {
            targetObj = new RedEnvelopeRecordVo();
            targetObj.setAvatar(sourceObj.getWechatHeadImg());
            targetObj.setNickName(sourceObj.getWechatNickName());
            targetObj.setUnionId(sourceObj.getWechatUnionid());
            targetObj.setReceiveTime(sourceObj.getCreateTime());
            targetObj.setMobile(sourceObj.getReceiveMobile());
            ReceivedRedEnvelopeVo recordVo = new ReceivedRedEnvelopeVo();
            recordVo.setType(RedEnvelopeType.get(sourceObj.getReType()).getValue());
            recordVo.setDesc(sourceObj.getReDesc());
            recordVo.setValue(sourceObj.getReValue());
            recordVo.setName(sourceObj.getReName());
            targetObj.setReceived(recordVo);
        }
        return targetObj;
    }

    /**
     * 红包记录转换
     *
     * @param sourceObjs
     * @return
     */
    private List<RedEnvelopeRecordVo> convertVo4Dto(List<RedEnvelopeRecordDTO> sourceObjs) {
        List<RedEnvelopeRecordVo> targetObjs = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(sourceObjs)) {
            for (RedEnvelopeRecordDTO sourceObj : sourceObjs) {
                targetObjs.add(this.convertVo4Dto(sourceObj));
            }
        }
        return targetObjs;
    }


    @Override
    @Transactional
    public Res<Long> createRedEnvelopeActivity(RedEnvelopeConfig config, MemberVO member, String bindId) {
        Res<Long> result = new Res<>();
        try{
            //1 校验配置是否合法
            this.checkRedEnvelopeConfig(config);
            RedEnvelopeDTO red = redEnvelopeService.getRedEnvelopeByOrder(bindId,member.getCode());
            if(null != red){
                result.setData(red.getId());
                return result;
            }
            //2 生成红包活动奖品池
            config = RedEnveloperDivideUtil.randomRedEnvelopeConfig(config);

            RedEnvelope redEnvelope = new RedEnvelope();
            redEnvelope.setType(config.getActivityType());   //活动类型
            redEnvelope.setBindId(bindId);    //绑定业务id
            redEnvelope.setReleasedCount(0);
            redEnvelope.setReAmount(config.getRedTotalNum());
            redEnvelope.setModifyTime(new Date());
            redEnvelope.setCreateTime(new Date());
            redEnvelope.setMemberCode(member.getCode());
            redEnvelope.setMemberId(member.getId().toString());
            redEnvelope.setTotalCutPrice(config.getTotalCutPrice());
            redEnvelope.setTemplateId(config.getTemplateId());
            redEnvelope.setTemplateType(config.getTemplateType());  //设置模板类型
            redEnvelope.setReType(config.getRedType());   //设置红包类型
            redEnvelope.setEndTime(config.getReEndTime());
            redEnvelope.setStartTime(config.getReStartTime());


            redEnvelopeService.newRedEnvelopeByOrder(redEnvelope);

            if(CollectionUtils.isNotEmpty(config.getRedGroup())){
                List<RedEnvelopePrizeGroup> prizes = Lists.newArrayList();
                for (RedPrize redPrize : config.getRedGroup()) {
                    RedEnvelopePrizeGroup prize = new RedEnvelopePrizeGroup();
                    prize.setModifyTime(new Date());
                    prize.setCreateTime(new Date());
                    prize.setReAmount(redPrize.getNum());
                    prize.setReCutPrice(redPrize.getCutPrice());
                    prize.setReValue(redPrize.getRedValue());
                    prize.setReDesc(redPrize.getRedDesc());
                    prize.setReName(redPrize.getRedName());
                    prize.setActivityId(redEnvelope.getId());
                    prize.setReType(config.getRedType());
                    prize.setReleasedCount(0);
                    prizes.add(prize);
                }
                redEnvelopePrizeGroupService.addRedEnvelopePrizeGroup(prizes);
            }else{
                logger.error("config:{} 奖品为空", JSON.toJSONString(config));
                throw new WrongArgException("奖品为空");
            }

            //3 返回分享活动对象
            result.setData(redEnvelope.getId());
        }catch (WrongArgException wae){
            result.setCode(400);
            result.setMessage(wae.getMessage());
            return result;
        }
        return result;
    }

    boolean checkRedEnvelopeConfig(RedEnvelopeConfig config) {
        if (null == config) {
            throw new WrongArgException("红包配置为空！！！");
        }
        if (null == RedEnvelopeTemplateType.get(config.getTemplateType())) {
            throw new WrongArgException("红包模板类型不合法");
        }
        if (NumberUtil.isNullOrZero(config.getRedTotalNum())) {
            throw new WrongArgException("分享红包总数不能为空");
        }
        if (RedEnvelopeTemplateType.RANDOM_TEMPLATE.getCode().equals(config.getTemplateType()) && NumberUtil.isNullOrZero(config.getTotalCutPrice())) {
            throw new WrongArgException("分享红包随机红包总值不能为空");
        }
        if(RedEnvelopeTemplateType.RANDOM_SEPARATE_TEMPLATE.getCode().equals(config.getTemplateType())){
            if (CollectionUtils.isEmpty(config.getRedPoolGroup())) {
                throw new WrongArgException("分享分档红包奖品池不能为空");
            }
        }else{
            if (CollectionUtils.isEmpty(config.getRedGroup())) {
                throw new WrongArgException("分享红包奖品不能为空");
            }
        }
        if (null == RedEnvelopeType.get(config.getRedType())) {
            throw new WrongArgException("分享红包奖品类型不能为空");
        }
        if (null == config.getReStartTime() || null == config.getReEndTime()) {
            throw new WrongArgException("分享红包活动的结束时间和开始时间不能为空");
        }
        return true;
    }


}
