package com.thebeastshop.pegasus.component.compatible.service.impl;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.component.compatible.old.CategoryItemVo;
import com.thebeastshop.pegasus.component.compatible.old.ItemDo;
import com.thebeastshop.pegasus.component.compatible.old.ListConstant;
import com.thebeastshop.pegasus.component.compatible.old.ResponseVO2;
import com.thebeastshop.pegasus.component.compatible.service.ListService;
import com.thebeastshop.pegasus.component.product.enums.ProductSortField;
import com.thebeastshop.pegasus.component.support.service.RedisService;
import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.vo.PsChnStockVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.util.comm.NullUtil;
import com.thebeastshop.support.AccessWay;
import com.thebeastshop.support.page.Page;
import com.thebeastshop.support.page.Pagination;
import com.thebeastshop.support.page.Sort;
import com.thebeastshop.support.util.HttpUtil;
import com.thebeastshop.support.util.JsonUtil;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class ListServiceImpl implements ListService {
	protected Logger logger = LoggerFactory.getLogger(getClass());

	@Autowired
	private McPsProductService mcPsProductService;

	@Autowired
	private RedisService redisService;

	public ListServiceImpl() {
		super();
	}

	// ===========private===========
	/**
	 * 调用官网获取list商品的方法
	 * 
	 * @param id
	 * @return
	 */
	private List<PsProductVO> _callOldFindById(Long id, String channelCode, MemberVO member, AccessWay accessWay) {
		List<PsProductVO> products = Lists.newArrayList();
		try {
			String url = ListConstant.FIND_LIST;
			Map<String, Object> params = Maps.newHashMap();
			params.put("currpage", 1);
			params.put("catid", id);
			params.put("pagenum", 1000);
			logger.info("接口名={}", url);
			logger.info("发送的参数 params={}", params);
			Date start = new Date();
			String resultStr = HttpUtil.doGet(url, params, ListConstant.CHART_SET);
			Date end = new Date();
			logger.info("请求官方服务器查询list商品的访问时间：{}毫秒", end.getTime() - start.getTime());
			logger.info("返回的参数 resultStr={}", resultStr);
			if (StringUtils.isNotBlank(resultStr)) {
				@SuppressWarnings("unchecked")
				ResponseVO2<Object> responseVO = JsonUtil.alibabaToObject(resultStr, ResponseVO2.class);
				if (null != responseVO && responseVO.isSuccess() && responseVO.getRetObj() != null
						&& StringUtils.isNotBlank(responseVO.getRetObj().toString())) {
					List<CategoryItemVo> list = JsonUtil.alibabaToList(responseVO.getRetObj().toString(),
							CategoryItemVo.class);
					if (CollectionUtils.isNotEmpty(list)) {
						final List<String> productCodes = Lists.newArrayList();
						for (CategoryItemVo vo : list) {
							ItemDo itemDo = vo.getItemDo();
							if (itemDo != null && StringUtils.isNotBlank(itemDo.getProductCode())) {
								productCodes.add(itemDo.getProductCode());
							}
						}
						MemberLevelEnum memberLevel = MemberLevelEnum.getEnumByCode(member.getMemberLevel());
						PsMatchCampaignCond cond = new PsMatchCampaignCond();
						cond.setMatchCampaign(true);
						cond.setAccessWay(accessWay.getId());
						cond.setMemberLevel(memberLevel.getCode());
						cond.setDiscount(memberLevel.getLevelAmountCondition());
						products = mcPsProductService.findTopicByCodes(channelCode, productCodes, cond, 1,
								productCodes.size());
						// ====TODO: 排序，为了和官网一样====
						Collections.sort(products, new Comparator<PsProductVO>() {
							@Override
							public int compare(PsProductVO o1, PsProductVO o2) {
								Integer i1 = getIndex(productCodes, o1);
								Integer i2 = getIndex(productCodes, o2);
								return i1.compareTo(i2);
							}

							private int getIndex(final List<String> productCodes, PsProductVO product) {
								if (CollectionUtils.isNotEmpty(productCodes)) {
									for (int i = 0; i < productCodes.size(); i++) {
										if (productCodes.get(i).equals(product.getCode())) {
											return i;
										}
									}
								}
								return -1;
							}
						});
						// ==============================
					}
				}
			}
		} catch (Exception e) {
			logger.error("获取官网获取list商品异常:{}", e);
		}
		return products;
	}

	/**
	 * 商品排序
	 * 
	 * @param products
	 * @param sortField
	 */
	private List<PsProductVO> sort(List<PsProductVO> products, ProductSortField sortField, final String sortOrder) {
		switch (sortField) {
		case RELEASE:
			if (sortOrder.equalsIgnoreCase("ASC")) {
				Collections.reverse(products);
			}
			break;
		case PRICE:
			Collections.sort(products, new Comparator<PsProductVO>() {
				@Override
				public int compare(PsProductVO o1, PsProductVO o2) {
					BigDecimal price1 = o1 != null && o1.getListPrice() != null ? o1.getListPrice() : BigDecimal.ZERO;
					BigDecimal price2 = o2 != null && o2.getListPrice() != null ? o2.getListPrice() : BigDecimal.ZERO;
					if (sortOrder.equalsIgnoreCase("ASC")) {
						return price1.compareTo(price2);
					} else {
						return price2.compareTo(price1);
					}
				}
			});
			break;
		default:
			break;
		}

		return products;
	}

	/**
	 * 根据库存排序，库存为0排在后边
	 * 
	 * @param products
	 * @param channel
	 * @return
	 */
	private List<PsProductVO> sortBySoldOut(List<PsProductVO> products, String channelCode) {
		List<PsProductVO> hasInventory = Lists.newArrayList();
		List<PsProductVO> noInventory = Lists.newArrayList();
		for (PsProductVO prod : products) {
			prod.setPicDetailMul(""); // json转换避免带逗号字符
			List<PsChnStockVO> psChnStockVOs = prod.getStocks();
			for (PsChnStockVO psChnStockVO : psChnStockVOs) {
				if (psChnStockVO.getChnCode().equalsIgnoreCase(channelCode)) {
					if (psChnStockVO.getStock() != null && psChnStockVO.getStock() > 0) {
						hasInventory.add(prod);
					} else {
						noInventory.add(prod);
					}
					break;
				}
			}

		}
		hasInventory.addAll(noInventory);
		return hasInventory;
	}

	@Override
	public Page<PsProductVO> findById(long id, int offset, int limit, ProductSortField sortField, String sortOrder,
			String channelCode, MemberVO member, AccessWay accessWay) {
		Page<PsProductVO> rt = new Page<PsProductVO>(
				new Pagination(offset, limit, new Sort(sortField.getName(), sortOrder)));
		List<PsProductVO> list_pro = this.getListProduct(id, channelCode, sortField, sortOrder, member, accessWay);
		logger.info("查询List数量" + list_pro.size());
		if (CollectionUtils.isNotEmpty(list_pro)) {
			int total = list_pro.size();
			rt.setTotal(total);
			// 分页设置
			int fromIndex = offset;
			if (fromIndex < total) {
				int toIndex = offset + limit;
				if (toIndex > total) {
					toIndex = total;
				}
				rt.setItems(list_pro.subList(fromIndex, toIndex));
			}
		} else {
			rt.setTotal(0);
		}
		return rt;
	}

	/**
	 * 获取列表页商品列表 soldout下沉，根据指定字段排序
	 * 
	 * @param id
	 * @param channel
	 * @param sortField
	 * @param sortOrder
	 * @return
	 */
	private List<PsProductVO> getListProduct(long id, String channelCode, ProductSortField sortField, String sortOrder,
			MemberVO member, AccessWay accessWay) {

		List<PsProductVO> products = Lists.newArrayList();
		// 取出list后再排序
		String prodJson = redisService
				.get(getListProductRidisKey(id, channelCode, sortField, sortOrder, member, accessWay));
		if (NullUtil.isNull(prodJson)) {
			List<PsProductVO> psProductVOs = _callOldFindById(id, channelCode, member, accessWay);
			filterInvalidProds(psProductVOs, channelCode);
			List<PsProductVO> sortPsProd = this.sort(psProductVOs, sortField, sortOrder);
			products = this.sortBySoldOut(sortPsProd, channelCode);
			if (!CollectionUtils.isEmpty(products)) {
				redisService.set(getListProductRidisKey(id, channelCode, sortField, sortOrder, member, accessWay),
						JsonUtil.toJson(products), 300);
				logger.info("加入缓存{},缓存时间{}",
						getListProductRidisKey(id, channelCode, sortField, sortOrder, member, accessWay), "300秒");
			} else {
				logger.info("加入缓存{} 查询数据为空不加入缓存",
						getListProductRidisKey(id, channelCode, sortField, sortOrder, member, accessWay));
			}
		} else {
			products = JsonUtil.alibabaToList(prodJson, PsProductVO.class);
			logger.info("取出缓存key={}", getListProductRidisKey(id, channelCode, sortField, sortOrder, member, accessWay));
		}
		return products;
	}

	/**
	 * 过滤掉无效商品（下架，不可见）
	 * 
	 * @param prods
	 * @param channelCode
	 */
	private void filterInvalidProds(List<PsProductVO> prods, String channelCode) {
		if (CollectionUtils.isNotEmpty(prods)) {
			Iterator<PsProductVO> prodIterators = prods.iterator();
			while (prodIterators.hasNext()) {
				PsProductVO product = prodIterators.next();
				// invisible
				if (CollectionUtils.isEmpty(product.getCanSeeSku(channelCode))) {
					prodIterators.remove();
					continue;
				}
				// offShelf
				List<String> onShelfChnCodes = product.getOnShelfChnCodes();
				if (CollectionUtils.isEmpty(onShelfChnCodes) || !onShelfChnCodes.contains(channelCode)
						|| CollectionUtils.isEmpty(product.getCanSaleSku(channelCode))) {
					prodIterators.remove();
					continue;
				}
			}
		}
	}

	/**
	 * 获取搜索List商品redis缓存key 格式：
	 * search_list_key_[id][channelCode][sortField][sortOrder][memberLevel][
	 * accessWay]
	 * 
	 * @param id
	 * @param channel
	 * @param sortField
	 * @param sortOrder
	 * @param member
	 * @param accessWay
	 * @return
	 */
	private String getListProductRidisKey(long id, String channelCode, ProductSortField sortField, String sortOrder,
			MemberVO member, AccessWay accessWay) {
		StringBuffer sb = new StringBuffer();
		sb.append("search_list_key_listId_").append(id).append(channelCode).append(sortField).append(sortOrder)
				.append(member.getMemberLevel()).append(accessWay);
		return sb.toString();
	}

}
