package com.thebeastshop.pegasus.component.category.dao.impl;

import java.util.Collection;
import java.util.List;

import com.thebeastshop.pegasus.component.category.CampaignCategory;
import com.thebeastshop.pegasus.component.category.support.DefaultCampaignCategory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.component.category.Category;
import com.thebeastshop.pegasus.component.category.dao.CategoryDao;
import com.thebeastshop.pegasus.component.category.dao.mapper.CategoryEntityMapper;
import com.thebeastshop.pegasus.component.category.model.CategoryEntity;
import com.thebeastshop.pegasus.component.category.model.CategoryEntityExample;
import com.thebeastshop.pegasus.component.category.support.OldModelAdaptorCategory;

/**
 * 
 * @author Paul-xiong
 *
 */
@Repository
public class CategoryDaoImpl implements CategoryDao {

	@Autowired
	private CategoryEntityMapper categoryMapper;

	private OldModelAdaptorCategory adapt(final CategoryEntity entity) {
		return new OldModelAdaptorCategory(entity);
	}

	@Override
	public Category create(Category one) {
		final CategoryEntity oldModel = new CategoryEntity();
		oldModel.setParentId(one.getId());
		oldModel.setInUse(OldModelAdaptorCategory.validToInUse(one));
		oldModel.setName(one.getName());
		oldModel.setParentId(one.getParentId());
		this.categoryMapper.insert(oldModel);
		return adapt(oldModel);
	}

	@Override
	public void modify(Category one) {
		final CategoryEntity exist = new CategoryEntity();
		exist.setId(one.getId());
		exist.setInUse(OldModelAdaptorCategory.validToInUse(one));
		exist.setName(one.getName());
		exist.setParentId(one.getParentId());
		this.categoryMapper.updateByPrimaryKey(exist);
	}

	@Override
	public void del(Category category) throws UnsupportedOperationException {
		this.categoryMapper.deleteByPrimaryKey(category.getId());
	}

	@Override
	public Category getById(Long id) {
		final CategoryEntity pcsSkuCategory = this.categoryMapper.selectByPrimaryKey(id);
		return adapt(pcsSkuCategory);
	}

	@Override
	public Collection<Category> list() {
		List<CategoryEntity> entities = this.categoryMapper.selectByExample(new CategoryEntityExample());
		List<Category> list = Lists.newArrayList();
		for (CategoryEntity entity : entities) {
			list.add(adapt(entity));
		}
		return list;
	}

	private static CampaignCategory entity2Domain(CategoryEntity entity) {
		DefaultCampaignCategory domain = new DefaultCampaignCategory();
		domain.setId(String.valueOf((null == entity.getId()) ? "" : entity.getId()));
		domain.setName(entity.getName());
		domain.setParent(String.valueOf((0 == entity.getIsLeaf()) ? "root" : entity.getParentId()));
		if(0 == entity.getIsLeaf()) {
			domain.setType("first");
		}else{
			domain.setType("second");
		}
		return domain;
	}

	/**
	 * 向上递归实现查找已勾选目录树（由子目录查找父目录）
	 *
	 * @param list	结果集合
	 *
	 * @param entity	待添加数据
	 */
	private void buildTreeNodeUp(List<CampaignCategory> list, CategoryEntity entity){
		list.add(entity2Domain(entity));
		if(1 == entity.getIsLeaf() && null != entity.getParentId()){
			Boolean isExist = false;
			Long parentId = entity.getParentId();
			if(null != list) {
				for (CampaignCategory item : list) {
					if(item.getId().equals(String.valueOf(parentId))) {
						isExist = true;
						break;
					}
				}
			}
			if(!isExist) {
				buildTreeNodeUp(list, categoryMapper.selectByPrimaryKey(entity.getParentId()));
			}
		}
	}

	/**
	 * 向下递归实现查找所有目录（由第一级目录查找所有子目录）
	 *
	 * @param list	结果集合
	 *
	 * @param entity	待遍历元素
	 */
	private void buildTreeNodeDown(List<CampaignCategory> list, CategoryEntity entity){
		list.add(entity2Domain(entity));
		if(1 != entity.getIsLeaf()){
			CategoryEntityExample example = new CategoryEntityExample();
			example.createCriteria().andParentIdEqualTo(entity.getId());
			for (CategoryEntity categoryEntity : categoryMapper.selectByExample(example)) {
				buildTreeNodeDown(list, categoryEntity);
			}
		}
	}

	@Override
	public List<CampaignCategory> getCategoryByCampaignId(Long campaignId, Boolean temp) {
		List<CategoryEntity> entities = categoryMapper.getCategoryByCampaignId(campaignId, (temp ? 1 : 0));
		List<CampaignCategory> list = Lists.newArrayList();
		for (CategoryEntity entity : entities) {
			buildTreeNodeUp(list, entity);
		}
		return list;
	}

	@Override
	public List<CampaignCategory> getAllCategories() {
		List<CategoryEntity> entities = categoryMapper.getAllLeafCategory();
		List<CampaignCategory> lists = Lists.newArrayList();
		for (CategoryEntity entity : entities) {
			buildTreeNodeUp(lists, entity);
		}
		return lists;
	}

	/**
	 * 递归构建查找最低级子节点(暂时不用了)
	 *
	 * @param result 查找结果
	 *
	 * @param entity 待判断实体
	 */
	private void buildLastTreeNode(List<Long> result, CategoryEntity entity) {
		if(null != entity) {
			if(entity.getIsLeaf() == 1){
				result.add(entity.getId());
			}
			else{
				CategoryEntityExample example = new CategoryEntityExample();
				example.createCriteria().andParentIdEqualTo(entity.getId());
				for (CategoryEntity categoryEntity : categoryMapper.selectByExample(example)) {
					buildLastTreeNode(result, categoryEntity);
				}
			}
		}
	}

	@Override
	public List<Long> getTreeNode(List<Long> categories) {
		if(null == categories || categories.isEmpty()) {
			return null;
		}
		List<CategoryEntity> entities = categoryMapper.getCategoryByIds(categories);
		List<Long> result = Lists.newArrayList();
		for (CategoryEntity entity : entities) {
			if(null != entity && 1 == entity.getIsLeaf()) {
				result.add(entity.getId());
			}
			//buildLastTreeNode(result, entity);
		}
		return result;
	}

	@Override
	public Collection<Category> directChildren(Category one) {
		final Logger logger = LoggerFactory.getLogger(getClass());
		logger.debug(getClass().getSimpleName() + ".directChildren(Category) start:" + one);
		final Long id = one.getId();
		Collection<Category> list = list();
		final Collection<Category> rt = Lists.newArrayList();
		for (Category category : list) {
			if (id.equals(category.getParentId())) {
				rt.add(category);
			}
		}
		logger.debug(getClass().getSimpleName() + ".directChildren(Category) end:" + rt);
		return rt;
	}

	@Override
	public Collection<Category> subordinates(Category one) {
		final Logger logger = LoggerFactory.getLogger(getClass());
		logger.debug(getClass().getSimpleName() + ".subordinates(Category) start:" + one);
		final Collection<Category> rt = Lists.newArrayList();
		Collection<Category> children = directChildren(one);
		for (Category category : children) {
			rt.add(category);
			rt.addAll(subordinates(category));
		}
		logger.debug(getClass().getSimpleName() + ".subordinates(Category) end:" + rt);
		return rt;
	}

	@Override
	public String getCodeById(Long bindingId) {
		if(bindingId == null) {
			return null;
		}
		return categoryMapper.selectByPrimaryKey(bindingId).getCode();
	}

	@Override
	public List<Long> getCategoryIdsByCampaignId(Long campaignId, Boolean temp) {
		List<CategoryEntity> entities = categoryMapper.getCategoryByCampaignId(campaignId, (temp ? 1 : 0));
		List<Long> list = Lists.newArrayList();
		for (CategoryEntity entity : entities) {
			list.add(entity.getId());
		}
		return list;
	}
}
