package com.thebeastshop.pegasus.component.campaign;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Map;

import com.google.common.collect.Lists;
import com.thebeastshop.pegasus.component.discount.DiscountType;
import com.thebeastshop.cart.ProductPack;
import com.thebeastshop.support.enums.CartPackSource;
import com.thebeastshop.support.util.PriceUtil;

/**
 * @author Liang Wenjian
 */
public interface CampaignResult {

	// --------------------------活动部分
	
	Campaign getCampaign();

	String getCampaignTitle();

	DiscountType getDiscountType();

	ProductScope getProductScope();

	// ---------------------商品包部分

	// -------------全局

	/**
	 * 是否满足参与活动的条件
	 *
	 * @return
	 */
	boolean isReach();

	/**
	 * 商品包和参与活动之前它的商品的价格（单件）的对应关系
	 * 
	 * @return
	 */
	Map<ProductPack, BigDecimal> getOriginPriceMap();

	/**
	 * 商品包和参与活动之后它的商品的价格（单件）的对应关系
	 *
	 * @return
	 */
	Map<ProductPack, BigDecimal> getFinalPriceMap();

	/**
	 * 获取原价。=参与包的参与价+未参与包的价钱
	 *
	 * @return
	 */
	BigDecimal getRawPrice();

	/**
	 * 已获得的所有包,=参与包s+未参与包s+额外包中已获得的
	 *
	 * @return
	 */
	Collection<ProductPack> getOwnedPacks();

	/**
	 * 已获得的所有商品的最终总价
	 *
	 * @return
	 */
	BigDecimal getOwnedPrice();

	/**
	 * 参与本活动节省的钱。（=参与活动的商品包在参与前的价格-参与后的价格）
	 *
	 * @return
	 */
	BigDecimal getSavedPrice();

	// -----------参与包

	/**
	 * 参与此活动的商品包。 注：是结果商品包，即包中的商品的属性（比如价格），是参与活动之后的结果。
	 *
	 * @return
	 */
	Collection<ProductPack> getParticipatingPacks();

	/**
	 * 参与活动的商品包最终价
	 *
	 * @return
	 */
	BigDecimal getParticipatedPrice();

	/**
	 * 参与活动的商品的原价
	 *
	 * @return
	 */
	BigDecimal getParticipatingPrice();

	// /**
	// * 参与活动的商品包和它的商品节省的钱（单件）的对应关系
	// *
	// * @return
	// */
	// default Map<ProductPack, BigDecimal> getParticipatingPacksSavedMap() {
	// final Map<ProductPack, BigDecimal> rt = Maps.newHashMap();
	// final Collection<ProductPack> participatingPacks =
	// getParticipatingPacks();
	// if (participatingPacks != null && participatingPacks.isEmpty()) {
	// final Map<ProductPack, BigDecimal> finalPriceMap = getFinalPriceMap();
	// for (final ProductPack participatingPack : participatingPacks) {
	// rt.put(participatingPack,
	// HasPrice.keepToCent(participatingPack.getPrice() -
	// finalPriceMap.get(participatingPack)));
	// }
	// }
	// return rt;
	// }

	// ---------------未参与包

	/**
	 * 未参与活动的商品包（不包括额外包）
	 *
	 * @return
	 */
	Collection<ProductPack> getNonParticipatingPacks();

	// ----------------额外包

	/**
	 * 额外获得的商品包
	 *
	 * @return
	 */
	Collection<ProductPack> getAdditionalPacks();

	// ============================================================

	class DefaultCampaignResultImpl implements CampaignResult {
		private final Campaign campaign;
		private final boolean reach;
		private final Map<ProductPack, BigDecimal> originPriceMap;
		private final Map<ProductPack, BigDecimal> finalPriceMap;
		private final Collection<ProductPack> participatingPacks;
		private final BigDecimal participatingPrice;
		private final Collection<ProductPack> nonParticipatingPacks;
		private final BigDecimal participatedPrice;

		private final Collection<ProductPack> additionalPacks;

		public DefaultCampaignResultImpl(final Campaign campaign, final boolean reach,
				final Collection<ProductPack> participatingPacks, final BigDecimal participatingPrice,
				final Map<ProductPack, BigDecimal> originPriceMap, final Map<ProductPack, BigDecimal> finalPriceMap,
				final Collection<ProductPack> nonParticipatingPacks, final BigDecimal participatedPrice,
				final Collection<ProductPack> additionalPacks) {
			this.campaign = campaign;
			this.participatingPacks = participatingPacks;
			this.originPriceMap = originPriceMap;
			this.finalPriceMap = finalPriceMap;
			this.nonParticipatingPacks = nonParticipatingPacks;
			this.participatingPrice = participatingPrice;
			this.participatedPrice = participatedPrice;
			this.additionalPacks = additionalPacks;
			this.reach = reach;
		}

		@Override
		public Campaign getCampaign() {
			return campaign;
		}

		@Override
		public String getCampaignTitle() {
			return campaign.getDescription();
		}

		@Override
		public DiscountType getDiscountType() {
			return campaign.getDiscountType();
		}

		@Override
		public ProductScope getProductScope() {
			return campaign.getProductScope();
		}

		@Override
		public Collection<ProductPack> getNonParticipatingPacks() {
			return nonParticipatingPacks;
		}

		@Override
		public Collection<ProductPack> getParticipatingPacks() {
			return participatingPacks;
		}

		@Override
		public BigDecimal getParticipatingPrice() {
			return participatingPrice;
		}
		
		@Override
		public Map<ProductPack, BigDecimal> getOriginPriceMap() {
			return originPriceMap;
		}

		@Override
		public Map<ProductPack, BigDecimal> getFinalPriceMap() {
			return finalPriceMap;
		}

		@Override
		public BigDecimal getParticipatedPrice() {
			return participatedPrice;
		}

		@Override
		public Collection<ProductPack> getAdditionalPacks() {
			return additionalPacks;
		}

		@Override
		public boolean isReach() {
			return reach;
		}

		@Override
		public String toString() {
			return getClass().getSimpleName() + hashCode() + ":{" + "campaign:" + campaign + ", reach:" + isReach()
					+ ", participatingPacks:" + getParticipatingPacks() + ", participatingPrice:"
					+ getParticipatingPrice() + ", finalPriceMap:" + getFinalPriceMap() + ", nonParticipatingPacks:"
					+ getNonParticipatingPacks() + ", participatedPrice:" + getParticipatedPrice()
					+ ", additionalPacks:" + getAdditionalPacks() + ", ownedPacks:" + getOwnedPacks() + ", ownedPrice:"
					+ getOwnedPrice() + '}';
		}

		@Override
		public Collection<ProductPack> getOwnedPacks() {
			final Collection<ProductPack> rt = Lists.newArrayList(getParticipatingPacks());
			rt.addAll(getNonParticipatingPacks());
			final Collection<ProductPack> additionalPacks = getAdditionalPacks();
			for (ProductPack productPack : additionalPacks) {
				if (productPack.getId() != null) {// 判断有包裹id 就是 客户 加入到购物车的商品
					rt.add(productPack);
				}
			}
			return rt;
		}

		@Override
		public BigDecimal getOwnedPrice() {
			final Collection<ProductPack> ownedPacks = getOwnedPacks();
			final Map<ProductPack, BigDecimal> finalPriceMap = getFinalPriceMap();
			BigDecimal rt = BigDecimal.ZERO;
			for (final ProductPack ownedPack : ownedPacks) {
				if (!ownedPack.getSource().equals(CartPackSource.GIFT)) { // 赠品的价格不加入
					final BigDecimal finalPrice = finalPriceMap.get(ownedPack);
					if (finalPrice == null) {
						rt = rt.add(ownedPack.getPrice());
					} else {
						rt = rt.add(finalPrice.multiply(BigDecimal.valueOf(ownedPack.getCount())));
					}
				}
			}
			return PriceUtil.keepToCent(rt);
		}

		@Override
		public BigDecimal getSavedPrice() {
			return PriceUtil.keepToCent(getParticipatingPrice().subtract(getParticipatedPrice()));
		}

		@Override
		public BigDecimal getRawPrice() {
			final BigDecimal participatingPrice = getParticipatingPrice();
			final Collection<ProductPack> nonParticipatingPacks = getNonParticipatingPacks();
			return PriceUtil.sumPrice(nonParticipatingPacks).add(participatingPrice);
		}
		
	}
}
