package com.thebeastshop.pegasus.component.adaptor.payment.service.impl;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.pegasus.component.adaptor.BaseService;
import com.thebeastshop.pegasus.component.adaptor.payment.config.AlipayConfig;
import com.thebeastshop.pegasus.component.adaptor.payment.constants.AlipayConstants;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api1.AlipayCrossBorderMobilePayParams;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api1.AlipayDirectPayParams;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api1.AlipayMobilePayParams;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api1.AlipayRequest_API1;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api2.AlipayTradeQueryParams;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api2.AlipayTradeQueryRequest;
import com.thebeastshop.pegasus.component.adaptor.payment.domain.alipay.api2.AlipayTradeQueryResponse;
import com.thebeastshop.pegasus.component.adaptor.payment.service.AliPayService;
import com.thebeastshop.support.exception.PlatErrInfo;
import com.thebeastshop.support.exception.PlatInternalException;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.util.BeanUtil;
import com.thebeastshop.support.util.HttpUtil;
import com.thebeastshop.support.util.JsonUtil;
import com.thebeastshop.support.util.MD5SignUtil;
import com.thebeastshop.support.util.RSASignUtil;

@Service
public class AliPayServiceImpl extends BaseService implements AliPayService {

	// ===============private==================
	/**
	 * 将支付宝返回字符串转换为对应Response对象
	 * 
	 * @param content
	 * @param clazz
	 * @param charset
	 * @return
	 * @throws Exception
	 */
	private <T> T toResponse(String content, Class<T> clazz, String charset) throws Exception {
		Map<String, Object> map = JsonUtil.toMap(content);
		Collection<Object> values = map.values();

		@SuppressWarnings("unchecked")
		Map<String, Object> dataMap = (Map<String, Object>) values.iterator().next();
		T response = BeanUtil.map2Bean(dataMap, clazz);

		String alipaySign = map.get("sign") != null ? map.get("sign").toString() : "";
		if (!RSASignUtil.rsaCheckSign(JsonUtil.toJson(dataMap), alipaySign, AlipayConfig.ALIPAY_PUBLIC_KEY, charset)) {
			throw new PlatInternalException(new PlatErrInfo("支付宝调用返回:签名错误", "sign", alipaySign));
		}
		return response;
	}

	// ==============华丽丽的分割线===============

	@Override
	public boolean notifyVerify(String notify_id, boolean isCrossBorder) {
		try {
			logger.info("验证是否是支付宝发来的通知请求数据：" + notify_id);

			String partner = "";
			if (isCrossBorder) {
				partner = AlipayConfig.CROSS_BORDER_PARTNER;
			} else {
				partner = AlipayConfig.PARTNER;
			}
			Map<String, String> params = Maps.newHashMap();
			params.put("service", "notify_verify");
			params.put("partner", partner);
			params.put("notify_id", notify_id);
			String rt = HttpUtil.doGet(AlipayConfig.API_1_0, params, AlipayConfig.CHARSET);
			if (StringUtils.isNotBlank(rt)) {
				return Boolean.valueOf(rt);
			}
		} catch (Exception e) {
			logger.error("验证是否是支付宝发来的通知错误! e={}", e);
			throw new UnknownError("验证是否是支付宝发来的通知异常: " + e.getMessage() + " " + e.getStackTrace());
		}
		return false;
	}

	@Override
	public String mobilePay(AlipayMobilePayParams params) {
		try {
			AlipayRequest_API1 request = params.toRequest();
			// 设置service
			request.setService(AlipayConfig.MOBILE_PAY_SERVICE);
			// 设置partner
			request.setPartner(AlipayConfig.PARTNER);
			// 设置签名
			request.setSign(this.sign(request, request.getSign_type(), false));

			logger.info("支付宝移动支付请求数据：" + request.toString());

			if (request.getSign_type().equalsIgnoreCase(AlipayConfig.RSA_SIGN_TYPE)) {
				return RSASignUtil.getSignContent(request);
			} else if (request.getSign_type().equalsIgnoreCase(AlipayConfig.MD5_SIGN_TYPE)) {
				return MD5SignUtil.getSignContent(request);
			} else {
				throw new WrongArgException("请输入正确签名方式（MD5或者RSA）", "sign_type", request.getSign_type());
			}
		} catch (WrongArgException e) {
			throw e;
		} catch (Exception e) {
			throw new UnknownError("支付宝移动支付异常: " + e.getMessage() + " " + e.getStackTrace());
		}
	}

	@Override
	public String crossBorderMobilePay(AlipayCrossBorderMobilePayParams params) {
		try {
			AlipayRequest_API1 request = params.toRequest();
			// 设置service
			request.setService(AlipayConfig.MOBILE_PAY_SERVICE);
			// 设置partner
			request.setPartner(AlipayConfig.CROSS_BORDER_PARTNER);
			// 设置签名
			request.setSign(this.sign(request, request.getSign_type(), true));

			logger.info("支付宝国际移动支付请求数据：" + request.toString());

			if (request.getSign_type().equalsIgnoreCase(AlipayConfig.RSA_SIGN_TYPE)) {
				return RSASignUtil.getSignContent(request);
			} else if (request.getSign_type().equalsIgnoreCase(AlipayConfig.MD5_SIGN_TYPE)) {
				return MD5SignUtil.getSignContent(request);
			} else {
				throw new WrongArgException("请输入正确签名方式（MD5或者RSA）", "sign_type", request.getSign_type());
			}
		} catch (WrongArgException e) {
			throw e;
		} catch (Exception e) {
			throw new UnknownError("支付宝国际移动支付异常: " + e.getMessage() + " " + e.getStackTrace());
		}
	}

	@Override
	public String directPay(AlipayDirectPayParams params) {
		try {
			AlipayRequest_API1 request = params.toRequest();
			// 设置service
			request.setService(AlipayConfig.DIRECT_PAY_SERVICE);
			// 设置partner
			request.setPartner(AlipayConfig.PARTNER);
			// 设置签名
			request.setSign(this.sign(request, AlipayConfig.MD5_SIGN_TYPE, false));

			logger.info("支付宝即时到账请求数据：" + request.toString());

			String link = AlipayConfig.API_1_0 + HttpUtil.buildQuery(BeanUtil.bean2Map(request),
					Lists.<String> newArrayList(), request.get_input_charset());

			logger.info("支付宝即时到账返回链接：" + link);

			return link;

		} catch (Exception e) {
			throw new UnknownError("支付宝即时到账异常: " + e.getMessage() + " " + e.getStackTrace());
		}
	}

	@Override
	public AlipayTradeQueryResponse tradeQuery(AlipayTradeQueryParams params) {
		try {
			AlipayTradeQueryRequest request = new AlipayTradeQueryRequest();
			// 设置appid
			request.setApp_id(AlipayConfig.APP_ID);
			// 设置service
			request.setMethod(AlipayConfig.TRADE_QUERY_METHOD);
			// 设置时间戳
			DateFormat df = new SimpleDateFormat(AlipayConstants.DATE_TIME_FORMAT);
			df.setTimeZone(TimeZone.getTimeZone(AlipayConstants.DATE_TIMEZONE));
			request.setTimestamp(df.format(new Date()));
			// 设置调用的接口版本
			request.setVersion(AlipayConfig.VERSION);
			// 设置业务请求接口的参数体
			request.setBiz_content(JsonUtil.toJson(params));
			// 设置签名
			request.setSign(this.sign(request, AlipayConfig.RSA_SIGN_TYPE, false));

			logger.info("支付宝交易查询请求数据：" + request.toString());

			String dataStr = HttpUtil.doFormPost(AlipayConfig.API_2_0, BeanUtil.bean2Map(request),
					request.getCharset());

			AlipayTradeQueryResponse response = this.toResponse(dataStr, AlipayTradeQueryResponse.class,
					request.getCharset());

			logger.info("支付宝交易查询返回数据：" + response);

			return response;

		} catch (Exception e) {
			throw new UnknownError("支付宝交易查询异常: " + e.getMessage() + " " + e.getStackTrace());
		}

	}

	@Override
	public String sign(Object object, String signType, boolean isCrossBorder) {
		try {
			if (signType.equalsIgnoreCase(AlipayConfig.RSA_SIGN_TYPE)) {
				// RSA签名
				String private_key = "";
				if (isCrossBorder) {
					private_key = AlipayConfig.CROSS_BORDER_PRIVATE_KEY;
				} else {
					private_key = AlipayConfig.PRIVATE_KEY;
				}
				return RSASignUtil.rsaSign(object, private_key, AlipayConfig.CHARSET);
			} else if (signType.equalsIgnoreCase(AlipayConfig.MD5_SIGN_TYPE)) {
				// MD5签名
				Map<String, Object> filterPara = HttpUtil.paraFilter(BeanUtil.bean2Map(object),
						Arrays.asList("sign", "sign_type"));
				String md5_key = "";
				if (isCrossBorder) {
					md5_key = AlipayConfig.CROSS_BORDER_MD5_KEY;
				} else {
					md5_key = AlipayConfig.MD5_KEY;
				}
				return MD5SignUtil.md5Sign(filterPara, md5_key, AlipayConfig.CHARSET, false);
			}
		} catch (Exception e) {
			throw new WrongArgException("支付宝签名出错", "object:", object);
		}
		return "";
	}

}
