package com.thebeastshop.pegasus.channelservice.service.impl;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.cart.ProductPack;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.pegasus.channelservice.service.LabelService;
import com.thebeastshop.pegasus.channelservice.service.OverseaService;
import com.thebeastshop.pegasus.channelservice.util.ConstUtil;
import com.thebeastshop.pegasus.component.campaign.Campaign;
import com.thebeastshop.pegasus.component.campaign.CampaignResult;
import com.thebeastshop.pegasus.component.campaign.CampaignSection;
import com.thebeastshop.pegasus.component.campaign.ConditionType;
import com.thebeastshop.pegasus.component.campaign.ProductScope;
import com.thebeastshop.pegasus.component.delivery.constants.PostageConstant;
import com.thebeastshop.pegasus.component.discount.DiscountType;
import com.thebeastshop.pegasus.merchandise.vo.PsCampaignVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuCombinationVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import com.thebeastshop.support.Label;
import com.thebeastshop.support.Label.CampaignVO;
import com.thebeastshop.support.Label.LabelMode;
import com.thebeastshop.support.Label.PostageVO;
import com.thebeastshop.support.util.EnumUtil;
import com.thebeastshop.support.util.PriceUtil;
import com.thebeastshop.support.util.label.LabelLinkUtil;

/**
 * 
 * @author Paul-xiong
 *
 */
@Service
public class LabelServiceImpl implements LabelService {

	@Autowired
	private OverseaService overseaService;

	// =================private===================
	private PostageVO getPostage(BigDecimal price) {
		PostageVO postageVO = null;
		if (price != null) {
			BigDecimal postagePrice = PostageConstant.MEET_PRICE;
			boolean reach = price.compareTo(postagePrice) >= 0 ? true : false;
			StringBuilder linkDesc = new StringBuilder();
			if (reach) {
				linkDesc.append("已满足");
			} else {
				linkDesc.append("还差").append(PriceUtil.keepToCent(postagePrice.subtract(price))).append("元，去凑单");
			}
			postageVO = new PostageVO();
			postageVO.setFulfilled(reach);
			postageVO.setLinkDesc(linkDesc.toString());
		}
		return postageVO;
	}

	/**
	 * 判断标签是否已存在
	 * 
	 * @param labels
	 * @param campaignResult
	 * @return
	 */
	private boolean checkLabelIsExist(List<Label> labels, CampaignResult campaignResult) {
		boolean flag = false;
		if (CollectionUtils.isNotEmpty(labels)) {
			for (Label label : labels) {
				if (label != null && label.getSummary().equals(campaignResult.getCampaignTitle())) {
					flag = true;
				}
			}
		}
		return flag;
	}

	/**
	 * 获取活动信息 活动区间满足，当所有区间都满足时返回已满足，其他提示差额和凑单信息
	 * 
	 * @param campaignResult
	 * @return
	 */
	private CampaignVO getCampaign(CampaignResult campaignResult) {
		CampaignVO campaignVO = null;
		if (null == campaignResult) {
			return campaignVO;
		} else {
			if (DiscountType.GIFT.equals(campaignResult.getDiscountType())) {
				Campaign campaign = campaignResult.getCampaign();
				boolean reach = true;
				StringBuilder linkDesc = new StringBuilder();
				if (CollectionUtils.isNotEmpty(campaign.getCampaignSections())) {
					BigDecimal price = campaignResult.getParticipatedPrice();
					int count = 0;
					Collection<ProductPack> participatingPacks = campaignResult.getParticipatingPacks();
					if (CollectionUtils.isNotEmpty(participatingPacks)) {
						for (final ProductPack productPack : participatingPacks) {
							count += productPack.getCount();
						}
					}
					if (campaignResult.isReach()) {
						if (campaign.isCumulative()) {
							linkDesc.append("继续购买可获取更多");
							reach = false;
						} else {
							for (CampaignSection campaignSection : campaign.getCampaignSections()) {
								if (!campaignSection.getConditionType().matchCondition(price, count,
										campaignSection.getLine())) {
									linkDesc.append("继续购买可获取更多");
									reach = false;
									break;
								}
							}
						}

					} else {
						CampaignSection campaignSection = campaign.getCampaignSections().get(0);
						BigDecimal line = campaignSection.getLine();
						if (!campaignSection.getConditionType().matchCondition(price, count, line)) {
							ConditionType conditionType = campaignSection.getConditionType();
							if (conditionType.equals(ConditionType.PRICE)) {
								linkDesc.append("还差").append(PriceUtil.keepToCent(line.subtract(price)))
										.append("元，去凑单");
							} else if (conditionType.equals(ConditionType.PRODUCT_COUNT)) {
								linkDesc.append("还差").append(line.intValue() - count).append("件，去凑单");
							}
							reach = false;
						}
					}

				}

				if (reach) {
					linkDesc.append("已满足");
				}
				campaignVO = new CampaignVO();
				campaignVO.setId(campaign.getId());
				campaignVO.setName(campaign.getName());
				campaignVO.setFulfilled(reach);
				campaignVO.setLinkDesc(linkDesc.toString());
			} else {
				// 多品凑单提醒
				if (campaignResult.getProductScope().equals(ProductScope.MULTI)) { // 多品活动
					Campaign campaign = campaignResult.getCampaign();
					boolean reach = campaignResult.isReach();
					StringBuilder linkDesc = new StringBuilder();
					if (reach) {
						linkDesc.append("已满足");
					} else {
						if (CollectionUtils.isNotEmpty(campaign.getCampaignSections())) {
							CampaignSection firstSection = campaign.getCampaignSections().get(0); // 第一个活动区间
							BigDecimal line = firstSection.getLine();
							ConditionType conditionType = firstSection.getConditionType();
							if (conditionType.equals(ConditionType.PRICE)) {
								BigDecimal participatedPrice = campaignResult.getParticipatedPrice();
								linkDesc.append("还差").append(PriceUtil.keepToCent(line.subtract(participatedPrice)))
										.append("元，去凑单");
							} else if (conditionType.equals(ConditionType.PRODUCT_COUNT)) {
								Collection<ProductPack> participatingPacks = campaignResult.getParticipatingPacks();
								int participatingCount = 0;
								if (CollectionUtils.isNotEmpty(participatingPacks)) {
									for (final ProductPack productPack : participatingPacks) {
										participatingCount += productPack.getCount();
									}
								}
								linkDesc.append("还差").append(line.intValue() - participatingCount).append("件，去凑单");
							}
						}
					}

					campaignVO = new CampaignVO();
					campaignVO.setId(campaign.getId());
					campaignVO.setName(campaign.getName());
					campaignVO.setFulfilled(reach);
					campaignVO.setLinkDesc(linkDesc.toString());
				}
			}

		}
		return campaignVO;
	}

	/**
	 * 获取单品活动Label详情信息 活动区间满足，当所有区间都满足时返回已满足，其他提示差额和凑单信息
	 *
	 * @param campaignResult
	 * @return
	 */
	private CampaignVO getSingleCampaign(CampaignResult campaignResult) {
		CampaignVO campaignVO = null;
		if (campaignResult != null && DiscountType.GIFT.equals(campaignResult.getDiscountType())
				&& ProductScope.SINGLE.equals(campaignResult.getProductScope())) {
			// 设置活动信息
			Campaign campaign = campaignResult.getCampaign();
			campaignVO = new CampaignVO();
			campaignVO.setId(campaign.getId());
			campaignVO.setName(campaign.getName());

			// 匹配活动是否满足，设置活动提醒
			boolean reach = campaignResult.isReach();
			StringBuilder linkDesc = new StringBuilder();
			List<CampaignSection> campaignSections = campaign.getCampaignSections();
			if (CollectionUtils.isNotEmpty(campaignSections)) {
				// 满足活动商品累计价格
				BigDecimal price = campaignResult.getParticipatedPrice();
				// 满足活动商品累计数量
				int count = 0;
				Collection<ProductPack> participatingPacks = campaignResult.getParticipatingPacks();
				if (CollectionUtils.isNotEmpty(participatingPacks)) {
					for (final ProductPack productPack : participatingPacks) {
						count += productPack.getCount();
					}
				}

				if (reach) {
					// 可累计活动，设置满足状态为false
					if (campaign.isCumulative()) {
						reach = false;
						linkDesc.append("继续购买可获更多");
					} else {
						// 阶梯活动满足提示
						for (CampaignSection campaignSection : campaignSections) {
							BigDecimal line = campaignSection.getLine();
							if (!campaignSection.getConditionType().matchCondition(price, count,
									campaignSection.getLine())) {
								ConditionType conditionType = campaignSection.getConditionType();
								if (conditionType.equals(ConditionType.PRICE)) {
									linkDesc.append("还差").append(PriceUtil.keepToCent(line.subtract(price)))
											.append("元");
								} else if (conditionType.equals(ConditionType.PRODUCT_COUNT)) {
									linkDesc.append("还差").append(line.intValue() - count).append("件");
								}
								reach = false;
								break;
							}
						}
					}
				} else { // 未满足设置差的件数
					ConditionType conditionType = campaignSections.get(0).getConditionType();
					BigDecimal line = campaignSections.get(0).getLine();
					if (conditionType.equals(ConditionType.PRICE)) {
						linkDesc.append("还差").append(PriceUtil.keepToCent(line.subtract(price))).append("元");
					} else if (conditionType.equals(ConditionType.PRODUCT_COUNT)) {
						linkDesc.append("还差").append(line.intValue() - count).append("件");
					}
					reach = false;
				}
			}

			if (reach) {
				linkDesc.append("已满足");
			}
			campaignVO.setFulfilled(reach);
			campaignVO.setLinkDesc(linkDesc.toString());

		}
		return campaignVO;
	}

	private Label buildOverSeaLabel() {
		return new Label("跨境商品", "#FFFFFF", "#000000", "本商品价格已含跨境电商综合税", "#333333", "", 0, true);
	}

	private Label buildOptionLabel(final CampaignResult campaignResult) {
		final Label label = new Label();
		label.setMode(LabelMode.CAMPAIGN);
		label.setType("N元选");
		label.setSummary(campaignResult.getCampaignTitle());
		label.setCampaign(getCampaign(campaignResult));
		return label;
	}

	private Label buildCheapenLabel(final CampaignResult campaignResult) {
		final Label label = new Label();
		label.setMode(LabelMode.CAMPAIGN);
		label.setType("满减");
		label.setSummary(campaignResult.getCampaignTitle());
		label.setCampaign(getCampaign(campaignResult));
		return label;
	}

	private Label buildDicLabel(final CampaignResult campaignResult) {
		final Label label = new Label();
		label.setMode(LabelMode.CAMPAIGN);
		label.setType("满折");
		label.setSummary(campaignResult.getCampaignTitle());
		label.setCampaign(getCampaign(campaignResult));
		return label;
	}

	private Collection<Label> _buildOldGiftLabel(final CampaignResult campaignResult, AccessWayEnum accessWay) {
		final Collection<Label> labels = Lists.newArrayList();
		if (campaignResult != null) {
			if (campaignResult.isReach()) { // 活动达标
				final Collection<ProductPack> gifts = campaignResult.getAdditionalPacks();
				if (CollectionUtils.isNotEmpty(gifts)) {
					for (ProductPack gift : gifts) {
						PsProductVO product = gift.getProduct();
						int count = gift.getCount();
						final Label label = new Label();
						label.setMode(LabelMode.CAMPAIGN);
						label.setType(count <= 0 ? "已赠完" : "赠品");
						label.setLink(LabelLinkUtil.getProductLink(product.getCode(), accessWay));
						label.setSummary(product.getNameCn());
						label.setSummaryColor(count <= 0 ? "#000000" : "#333333");
						label.setValid(count <= 0 ? false : true);
						label.setCount(count);
						CampaignVO campaignVO = new CampaignVO();
						campaignVO.setId(campaignResult.getCampaign().getId());
						campaignVO.setName(campaignResult.getCampaign().getName());
						campaignVO.setLinkDesc(count <= 0 ? "已赠完" : "已获得");
						campaignVO.setFulfilled(true);
						label.setCampaign(campaignVO);
						labels.add(label);
					}
				}
			}
		}
		return labels;
	}

	private Collection<Label> buildGiftLabel(final CampaignResult campaignResult, AccessWayEnum accessWay) {
		Collection<Label> labels = Lists.newArrayList();
		if (accessWay.equals(AccessWayEnum.PC)) {
			labels = _buildOldGiftLabel(campaignResult, accessWay);
		}
		if (campaignResult != null) { // 多品活动
			final Label label = new Label();
			label.setMode(LabelMode.CAMPAIGN);
			if (campaignResult.getCampaign().getId() == -1) {
				// 首单赠品活动
				label.setType("首单赠");
			} else {
				label.setType("满赠");
			}
			label.setSummary(campaignResult.getCampaignTitle());
			CampaignVO campaignVO = getCampaign(campaignResult);
			label.setCampaign(campaignVO);
			if (!campaignVO.getFulfilled()) {
				labels.add(label);
			}
		}
		return labels;
	}

	private Collection<Label> buildSingleGiftLabel(final CampaignResult campaignResult, AccessWayEnum accessWay) {
		Collection<Label> labels = Lists.newArrayList();
		if (campaignResult != null && campaignResult.getProductScope().equals(ProductScope.SINGLE)) { // 单品活动
			if (accessWay.equals(AccessWayEnum.PC)) {
				labels = _buildOldGiftLabel(campaignResult, accessWay);
			}
			final Label label = new Label();
			label.setMode(LabelMode.CAMPAIGN);
			label.setType("满赠");
			label.setSummary(campaignResult.getCampaignTitle());
			label.setCampaign(getSingleCampaign(campaignResult));
			label.setJump(false);
			// 不满足活动显示单品活动凑单信息
			if (!label.getCampaign().getFulfilled()) {
				labels.add(label);
			}

		}
		return labels;
	}

	// ==============================================
	@Override
	public Map<PsProductVO, List<Label>> getProductLabels(Collection<PsProductVO> products) {
		Map<PsProductVO, List<Label>> map = Maps.newLinkedHashMap();
		for (PsProductVO product : products) {
			List<Label> labels = Lists.newArrayList();
			// if (product.getCrossBorderFlag() != null &&
			// product.getCrossBorderFlag().equals(1)) {
			// labels.add(buildOverSeaLabel());
			// }
			// 活动标签
			List<PsCampaignVO> campaigns = product.getCampaignList();

			List<PsSpvVO> spvs = product.getSpvList();
			boolean status = false;
			for (PsSpvVO spv : spvs) {
				List<PsCampaignVO> spvCampaigns = spv.getCampaignList();
				if (null != spvCampaigns && spvCampaigns.size() > 0) {
					for (PsCampaignVO spvCampaign : spvCampaigns) {
						DiscountType discountType = EnumUtil.valueOf(spvCampaign.getDiscountTypeId(),
								DiscountType.class);
						if (DiscountType.PANIC_BUY.equals(discountType)) {
							status = true;
							break;
						}
					}
				}
			}

			if (CollectionUtils.isNotEmpty(campaigns)) {

				for (PsCampaignVO psCampaignVO : campaigns) {
					// TODO rd_3821 过滤详情换购购物袋
					// List<String> excludeCampaigns =
					// Lists.newArrayList("CP000000000382","CP000000000383","CP000000000765");
					// excludeCampaigns.add("CP000000001503");
					if (ConstUtil.excludeCampaigns.contains(psCampaignVO.getCode())) {
						continue;
					}
					DiscountType discountType = EnumUtil.valueOf(psCampaignVO.getDiscountTypeId(), DiscountType.class);
					if (DiscountType.OFF.equals(discountType)) {
						continue;
					}
					if (status) {
						if (DiscountType.CHEAPEN.equals(discountType)) {
							continue;
						}
						if (DiscountType.DIC.equals(discountType)) {
							continue;
						}
						if (DiscountType.OPTION.equals(discountType)) {
							continue;
						}
					}
					Label label = new Label(psCampaignVO.getDiscountTypeName(), "#FFFFFF", "#000000",
							psCampaignVO.getTitle(), "#333333", "", 0, true);
					labels.add(label);
				}
			}

			map.put(product, labels);
		}
		return map;
	}

	@Override
	public Map<PsSpvVO, List<Label>> getSpvLabels(Collection<PsSpvVO> spvs, AccessWayEnum accessWay) {
		Map<PsSpvVO, List<Label>> map = Maps.newLinkedHashMap();
		// Map<PsSpvVO, Boolean> spvsIsOverseaMap =
		// overseaService.mapSpvsIsOversea(spvs);
		for (PsSpvVO spv : spvs) {
			List<Label> labels = Lists.newArrayList();
			if (spv.getSkuVO() != null) {
				// 如果是海淘商品
				if (spv.getSkuVO().getCrossBorderFlag() != null && spv.getSkuVO().getCrossBorderFlag() == 1) {
					labels.add(buildOverSeaLabel());
				}

				// 如果是组合商品
			} else if (spv.getCombined() && CollectionUtils.isNotEmpty(spv.getSkuCombinationList())) {

				for (PsSkuCombinationVO psSkuCombinationVO : spv.getSkuCombinationList()) {

					if (psSkuCombinationVO.getSubSkuVO().getCrossBorderFlag() != null
							&& psSkuCombinationVO.getSubSkuVO().getCrossBorderFlag() == 1) {
						labels.add(buildOverSeaLabel());
						break;
					}
				}

			}
			map.put(spv, labels);
		}
		return map;
	}

	@Override
	public List<Label> getCartLabels(List<CampaignResult> campaignResults, AccessWayEnum accessWay) {
		final List<Label> cartLabels = Lists.newArrayList();// 针对购物车的标签
		if (campaignResults != null && !campaignResults.isEmpty()) {
			// 把有单品折扣的商品包的id记录下来，为了贴“折上折”标签
			for (int i = 0, size = campaignResults.size(); i < size; i++) {
				final CampaignResult campaignResult = campaignResults.get(i);
				final ProductScope productScope = campaignResult.getProductScope();
				final DiscountType discountType = campaignResult.getDiscountType();
				if (checkLabelIsExist(cartLabels, campaignResult)) {
					continue;
				}
				if (productScope == ProductScope.MULTI) {// 多品活动
					if (discountType == DiscountType.CHEAPEN) {// 满减：给购物车贴标签。
						cartLabels.add(buildCheapenLabel(campaignResult));
					} else if (discountType == DiscountType.GIFT) {// 赠品：给购物车贴标签。
						cartLabels.addAll(buildGiftLabel(campaignResult, accessWay));
					} else if (discountType == DiscountType.DIC) {// 满折：给购物车贴标签。
						cartLabels.add(buildDicLabel(campaignResult));
					} else if (discountType == DiscountType.OPTION) {// N元选：给商品包贴标签
						cartLabels.add(buildOptionLabel(campaignResult));
					}
				} else if (campaignResult.isReach() && productScope == ProductScope.GLOBAL) {// 全场活动
					if (discountType == DiscountType.CHEAPEN) {// 满减：给购物车贴标签。
						cartLabels.add(buildCheapenLabel(campaignResult));
					} else if (discountType == DiscountType.GIFT) {// 赠品：给购物车贴标签。
						cartLabels.addAll(buildGiftLabel(campaignResult, accessWay));
					} else if (discountType == DiscountType.DIC) {// 满折：给购物车贴标签。
						cartLabels.add(buildDicLabel(campaignResult));
					} else if (discountType == DiscountType.OPTION) {// N元选：给商品包贴标签
						cartLabels.add(buildOptionLabel(campaignResult));
					}
				}
			} // for
		}
		return cartLabels;

	}

	@Override
	public List<Label> getLabels(ProductPack pack, List<CampaignResult> campaignResults, AccessWayEnum accessWay) {
		final List<Label> labels = Lists.newArrayList();
		if (campaignResults != null && !campaignResults.isEmpty()) {
			for (int i = 0, size = campaignResults.size(); i < size; i++) {
				final CampaignResult campaignResult = campaignResults.get(i);
				final ProductScope productScope = campaignResult.getProductScope();
				final DiscountType discountType = campaignResult.getDiscountType();

				final Collection<ProductPack> participatingPacks = campaignResult.getParticipatingPacks();

				List<Long> ids = Lists.newArrayList();
				for (ProductPack productPack : participatingPacks) {
					ids.add(productPack.getSpvId());
				}
				if (ids.contains(pack.getSpvId())) {

					if (campaignResult.isReach()) {
						if (checkLabelIsExist(labels, campaignResult)) {
							continue;
						}
						if (productScope == ProductScope.SINGLE) {// 单品活动
							if (discountType == DiscountType.CHEAPEN) {// 满减：给商品包贴标签
								labels.add(buildCheapenLabel(campaignResult));
							} else if (discountType == DiscountType.GIFT) {// 赠品：给商品包贴标签
								labels.addAll(buildSingleGiftLabel(campaignResult, accessWay));
							} else if (discountType == DiscountType.DIC) {// 满折：给商品包贴标签
								labels.add(buildDicLabel(campaignResult));
							} else if (discountType == DiscountType.OPTION) {// N元选：给商品包贴标签
								labels.add(buildOptionLabel(campaignResult));
							}
						}
					} else {
						if (productScope == ProductScope.SINGLE && discountType == DiscountType.GIFT) {// 单品活动
																										// 赠品活动
							labels.addAll(buildSingleGiftLabel(campaignResult, accessWay));
						}
					}
				}

			} // for
		}
		return labels;
	}

	@Override
	public List<Label> getPostageLabel(BigDecimal price, boolean hasOverSea, AccessWayEnum accessWay) {
		final List<Label> labels = Lists.newArrayList();// 邮费的标签
		if (price != null && !hasOverSea) {
			final Label label = new Label();
			label.setMode(LabelMode.POSTAGE);
			label.setType("免邮");
			label.setSummary("满" + PostageConstant.MEET_PRICE + "元免邮");
			label.setPostage(getPostage(price));
			labels.add(label);
		}
		return labels;
	}

}
