package com.thebeastshop.pegasus.channelservice.service.impl;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.thebeastshop.cart.ProductPack;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.coupon.enums.CpCouponTypeEnum;
import com.thebeastshop.coupon.vo.CpCouponSampleVO;
import com.thebeastshop.coupon.vo.CpFullCutPriceVO;
import com.thebeastshop.pegasus.channelservice.service.CouponSampleVerifyService;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.support.YesOrNo;

/**
 * Created by GFL on 2017/5/24.
 */
@Service
public class CouponSampleVerifyServiceImpl implements CouponSampleVerifyService {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public boolean verifySampleUsableForProducts (Collection<? extends ProductPack> sourcePacks, CpCouponSampleVO couponSample, AccessWayEnum accessWay) {

        //1. 优惠券模板不能为空
        if(null == couponSample || CollectionUtils.isEmpty(sourcePacks)){
            return false;
        }

        //2.校验优惠券支持平台
        //Collection<AccessWayEnum> accessWays = couponSample.getAccessWays();
        Collection<Integer> accessWays = couponSample.getAccessWayIds();
        if (CollectionUtils.isNotEmpty(accessWays)) {
            if (!accessWays.contains(accessWay.getCode())) {
                return false;
            }
        }


        //3.校验是否满足优惠券使用限制
        // // 满减优惠券类型
        if (couponSample.getCouponTypeId().equals(CpCouponTypeEnum.CHEAPEN_LINE.getId())) {

            CpFullCutPriceVO fullCutCoupon = (CpFullCutPriceVO) couponSample.getCouponContent();
            //FullCutCouponImpl fullCutCoupon = new FullCutCouponImpl(null, couponSample);

            // 判断商品总额是否满足优惠券的满额
            BigDecimal sumCouponPrice = sumCouponPrice(sourcePacks, couponSample);
            if (sumCouponPrice.equals(BigDecimal.ZERO)) {
                // 无满足优惠券的商品情况，优惠券不可用
                return false;
            } else {
                // 无门槛的优惠券直接可用，或者总金额满足优惠券门槛则可用
                if (fullCutCoupon.getFullPrice().equals(BigDecimal.ZERO)
                        || sumCouponPrice.compareTo(fullCutCoupon.getFullPrice()) >= 0) {
                    return true;
                }
            }
        }
        return false;
    }


    @Override
    public Map<Long, CpCouponSampleVO> getUsableSampleForProducts(Collection<? extends ProductPack> sourcePacks, List<CpCouponSampleVO> couponSamples, AccessWayEnum accessWay) {
        Map<Long,CpCouponSampleVO> resultMap = new HashMap<>();
        for(CpCouponSampleVO couponSample : couponSamples){
            if(this.verifySampleUsableForProducts(sourcePacks,couponSample,accessWay)){
                resultMap.put(couponSample.getId(),couponSample);
            }
        }
        return resultMap;
    }



    @Override
    public Map<Long, CpCouponSampleVO> getUsableSampleDefaultForProducts(Collection<? extends ProductPack> sourcePacks, List<CpCouponSampleVO> couponSamples, AccessWayEnum accessWay) {
        Map<Long,CpCouponSampleVO> resultMap = new HashMap<>();
        for(CpCouponSampleVO couponSample : couponSamples){
            if(this.verifySampleUsableDefaultForProducts(sourcePacks,couponSample,accessWay)){
                resultMap.put(couponSample.getId(),couponSample);
            }
        }
        return resultMap;
    }


    /**
     * 校验模板是否可作为指定商品的默认优惠券模板
     * @param sourcePacks
     * @param couponSample
     * @param accessWay
     * @return
     */
    private boolean verifySampleUsableDefaultForProducts (Collection<? extends ProductPack> sourcePacks, CpCouponSampleVO couponSample, AccessWayEnum accessWay) {

        //1. 优惠券模板不能为空
        if(null == couponSample || CollectionUtils.isEmpty(sourcePacks)){
            return false;
        }

        //2.校验优惠券支持平台
        //Collection<AccessWayEnum> accessWays = couponSample.getAccessWays();
        Collection<Integer> accessWays = couponSample.getAccessWayIds();
        if (CollectionUtils.isNotEmpty(accessWays)) {
            if (!accessWays.contains(accessWay.getCode())) {
                return false;
            }
        }


        //3.校验是否满足优惠券使用限制
        // // 满减优惠券类型
        if (couponSample.getCouponTypeId().equals(CpCouponTypeEnum.CHEAPEN_LINE.getId())) {

            CpFullCutPriceVO fullCutCoupon = (CpFullCutPriceVO) couponSample.getCouponContent();
            //FullCutCouponImpl fullCutCoupon = new FullCutCouponImpl(null, couponSample);

            // 判断商品总额是否满足优惠券的满额
            BigDecimal sumCouponPrice = sumCouponPrice(sourcePacks, couponSample);
            if (sumCouponPrice.equals(BigDecimal.ZERO)) {
                // 无满足优惠券的商品情况，优惠券不可用
                return false;
            } else {
                // 无门槛的优惠券直接可用，或者总金额满足优惠券门槛则可用
                if ((fullCutCoupon.getFullPrice().equals(BigDecimal.ZERO)
                        || sumCouponPrice.compareTo(fullCutCoupon.getFullPrice()) >= 0) && fullCutCoupon.getCutPrice().compareTo(sumCouponPrice)<=0) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 校验模板对于商品月送属性支持
     * @param product
     * @param couponSample
     * @return
     */
    private boolean checkSampleWeekFlower(PsProductVO product, CpCouponSampleVO couponSample){
        boolean flag = false;
        //获取商品是否月送
        boolean isWeekFlag = false;
        if(YesOrNo.YES.getId().equals(product.getIsMonthSend())){
            isWeekFlag = true;
        }
        // 月送鲜花 0:不限制 1:周送 2:非周送
        Integer weekflag = couponSample.getWeekFlower() == null ? 2 : couponSample.getWeekFlower();
        switch (weekflag){
            case 0:
                flag = true;
                break;
            case 1:
                flag= isWeekFlag;
                break;
            case 2:
                flag= isWeekFlag?false:true;
                break;
        }
        return flag;
    }

    /**
     * 校验优惠券模板海淘支持
     * @param product
     * @param couponSample
     * @return
     */
    private boolean checkSampleCroosBorder(PsProductVO product, CpCouponSampleVO couponSample){
        boolean flag = false;
        boolean isCrossflag = false;
        //获取商品包的月送和海淘类别属性
        if(YesOrNo.YES.getId().equals(product.getCrossBorderFlag())){
            isCrossflag = true;
        }

        /* ********************************
         * 获取优惠券商品类别支持属性
         */
        // 海外购 0:不限制 1:海淘 2:非海淘
        Integer crossflag = couponSample.getCrossBorder() == null ? 0 : couponSample.getCrossBorder();
        switch (crossflag){
            case 0:
                flag = true;
                break;
            case 1:
                flag= isCrossflag;
                break;
            case 2:
                flag= isCrossflag?false:true;
                break;
        }
        return flag;
    }


    // =================private================
    private BigDecimal sumCouponPrice(Collection<? extends ProductPack> sourcePacks, CpCouponSampleVO couponSample) {
        BigDecimal sumPrice = BigDecimal.ZERO;
        for (ProductPack pack : sourcePacks) {
            PsProductVO product = pack.getProduct();
            // 判断满足商品是否满足优惠券的条件
            if (checkProductInCouponSample(product, couponSample) && checkSampleCroosBorder(product,couponSample) && checkSampleWeekFlower(product,couponSample)) {
                sumPrice = sumPrice.add(pack.getPrice());
            }
        }
        return sumPrice;
    }
    // ==============华丽丽的分隔符================

    private Boolean checkProductInCouponSample(PsProductVO product, CpCouponSampleVO couponSample) {
        logger.info("product.getId()=" + product.getId());
        // 判断是否优惠券模板全场优惠
        if (couponSample.isForAllProducts() && (CollectionUtils.isEmpty(couponSample.getExcludeProductIds())
                || !couponSample.getExcludeProductIds().contains(Long.valueOf(product.getId())))) {
            return true;
        }
        // 判断是否在优惠券模板的商品列表中
        if (CollectionUtils.isNotEmpty(couponSample.getProductIds())
                && couponSample.getProductIds().contains(Long.valueOf(product.getId()))) {
            return true;
        }
        // 判断是否在优惠券模板的商品类目列表中
        if (CollectionUtils.isNotEmpty(couponSample.getCategoryIds())) {
            if (product.getCategoryId() != null && couponSample.getCategoryIds().contains(product.getCategoryId())) {
                return true;
            }
        }
        return false;
    }

}
