package com.thebeastshop.pegasus.channelservice.enums;

import java.util.Collections;
import java.util.List;

import com.google.common.collect.Lists;
import com.thebeastshop.support.exception.WrongStateException;
import com.thebeastshop.support.mark.State;

/**
 * 订单流程状态(后端订单状态)。
 *
 * @author Liang Wenjian
 */
public enum OrderFlowState implements State {

	CANCELLED(0, "已取消") {
		@Override
		public OrderFlowState cancel() {
			return this;
		}
	},
	CONFIRMED(1, "未付款") {
		@Override
		public OrderFlowState confirm() {
			return this;
		}

		@Override
		public OrderFlowState pay() {
			return PAID;
		}

		@Override
		public OrderFlowState payTimeout() {
			return CANCELLED;
		}

		@Override
		public OrderFlowState customerHangUp() {
			return CUSTOMER_HUNG_UP;
		}

		@Override
		public OrderFlowState cancel() {
			return CANCELLED;
		}
	},
	CUSTOMER_HUNG_UP(2, "未付款挂起"),
	/**
	 * 这个状态是第一个状态
	 */
	CREATED(3, "新建待确认") {
		@Override
		public OrderFlowState confirm() {
			return CONFIRMED;
		}

		@Override
		public OrderFlowState cancel() {
			return CANCELLED;
		}
	},
	CHANGE_AFTER_RECEIVE(5, "换货待收货完成"), AUDIT_HUNG_UP(9, "审单挂起"), PAID(10,
			"待审单"), AUDITED(11, "已审单"), DONE(12, "已完成"), ;

	/**
	 * 已确认的订单状态组
	 */
	public static final List<OrderFlowState> IS_CONFIRMED_ARRAYS = Collections
			.unmodifiableList(Lists.newArrayList(CONFIRMED, CUSTOMER_HUNG_UP));

	public static boolean isConfirmed(int id) {
		for (OrderFlowState state : IS_CONFIRMED_ARRAYS) {
			if (state.getId().intValue() == id)
				return true;
		}
		return false;
	}

	/**
	 * 已支付的订单状态组
	 */
	public static final List<OrderFlowState> IS_PAID_ARRAYS = Collections
			.unmodifiableList(Lists.newArrayList(AUDIT_HUNG_UP,PAID, AUDITED, DONE));

	public static boolean isPaid(int id) {
		for (OrderFlowState state : IS_PAID_ARRAYS) {
			if (state.getId().intValue() == id)
				return true;
		}
		return false;
	}

	/**
	 * 已取消的订单状态组
	 */
	public static final List<OrderFlowState> IS_CANCELLED_ARRAYS = Collections
			.unmodifiableList(Lists.newArrayList(CANCELLED));

	public static boolean isCanclled(int id) {
		for (OrderFlowState state : IS_CANCELLED_ARRAYS) {
			if (state.getId().intValue() == id)
				return true;
		}
		return false;
	}

	public final int id;
	public final String name;

	OrderFlowState(final int id, final String name) {
		this.id = id;
		this.name = name;
	}

	@Override
	public Integer getId() {
		return id;
	}

	@Override
	public String getName() {
		return name;
	}

	public OrderFlowState confirm() {
		throw new WrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 取消
	 *
	 * @return
	 */
	public OrderFlowState cancel() {
		throw new WrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 超时未支付
	 *
	 * @return
	 */
	public OrderFlowState payTimeout() {
		throw new WrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 客户挂起
	 *
	 * @return
	 */
	public OrderFlowState customerHangUp() {
		throw new WrongStateException("订单状态不对。", "state", this);
	}

	/**
	 * 支付
	 *
	 * @return
	 */
	public OrderFlowState pay() {
		throw new WrongStateException("订单状态不对。", "state", this);
	}
}
