package com.thebeastshop.pegasus.channelservice.controller.idcard;

import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.alibaba.fastjson.JSONObject;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.member.dto.UpdateIdCardImgDTO;
import com.thebeastshop.member.enums.IdCardImgStatusEnum;
import com.thebeastshop.member.service.IdCardQueryService;
import com.thebeastshop.member.service.IdCardService;
import com.thebeastshop.member.vo.MemIdCardVO;
import com.thebeastshop.pegasus.channelservice.ApiAuth;
import com.thebeastshop.pegasus.channelservice.JsonApiResult;
import com.thebeastshop.pegasus.channelservice.constants.ErrorCode;
import com.thebeastshop.pegasus.channelservice.util.IdentityUtil;
import com.thebeastshop.pegasus.channelservice.vo.IdCardVO;
import com.thebeastshop.pegasus.service.operation.PegasusChannelServiceFacade;
import com.thebeastshop.pegasus.service.operation.param.UploadPictureParam;
import com.thebeastshop.pegasus.service.operation.vo.OpSaleOrderIdentityVO;

@Controller("IdCardController")
@RequestMapping("/api/idcard")
public class IdCardController {
	private Logger logger = LoggerFactory.getLogger(IdCardController.class);
	private PegasusChannelServiceFacade instance = PegasusChannelServiceFacade.getInstance();

	@Autowired
	private IdCardQueryService idCardQueryService;

	@Autowired
	private IdCardService idCardService;

	/**
	 * 查询身份证信息
	 * 
	 * @param data
	 *            格式{"userCode":"xxxx","name":"xxxx"}
	 * @return
	 * @author lijing 2016 05 05
	 */
	@ApiAuth
	@RequestMapping(value = "/searchIdCard")
	@ResponseBody
	public JsonApiResult searchIdCard(@RequestParam("DATA") String data) {
		logger.info("查询身份证参数  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String memberCode = jsonData.getString("memberCode");
			String name = jsonData.getString("name");
			if (StringUtils.isBlank(memberCode) || StringUtils.isBlank(name)) {
				logger.info("必要查询参数没有传入");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			MemIdCardVO memIdCardVO = idCardQueryService.searchIdCardByParam(memberCode, name);
			logger.info("查询身份证信息结果为  opIdCardVO={} ", memIdCardVO);
			if (memIdCardVO == null) {
				memIdCardVO = new MemIdCardVO();
			}
			jsonApiResult = JsonApiResult.ofSuccessResult(memIdCardVO);
			return jsonApiResult;
		} catch (RuntimeException e) {
			logger.error("参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
		} catch (Exception e) {
			logger.error("系统异常 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 验证身份证信息
	 * 
	 * @param data
	 *            格式{"userCode":"xxxx","name":"xxxx","idNumber":"xxx"}
	 * @return
	 * @author lijing 2016 05 05
	 */
	@ApiAuth
	@RequestMapping(value = "/validIdCard")
	@ResponseBody
	public JsonApiResult validIdCard(@RequestParam("DATA") String data) {
		logger.info("验证身份证参数  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String memberCode = jsonData.getString("memberCode");
			String name = jsonData.getString("name");
			String idNumber = jsonData.getString("idNumber");
			if (StringUtils.isBlank(memberCode) || StringUtils.isBlank(name) || StringUtils.isBlank(idNumber)) {
				logger.info("必要验证参数没有传入");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			if (!IdentityUtil.checkIdentity(idNumber)) {
				logger.info("非法的身份证号格式");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "非法的身份证号格式");
			}
			ServiceResp<Boolean> result = idCardService.verifyIdCard(memberCode, name, idNumber);
			if (result != null && result.isSuccess() && result.getBean()) {
				jsonApiResult = JsonApiResult.ofSuccessResult(true);
			} else {
				jsonApiResult = JsonApiResult.ofErrorResult(-100, "收件人姓名与身份证号码不匹配");
			}
			return jsonApiResult;
		} catch (RuntimeException e) {
			logger.error("参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), e.getMessage());
		} catch (Exception e) {
			logger.error("系统异常 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 上传身份证图片
	 * 
	 * @param data
	 *            格式{"userCode":"xxxx","name":"xxxx","idNumber":"xxx",
	 *            "orderNumber":123213,"frontCardPhoto":"xxxx","backCardPhoto":
	 *            "xxxxx"}
	 * @return
	 * @author lijing 2016 05 05
	 */
	@ApiAuth
	@RequestMapping(value = "/uploadIdCardPicture")
	@ResponseBody
	public JsonApiResult uploadIdCardPicture(@RequestParam("DATA") String data) {
		logger.info("上传身份证图片  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String memberCode = jsonData.getString("memberCode");
			String frontCardPhoto = jsonData.getString("frontCardPhoto");
			String backCardPhoto = jsonData.getString("backCardPhoto");
			String idNumber = jsonData.getString("idNumber");
			String name = jsonData.getString("name");
			if (StringUtils.isBlank(memberCode) || StringUtils.isBlank(name) || StringUtils.isBlank(idNumber)
					|| StringUtils.isBlank(frontCardPhoto) || StringUtils.isBlank(backCardPhoto)) {
				logger.info("必要验证参数没有传入");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			List<OpSaleOrderIdentityVO> opSaleOrderIdentityVOList = instance.findOrderIdentityByMember(memberCode, name,
					idNumber);
			if (CollectionUtils.isNotEmpty(opSaleOrderIdentityVOList)) {

				if (opSaleOrderIdentityVOList.get(0).getAuditStatus() == 1
						|| opSaleOrderIdentityVOList.get(0).getAuditStatus() == 2) {
					logger.warn("图片不可以上传 集合为空 或则 状态是待审核  或则 审核通过状态");
					return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "照片已上传");
				}
			}

			UploadPictureParam uploadPictureParam = new UploadPictureParam();
			uploadPictureParam.setIdNumber(idNumber);
			uploadPictureParam.setName(name);
			uploadPictureParam.setBackCardPhoto(backCardPhoto);
			uploadPictureParam.setFrontCardPhoto(frontCardPhoto);
			uploadPictureParam.setUserCode(memberCode);
			Boolean result = instance.updateSalesOrderIdentityAuditStatus(uploadPictureParam);
			if (result) {
				jsonApiResult = JsonApiResult.ofSuccessResult(result);
			} else {
				jsonApiResult = JsonApiResult.ofErrorResult(-100, "图片上传错误");
			}
			return jsonApiResult;
		} catch (RuntimeException e) {
			logger.error("参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), e.getMessage());
		} catch (Exception e) {
			logger.error("系统异常 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 通过订单编号查询
	 * 
	 * @param data
	 *            格式{"orderNumber":"123213"}
	 * @return
	 * @author lijing 2016 05 05
	 */
	@ApiAuth
	@RequestMapping(value = "/searchOrderIdentByOrderNo")
	@ResponseBody
	public JsonApiResult searchOrderIdentByOrderNo(@RequestParam("DATA") String data) {
		logger.info("通过订单编号查询  data={}", data);
		try {

			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String orderCode = jsonData.getString("orderNumber");
			OpSaleOrderIdentityVO opSaleOrderIdentityVO = instance.getOpSaleOrderIdenByOrderNo(orderCode);
			if (opSaleOrderIdentityVO != null) {
				IdCardVO vo = new IdCardVO();
				vo.setAuditStatus(opSaleOrderIdentityVO.getAuditStatus());
				vo.setBackCardPhoto(opSaleOrderIdentityVO.getImgBack());
				vo.setFrontCardPhoto(opSaleOrderIdentityVO.getImgFront());
				vo.setId(opSaleOrderIdentityVO.getId());
				vo.setIdNumber(opSaleOrderIdentityVO.getIdentitytId());
				vo.setName(opSaleOrderIdentityVO.getName());
				vo.setOrderId(opSaleOrderIdentityVO.getSalesOrderId());
				return JsonApiResult.ofSuccessResult(vo);
			} else {
				return JsonApiResult.ofSuccessResult(null);
			}

		} catch (Exception e) {
			logger.error("系统异常 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 修改身份证图片 和身份证图片状态
	 * 
	 * @param data
	 *            格式{"userCode":"xxxx","name":"xxxx","idNumber":"xxx",
	 *            "orderNumber":123213,"frontCardPhoto":"xxxx","backCardPhoto":
	 *            "xxxxx"}
	 * @return
	 * @author lijing 2016 06 23
	 */
	@ApiAuth
	@RequestMapping(value = "/uploadImgAndImgStatus")
	@ResponseBody
	public JsonApiResult uploadIdCardPictureAndImgStatus(@RequestParam("DATA") String data) {
		logger.info("修改身份证图片 和身份证图片状态  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String memberCode = jsonData.getString("memberCode");
			String frontCardPhoto = jsonData.getString("frontCardPhoto");
			String backCardPhoto = jsonData.getString("backCardPhoto");
			String idNumber = jsonData.getString("idNumber");
			String name = jsonData.getString("name");
			if (StringUtils.isBlank(memberCode) || StringUtils.isBlank(name) || StringUtils.isBlank(idNumber)
					|| StringUtils.isBlank(frontCardPhoto) || StringUtils.isBlank(backCardPhoto)) {
				logger.info("必要验证参数没有传入");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			List<OpSaleOrderIdentityVO> opSaleOrderIdentityVOList = instance.findOrderIdentityByMember(memberCode, name,
					idNumber);
			if (CollectionUtils.isNotEmpty(opSaleOrderIdentityVOList)) {

				if (opSaleOrderIdentityVOList.get(0).getAuditStatus() == 1
						|| opSaleOrderIdentityVOList.get(0).getAuditStatus() == 2) {
					logger.warn("图片不可以上传 集合为空 或则 状态是待审核  或则 审核通过状态");
					return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "照片已上传");
				}
			}

			MemIdCardVO memIdCardVO = idCardQueryService.getIdCardByIdNumberAndNameAndMemberCode(idNumber, name,
					memberCode);
			if (null == memIdCardVO || null == memIdCardVO.getStatus() || memIdCardVO.getStatus() != 1
					|| memIdCardVO.getImgStatus() == 2) {
				logger.warn("身份证修改图片失败，状态或则  身份证错误！opIdCardVO={}", memIdCardVO);
				jsonApiResult = JsonApiResult.ofErrorResult(-200, "图片不可修改，身份证状态不对");
			}

			UpdateIdCardImgDTO dto = new UpdateIdCardImgDTO();
			dto.setIdNumber(idNumber);
			dto.setName(name);
			dto.setBackCardPhoto(backCardPhoto);
			dto.setFrontCardPhoto(frontCardPhoto);
			dto.setMemberCode(memberCode);
			dto.setImgStatus(IdCardImgStatusEnum.UNAUDITED.getCode());
			ServiceResp<Boolean> resp = idCardService.updateIdCardImgAndImgStatus(dto);
			if (resp != null && resp.isSuccess() && resp.getBean()) {
				UploadPictureParam uploadPictureParam = new UploadPictureParam();
				uploadPictureParam.setIdNumber(idNumber);
				uploadPictureParam.setName(name);
				uploadPictureParam.setBackCardPhoto(backCardPhoto);
				uploadPictureParam.setFrontCardPhoto(frontCardPhoto);
				uploadPictureParam.setUserCode(memberCode);
				instance.updateSalesOrderIdentityAuditStatus(uploadPictureParam);
				jsonApiResult = JsonApiResult.ofSuccessResult(true);
			} else {
				jsonApiResult = JsonApiResult.ofErrorResult(-100, "图片上传错误");
			}
			return jsonApiResult;
		} catch (RuntimeException e) {
			logger.error("修改身份证图片 和身份证图片状态 参数错误  或则 数据库 修改错误  identity表没有修改  e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), e.getMessage());
		} catch (Exception e) {
			logger.error("修改身份证图片 和身份证图片状态 参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 查询身份证通过id
	 * 
	 * @param data
	 *            格式{"id":xxxx}
	 * @return
	 * @author lijing 2016 06 23
	 */
	@ApiAuth
	@RequestMapping(value = "/detail")
	@ResponseBody
	public JsonApiResult detail(@RequestParam("DATA") String data) {
		logger.info("查询身份证通过id  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			Integer id = jsonData.getInteger("id");

			if (null == id) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			MemIdCardVO memIdCardVO = idCardQueryService.getIdCardById(id);
			if (null == memIdCardVO || memIdCardVO.getId() == null) {
				jsonApiResult = JsonApiResult.ofSuccessResult(null);
			} else {
				jsonApiResult = JsonApiResult.ofSuccessResult(memIdCardVO);
			}
			return jsonApiResult;
		} catch (Exception e) {
			logger.error("查询身份证通过 参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 验证身份证信息 成功返回 idCardVo对象
	 * 
	 * @param data
	 *            格式{"userCode":"xxxx","name":"xxxx","idNumber":"xxx"}
	 * @return
	 * @author lijing 2016 06 24
	 */
	@ApiAuth
	@RequestMapping(value = "/validIdCardAndReturnId")
	@ResponseBody
	public JsonApiResult validIdCardAndReturnId(@RequestParam("DATA") String data) {
		logger.info("验证身份证参数  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String memberCode = jsonData.getString("memberCode");
			String name = jsonData.getString("name");
			String idNumber = jsonData.getString("idNumber");
			if (StringUtils.isBlank(memberCode) || StringUtils.isBlank(name) || StringUtils.isBlank(idNumber)) {
				logger.info("必要验证参数没有传入");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			if (!IdentityUtil.checkIdentity(idNumber)) {
				logger.info("非法的身份证号格式");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "非法的身份证号格式");
			}

			ServiceResp<Boolean> result = idCardService.verifyIdCard(memberCode, name, idNumber);
			if (result != null && result.isSuccess() && result.getBean()) {
				MemIdCardVO memIdCardVO = idCardQueryService.getIdCardByIdNumberAndNameAndMemberCode(idNumber, name,
						memberCode);
				jsonApiResult = JsonApiResult.ofSuccessResult(memIdCardVO);
			} else {
				jsonApiResult = JsonApiResult.ofErrorResult(-100, "收件人姓名与身份证号码不匹配");
			}
			return jsonApiResult;
		} catch (RuntimeException e) {
			logger.error("参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), e.getMessage());
		} catch (Exception e) {
			logger.error("系统异常 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}

	/**
	 * 先验证身份，再上传图片
	 * 
	 * @param data
	 *            格式{"userCode":"xxxx","name":"xxxx","idNumber":"xxx",
	 *            "orderNumber":123213,"frontCardPhoto":"xxxx","backCardPhoto":
	 *            "xxxxx"}
	 * @return
	 */
	@ApiAuth
	@RequestMapping(value = "/validAndUploadImg")
	@ResponseBody
	public JsonApiResult validAndUploadImg(@RequestParam("DATA") String data) {
		logger.info("修改身份证图片 和身份证图片状态  data={}", data);
		try {
			JsonApiResult jsonApiResult;
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			JSONObject jsonData = JSONObject.parseObject(data);
			String memberCode = jsonData.getString("memberCode");
			String frontCardPhoto = jsonData.getString("frontCardPhoto");
			String backCardPhoto = jsonData.getString("backCardPhoto");
			String idNumber = jsonData.getString("idNumber");
			String name = jsonData.getString("name");

			logger.info("验证身份证参数  data={}", data);
			if (StringUtils.isBlank(data)) {
				logger.info("DATA 没有数据");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			if (StringUtils.isBlank(memberCode) || StringUtils.isBlank(name) || StringUtils.isBlank(idNumber)) {
				logger.info("必要验证参数没有传入");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "参数错误");
			}
			if (!IdentityUtil.checkIdentity(idNumber)) {
				logger.info("非法的身份证号格式");
				return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "非法的身份证号格式");
			}
			ServiceResp<Boolean> result = idCardService.verifyIdCard(memberCode, name, idNumber);
			if (result != null && result.isSuccess() && result.getBean()) {
				jsonApiResult = JsonApiResult.ofSuccessResult(true);
			} else {
				return JsonApiResult.ofErrorResult(-100, "姓名与身份证号码不匹配");
				
			}
			//if (StringUtils.isNotBlank(frontCardPhoto) && StringUtils.isNotBlank(backCardPhoto)) {

				List<OpSaleOrderIdentityVO> opSaleOrderIdentityVOList = instance.findOrderIdentityByMember(memberCode,
						name, idNumber);
				if (CollectionUtils.isNotEmpty(opSaleOrderIdentityVOList)) {

					if (opSaleOrderIdentityVOList.get(0).getAuditStatus() == 1
							|| opSaleOrderIdentityVOList.get(0).getAuditStatus() == 2) {
						logger.warn("图片不可以上传 集合为空 或则 状态是待审核  或则 审核通过状态");
						return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), "照片已上传");
					}
				}

				MemIdCardVO memIdCardVO = idCardQueryService.getIdCardByIdNumberAndNameAndMemberCode(idNumber, name,
						memberCode);
				if (null == memIdCardVO || null == memIdCardVO.getStatus() || memIdCardVO.getStatus() != 1
						|| memIdCardVO.getImgStatus() == 2) {
					logger.warn("身份证修改图片失败，状态或则  身份证错误！opIdCardVO={}", memIdCardVO);
					jsonApiResult = JsonApiResult.ofErrorResult(-200, "图片不可修改，身份证状态不对");
				}

				UpdateIdCardImgDTO dto = new UpdateIdCardImgDTO();
				dto.setIdNumber(idNumber);
				dto.setName(name);
				dto.setBackCardPhoto(backCardPhoto);
				dto.setFrontCardPhoto(frontCardPhoto);
				dto.setMemberCode(memberCode);
				dto.setImgStatus(IdCardImgStatusEnum.UNAUDITED.getCode());
				ServiceResp<Boolean> resp = idCardService.updateIdCardImgAndImgStatus(dto);
				if (resp != null && resp.isSuccess() && resp.getBean()) {
					UploadPictureParam uploadPictureParam = new UploadPictureParam();
					uploadPictureParam.setIdNumber(idNumber);
					uploadPictureParam.setName(name);
					uploadPictureParam.setBackCardPhoto(backCardPhoto);
					uploadPictureParam.setFrontCardPhoto(frontCardPhoto);
					uploadPictureParam.setUserCode(memberCode);
					instance.updateSalesOrderIdentityAuditStatus(uploadPictureParam);
					jsonApiResult = JsonApiResult.ofSuccessResult(true);
				} else {
					jsonApiResult = JsonApiResult.ofErrorResult(-100, "图片上传错误");
				}
			//}
			return jsonApiResult;
		} catch (RuntimeException e) {
			logger.error("修改身份证图片 和身份证图片状态 参数错误  或则 数据库 修改错误  identity表没有修改  e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(), e.getMessage());
		} catch (Exception e) {
			logger.error("修改身份证图片 和身份证图片状态 参数错误 e={}", e);
			return JsonApiResult.ofErrorResult(ErrorCode.DEFAULT_EXCEPTION.getCode(),
					ErrorCode.DEFAULT_EXCEPTION.getMesage());
		}
	}
}
