package com.thebeastshop.pegasus.channelservice.controller.exchange;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import com.thebeastshop.common.utils.NumberUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.thebeastshop.common.PageQueryResp;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.exchange.dto.ExchgSpuConfigQueryDTO;
import com.thebeastshop.exchange.enums.ExchgConfigStatusEnum;
import com.thebeastshop.exchange.enums.ExchgModeEnum;
import com.thebeastshop.exchange.enums.ExchgTypeEnum;
import com.thebeastshop.exchange.service.ExchangeService;
import com.thebeastshop.exchange.vo.ExchgRecordVO;
import com.thebeastshop.exchange.vo.ExchgSpuConfigVO;
import com.thebeastshop.exchange.vo.MemberExchgVO;
import com.thebeastshop.member.service.MemberQueryService;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.payment.dto.PIntegralPaymentDTO;
import com.thebeastshop.payment.enums.PTransTypeEnum;
import com.thebeastshop.payment.service.PPaymentService;
import com.thebeastshop.payment.vo.PPaymentTradeVO;
import com.thebeastshop.pegasus.channelservice.ApiAuth;
import com.thebeastshop.pegasus.channelservice.JsonApiResult;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeCouponGoodsAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeGroupCouponGoodsAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeProductGoodsAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeProductGoodsDetailAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeThirdCouponGoodsAdapter;
import com.thebeastshop.pegasus.channelservice.exception.ChannelException;
import com.thebeastshop.pegasus.merchandise.service.McOpChannelService;
import com.thebeastshop.pegasus.merchandise.vo.OpChannelVO;
import com.thebeastshop.support.page.Page;
import com.thebeastshop.support.page.Pagination;
import com.thebeastshop.support.vo.exchange.ExchangeGoods;
import com.thebeastshop.support.vo.exchange.ExchangeGroupVO;
import com.thebeastshop.support.vo.exchange.ExchangeProductGoods;
import com.thebeastshop.support.vo.exchange.ExchangeSimpleProductGoods;

/**
 * @author feilong.gao
 * @Date 2018/5/23 14:16.
 * @Description 兑换服务
 */
@RestController
@RequestMapping("api/exchange")
public class ExchangeController {

    Logger logger = LoggerFactory.getLogger(ExchangeController.class);

    @Autowired
    ExchangeService exchangeService;


    @Autowired
    ExchangeAdapter exchangeAdapter;

    @Autowired
    ExchangeProductGoodsAdapter exchangeGoodsAdapter;

    @Autowired
    ExchangeCouponGoodsAdapter exchangeCouponGoodsAdapter;

    @Autowired
    ExchangeThirdCouponGoodsAdapter exchangeThirdCouponGoodsAdapter;

    @Autowired
    ExchangeGroupCouponGoodsAdapter exchangeGroupCouponGoodsAdapter;

    @Autowired
    ExchangeProductGoodsDetailAdapter exchangeProductGoodsDetailAdapter;

    @Autowired
    PPaymentService pPaymentService;

    @Autowired
    MemberQueryService memberQueryService;

    @Autowired
    private McOpChannelService mcOpChannelService;

    /**
     * 查询兑换商品列表
     *
     * @param code
     * @param data
     * @return
     */
    @ApiAuth
    @RequestMapping("/goods/list")
    public JsonApiResult getGoodsList(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {

        logger.info("getGoodsList 入参：code={} data={}");
        ExchangeGroupVO result = new ExchangeGroupVO();
        JSONObject jsonData = JSONObject.parseObject(data);
        Integer accessWayId = jsonData.getInteger("accessWay");
        List<ExchgConfigStatusEnum> statusEnums = Lists.newArrayList(ExchgConfigStatusEnum.ONLINE, ExchgConfigStatusEnum.WAITING_ONLINE);
        String modeStr = jsonData.getString("mode");
        ExchgModeEnum exchgModeEnum = null;
        if (ExchgModeEnum.POINT_PLUS_PRICE.name().equalsIgnoreCase(modeStr)) {
            exchgModeEnum = ExchgModeEnum.POINT_PLUS_PRICE;
        }else if (ExchgModeEnum.POINT.name().equalsIgnoreCase(modeStr)) {
            exchgModeEnum = ExchgModeEnum.POINT;
        }

        ExchgSpuConfigQueryDTO dto = new ExchgSpuConfigQueryDTO();
        dto.setExchangeType(null);
        dto.setExchgMode(exchgModeEnum);
        dto.setStatusList(statusEnums);
        dto.setQueryValid(Boolean.FALSE);      // 上线状态过滤当前时间
        dto.setCurrpage(1);
        dto.setPagenum(Integer.MAX_VALUE);
        PageQueryResp<ExchgSpuConfigVO> pageResp = exchangeService.queryExchgSpuConfig(dto);
        List<ExchgSpuConfigVO> productSimples = Lists.newArrayList();
        List<ExchgSpuConfigVO> couponSimples = Lists.newArrayList();
        List<ExchgSpuConfigVO> thirdCouponSimples = Lists.newArrayList();
        List<ExchgSpuConfigVO> groupCouponSimples = Lists.newArrayList();
        if (null != pageResp && CollectionUtils.isNotEmpty(pageResp.getBeanList())) {
            for (ExchgSpuConfigVO spuConfigSimple : pageResp.getBeanList()) {
                switch (spuConfigSimple.getExchangeType()) {
                    case SPU:
                        productSimples.add(spuConfigSimple);
                        break;
                    case THIRD_COUPON:
                        thirdCouponSimples.add(spuConfigSimple);
                        break;
                    case COUPON:
                        couponSimples.add(spuConfigSimple);
                        break;
                    case GROUP_COUPON:
                        groupCouponSimples.add(spuConfigSimple);
                        break;
                }
            }
        }

        // 设置兑换商品
        if (CollectionUtils.isNotEmpty(productSimples)) {
            List<ExchangeSimpleProductGoods> exchangeSimpleProductGoods = exchangeGoodsAdapter.getExchangeSimpleProduct(productSimples, code, accessWayId);
            exchangeGoodsAdapter.buildExchageSupGoodsSetProductAndSecKillList(result, exchangeSimpleProductGoods);
        }

        List<ExchangeGoods> couponGoods = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(couponSimples)) {
            couponGoods.addAll(exchangeCouponGoodsAdapter.getExchangeCoupon(couponSimples));
        }
        if (CollectionUtils.isNotEmpty(thirdCouponSimples)) {
            couponGoods.addAll(exchangeThirdCouponGoodsAdapter.getExchangeThirdCoupon(thirdCouponSimples));
        }
        if (CollectionUtils.isNotEmpty(groupCouponSimples)) {
            couponGoods.addAll(exchangeGroupCouponGoodsAdapter.getGroupCouponExchangeGoods(groupCouponSimples));
        }
        result.setCoupons(couponGoods);

        return JsonApiResult.ofSuccessResult(result);

    }

    /**
     * 查询兑换商品列表
     *
     * @param code
     * @param data
     * @return
     */
    @ApiAuth
    @RequestMapping("/goods/page")
    public JsonApiResult getPointPlusPriceGoodsByPage(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {

        logger.info("getGoodsList 入参：code={} data={}");
        ExchangeGroupVO result = new ExchangeGroupVO();
        JSONObject jsonData = JSONObject.parseObject(data);
        Integer accessWayId = jsonData.getInteger("accessWay");
        Integer offset = jsonData.getInteger("offset");
        Integer limit = jsonData.getInteger("limit");
        Integer minPoint = jsonData.getInteger("minPoint");
        Integer maxPoint = jsonData.getInteger("maxPoint");
        String modeStr = jsonData.getString("mode");
        ExchgModeEnum exchgModeEnum = null;
        if (ExchgModeEnum.POINT_PLUS_PRICE.name().equalsIgnoreCase(modeStr)) {
            exchgModeEnum = ExchgModeEnum.POINT_PLUS_PRICE;
        }else if (ExchgModeEnum.POINT.name().equalsIgnoreCase(modeStr)) {
            exchgModeEnum = ExchgModeEnum.POINT;
        }

        Page<ExchangeSimpleProductGoods> rtPage = new Page<ExchangeSimpleProductGoods>(
                new Pagination(offset, limit));

        List<ExchgConfigStatusEnum> statusEnums = Lists.newArrayList(ExchgConfigStatusEnum.ONLINE, ExchgConfigStatusEnum.WAITING_ONLINE);

        ExchgSpuConfigQueryDTO dto = new ExchgSpuConfigQueryDTO();
        dto.setExchangeType(ExchgTypeEnum.SPU);
        dto.setExchgMode(exchgModeEnum);
        dto.setStatusList(statusEnums);
        dto.setQueryValid(Boolean.FALSE);      // 上线状态过滤当前时间
        if (!NumberUtil.isNullOrZero(minPoint)) {
            dto.setPointAreaStart(minPoint);
        }
        if (!NumberUtil.isNullOrZero(maxPoint)) {
            dto.setPointAreaEnd(maxPoint);
        }
        dto.setOffset(offset);
        dto.setPagenum(limit);
        PageQueryResp<ExchgSpuConfigVO> pageResp = exchangeService.queryExchgSpuConfig(dto);

        if (null != pageResp && CollectionUtils.isNotEmpty(pageResp.getBeanList())) {
            rtPage.setTotal(pageResp.getTotalCnt());
            List<ExchangeSimpleProductGoods> exchangeSimpleProductGoods = exchangeGoodsAdapter.getExchangeSimpleProduct(pageResp.getBeanList(), code, accessWayId);
            rtPage.setItems(exchangeSimpleProductGoods);
        } else {
            rtPage.setTotal(0);
        }
        return JsonApiResult.ofSuccessResult(rtPage);

    }


    @ApiAuth
    @RequestMapping("/goods/detail")
    public JsonApiResult getGoodsDetail(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {

        JSONObject jsonData = JSONObject.parseObject(data);
        String productCode = jsonData.getString("productCode");// product code
        Long memberId = jsonData.getLong("memberId");
        MemberVO member = memberQueryService.getById(memberId);
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(jsonData.getInteger("accessWay"));
        OpChannelVO opChannel = mcOpChannelService.findByCode(code);

        ExchgSpuConfigVO exchgSpuConfig = exchangeService.queryExchgSpuConfigByReferenceCode(productCode);
        ExchangeProductGoods productGoods = exchangeProductGoodsDetailAdapter.getExchangeProductDetail(exchgSpuConfig, member, opChannel, accessWay);
        return JsonApiResult.ofSuccessResult(productGoods);
    }


    @ApiAuth
    @RequestMapping("/detail")
    public JsonApiResult getExchangeDetail(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {

        JSONObject jsonData = JSONObject.parseObject(data);
        String referenceCode = jsonData.getString("referenceCode");// product code
        Long memberId = jsonData.getLong("memberId");
        MemberVO member = memberQueryService.getById(memberId);
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(jsonData.getInteger("accessWay"));
        OpChannelVO opChannel = mcOpChannelService.findByCode(code);

        ExchgSpuConfigVO exchgSpuConfig = exchangeService.queryExchgSpuConfigByReferenceCode(referenceCode);
        if (exchgSpuConfig == null) return JsonApiResult.ofSuccessResult(null);
        List<ExchangeGoods> goods = new ArrayList<>();
        switch (exchgSpuConfig.getExchangeType()) {
            case SPU:
                goods.add(exchangeProductGoodsDetailAdapter.getExchangeProductDetail(exchgSpuConfig, member, opChannel, accessWay));
                break;
            case THIRD_COUPON:
                goods.addAll(exchangeThirdCouponGoodsAdapter.getExchangeThirdCoupon(Lists.newArrayList(exchgSpuConfig)));
                break;
            case COUPON:
                goods.addAll(exchangeCouponGoodsAdapter.getExchangeCoupon(Lists.newArrayList(exchgSpuConfig)));
                break;
            case GROUP_COUPON:
                goods.addAll(exchangeGroupCouponGoodsAdapter.getGroupCouponExchangeGoods(Lists.newArrayList(exchgSpuConfig)));
                break;
        }
        ExchangeGoods result = CollectionUtils.isEmpty(goods) ? null : goods.get(0);
        return JsonApiResult.ofSuccessResult(result);
    }


    @ApiAuth
    @RequestMapping("/coupon")
    public JsonApiResult exchangeCoupon(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        JSONObject jsonData = JSONObject.parseObject(data);
        String referenceCode = jsonData.getString("referenceCode");// product code
        Long memberId = jsonData.getLong("memberId");
        MemberVO member = memberQueryService.getById(memberId);
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(jsonData.getInteger("accessWay"));
        AccessWayEnum accessWayEnum = AccessWayEnum.getEnumByCode(accessWay.getCode());
        OpChannelVO opChannel = mcOpChannelService.findByCode(code);

        // 校验是否可以兑换
        MemberExchgVO memberExchg = exchangeAdapter.checkExchangeCouponGoods(member, referenceCode);
        // 保存兑换记录
        ExchgRecordVO initRecord = new ExchgRecordVO();
        initRecord.setMemberCode(member.getCode());
        initRecord.setMemberId(memberId.intValue());
        initRecord.setMemberNickName(member.getNickName());
        initRecord.setReferenceCode(memberExchg.getReferenceCode());
        initRecord.setReferenceName(memberExchg.getReferenceName());
        initRecord.setExchangeNum(1);
        initRecord.setExchangeAmount(memberExchg.getExchangeAmount());

        ServiceResp<String> initResp = exchangeService.initExchgRecord(initRecord);
        if (initResp.isSuccess()) {
            String tradeNo = initResp.getBean();
            // 支付积分
            PIntegralPaymentDTO paymentDTO = new PIntegralPaymentDTO();
            paymentDTO.setMemberId(memberId);
            paymentDTO.setAccessWay(accessWayEnum);
            paymentDTO.setChannelCode(opChannel.getCode());
            paymentDTO.setChannelName(opChannel.getName());
            paymentDTO.setOutTradeCode(tradeNo);
            paymentDTO.setPaymentAmount(new BigDecimal(memberExchg.getExchangeAmount()));
            paymentDTO.setTransType(PTransTypeEnum.EXCHANGE_PAY);
            ServiceResp<PPaymentTradeVO> payResp = pPaymentService.payForLocal(paymentDTO);
            if (payResp.isSuccess() && payResp.getBean()!=null) {
                // TODO 更改兑换状态
                exchangeService.processExchgRecordByExchgNo(tradeNo);
            } else {
                throw new ChannelException(4000, payResp.getRespMsg());
            }
        } else {
            throw new ChannelException(4000, initResp.getRespMsg());
        }

        return JsonApiResult.ofSuccessResult(Boolean.TRUE);
    }

}
