package com.thebeastshop.pegasus.channelservice.controller.cart;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.thebeastshop.cart.resp.CartProductPack;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.member.service.MemberQueryService;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.ApiAuth;
import com.thebeastshop.pegasus.channelservice.JsonApiResult;
import com.thebeastshop.pegasus.channelservice.adapter.exchange.ExchangeShopCartAdapter;
import com.thebeastshop.pegasus.channelservice.service.ExchangeCartService;
import com.thebeastshop.pegasus.merchandise.service.McOpChannelService;
import com.thebeastshop.pegasus.merchandise.vo.OpChannelVO;
import com.thebeastshop.support.vo.cart.CartProductPackVO;
import com.thebeastshop.support.vo.cart.CartVO;
import com.thebeastshop.support.vo.cart.RequestPack;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import java.util.Date;
import java.util.List;

/**
 * @author feilong.gao
 * @Date 2018/9/18 11:33.
 * @Description 兑换购物车
 */
@Controller
@RequestMapping("/api/exchangeCart")
@ResponseBody
public class ExchangeCartController {


    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private McOpChannelService mcOpChannelService;

    @Autowired
    private MemberQueryService memberQueryService;

    @Autowired
    ExchangeShopCartAdapter exchangeShopCartAdapter;

    @Autowired
    ExchangeCartService exchangeCartService;

    /**
     * 获取购物车
     *
     * @param code
     * @param data
     * @return
     */
    @ApiAuth
    @RequestMapping(value = "")
    public JsonApiResult getCart(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("获取积分兑换购物车开始");
        long startTimeMillis = System.currentTimeMillis();
        JSONObject jsonData = JSONObject.parseObject(data);
        Long memberId = jsonData.getLong("memberId");
        Integer accessWayId = jsonData.getInteger("accessWay");
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(accessWayId);
        MemberVO member = memberQueryService.getById(memberId);
        OpChannelVO opChannel = mcOpChannelService.findByCode(code);
        // 获取购物车记录
        List<RequestPack> requestPacks = exchangeCartService.getCart(member.getId());

        CartVO cart = exchangeShopCartAdapter.getCart(requestPacks, member, accessWay, opChannel);

        logger.info("获取积分兑换购物车耗时：{}毫秒", startTimeMillis - System.currentTimeMillis());
        return JsonApiResult.ofSuccessResult(cart);
    }


    @ApiAuth
    @RequestMapping(value = "/count")
    public JsonApiResult count(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("execute count");
        JSONObject jsonData = JSONObject.parseObject(data);
        Integer accessWayId = jsonData.getInteger("accessWay");
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(accessWayId);
        Long memberId = jsonData.getLong("memberId");
        MemberVO member = memberQueryService.getById(memberId);
        OpChannelVO channel = mcOpChannelService.findByCode(code);
        List<RequestPack> requestPacks = exchangeCartService.getCart(member.getId());
        List<CartProductPack> validPacks = exchangeShopCartAdapter.filterPackByByRequest(true, requestPacks, member, accessWay, channel);

        return JsonApiResult.ofSuccessResult(validPacks.size());
    }

    /**
     * 加入购物车
     *
     * @param code
     * @param data
     */
    @ApiAuth
    @RequestMapping(value = "/addPacks")
    public JsonApiResult addPack(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("execute addPacks");
        JSONObject jsonData = JSONObject.parseObject(data);
        Long memberId = jsonData.getLong("memberId");
        List<RequestPack> packs = JSON.parseArray(jsonData.getString("packs"), RequestPack.class);
        List<Long> packIds = exchangeCartService.addPacks(memberId, packs);
        return JsonApiResult.ofSuccessResult(packIds);
    }

    /**
     * 编辑购物车商品
     *
     * @param code
     * @param data
     */
    @ApiAuth
    @RequestMapping(value = "/editPacks")
    public JsonApiResult modifyPack(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("execute editPacks");
        Date start = new Date();
        // ==============解析数据=============
        JSONObject jsonData = JSONObject.parseObject(data);
        Long memberId = jsonData.getLong("memberId");
        MemberVO member = memberQueryService.getById(memberId);
        OpChannelVO channel = mcOpChannelService.findByCode(code);
        Integer accessWayId = jsonData.getInteger("accessWay");
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(accessWayId);
        List<RequestPack> editPacks = JSON.parseArray(jsonData.getString("packs"), RequestPack.class);
        List<RequestPack> packs = exchangeCartService.modifyPacks(memberId, editPacks);

        // =============业务逻辑=============
        List<CartProductPackVO> packsAll = exchangeShopCartAdapter.buildPackVOsByRequest(packs, member, accessWay, channel);

        Date end = new Date();
        logger.info("方法editPacks花费时间：{}毫秒", end.getTime() - start.getTime());
        return JsonApiResult.ofSuccessResult(packsAll);
    }

    /**
     * 删除商品包
     *
     * @param code
     * @param data
     */
    @ApiAuth
    @RequestMapping(value = "/deletePacks")
    public JsonApiResult delPacks(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("execute deletePacks");
        JSONObject jsonData = JSONObject.parseObject(data);
        Long memberId = jsonData.getLong("memberId");
        List<Long> packIds = JSON.parseArray(jsonData.getString("packIds"), Long.class);
        if (CollectionUtils.isNotEmpty(packIds)) {
            exchangeCartService.delPacks(memberId, packIds);
        }
        return JsonApiResult.ofSuccessResult(true);
    }

    /**
     * 清空失效商品
     *
     * @return
     */
    @ApiAuth
    @RequestMapping(value = "/clearInvalidPacks")
    public JsonApiResult clearInvalid(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("execute clearInvalidPacks");
        JSONObject jsonData = JSONObject.parseObject(data);
        Long memberId = jsonData.getLong("memberId");
        MemberVO member = memberQueryService.getById(memberId);
        Integer accessWayId = jsonData.getInteger("accessWay");
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(accessWayId);
        OpChannelVO channel = mcOpChannelService.findByCode(code);
        // 获取购物车记录
        List<RequestPack> requestPacks = exchangeCartService.getCart(member.getId());
        List<CartProductPack> invalidPacks = exchangeShopCartAdapter.filterPackByByRequest(false, requestPacks, member, accessWay, channel);

        if (CollectionUtils.isNotEmpty(invalidPacks)) {
            List<Long> delPackIds = Lists.newArrayList();
            for (CartProductPack invalidPack : invalidPacks) {
                delPackIds.add(invalidPack.getId());
            }
            exchangeCartService.delPacks(memberId, delPackIds);
        }
        return JsonApiResult.ofSuccessResult(true);
    }


    /**
     * 预览订单。 在购物车里勾选
     *
     * @param code
     * @param data
     * @return
     */
    @ApiAuth
    @RequestMapping(value = "/preview")
    public JsonApiResult preview(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("execute preview");
        Date start = new Date();
        JSONObject jsonData = JSONObject.parseObject(data);
        Long memberId = jsonData.getLong("memberId");
        Integer accessWayId = jsonData.getInteger("accessWay");
        AccessWayEnum accessWay = AccessWayEnum.getEnumByCode(accessWayId);
        MemberVO member = memberQueryService.getById(memberId);
        OpChannelVO channel = mcOpChannelService.findByCode(code);
        List<RequestPack> requestPacks = JSON.parseArray(jsonData.getString("packs"), RequestPack.class);
        CartVO cartVo = null;
        if (CollectionUtils.isNotEmpty(requestPacks)) {
            List<Long> packIds = Lists.newArrayList();
            for (RequestPack requestPack : requestPacks) {
                packIds.add(requestPack.getId());
            }
            List<RequestPack> packs = exchangeCartService.getPacksByIds(memberId, packIds);
            cartVo = exchangeShopCartAdapter.previewExchangeCart(packs, member, channel, accessWay);
        }
        Date end = new Date();
        logger.info("方法preview花费时间：{}毫秒", end.getTime() - start.getTime());
        return JsonApiResult.ofSuccessResult(cartVo);
    }


}
