package com.thebeastshop.pegasus.channelservice.controller.address.app;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.alibaba.fastjson.JSONObject;
import com.thebeastshop.member.service.MemberAddressService;
import com.thebeastshop.member.service.MemberQueryService;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.ApiAuth;
import com.thebeastshop.pegasus.channelservice.JsonApiResult;
import com.thebeastshop.pegasus.channelservice.adapter.apiaddressfilter.AppEmojiFilter;
import com.thebeastshop.pegasus.channelservice.adapter.member.MemberAddressAdapter;
import com.thebeastshop.pegasus.channelservice.constants.ErrorCode;
import com.thebeastshop.pegasus.channelservice.util.MobileUtil;
import com.thebeastshop.pegasus.component.compatible.domain.Ring;
import com.thebeastshop.pegasus.component.compatible.service.RingService;
import com.thebeastshop.pegasus.component.support.ComponentServiceLoader;
import com.thebeastshop.support.vo.member.MemberAddressDTO;
import com.thebeastshop.support.vo.member.MemberAddressVO;

/**
 * 地址服务接口（目前app使用）
 * 
 * @author Paul-xiong
 *
 */
@Controller("AppAddressController")
@RequestMapping("/api/app/address")
@ResponseBody
public class AddressController {

	private final Logger logger = LoggerFactory.getLogger(AddressController.class);

	@Autowired
	private MemberAddressService addressService;

	@Autowired
	private MemberQueryService memberQueryService;

	private final RingService _ringService = ComponentServiceLoader.getBean(RingService.class);

	@Autowired
	private MemberAddressAdapter memberAddressAdapter;

	/**
	 * 创建一个地址
	 * 
	 * @return
	 */
	@ApiAuth
	@RequestMapping(value = "/add")
	public JsonApiResult createAddress(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
		JSONObject jsonData = JSONObject.parseObject(data);
		Long memberId = jsonData.getLong("memberId");
		MemberAddressDTO addAddress = jsonData.getObject("addAddress", MemberAddressDTO.class);
		if (addAddress == null) {
			logger.error(ErrorCode.DEFAULT_EXCEPTION.getCode() + ":" + ErrorCode.DEFAULT_EXCEPTION.getMesage());
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(),
					ErrorCode.PARAMS_ILLEGAL.getMesage());
		}
		if(StringUtils.isNotBlank(addAddress.getReceiverPhone()) && !MobileUtil.isValidPhoneNumber(MobileUtil.getMobile(addAddress.getReceiverPhone()))){
			logger.warn("非法的手机号");
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(),
					"非法的手机号");
		}

		MemberVO member = memberId != null ? memberQueryService.getById(memberId) : null;
		com.thebeastshop.member.vo.MemberAddressVO address = new com.thebeastshop.member.vo.MemberAddressVO();
		address.setMemberId(memberId);
		address.setReceiver(AppEmojiFilter.filterEmoji(addAddress.getReceiverName()));
		address.setReceiverPhone(addAddress.getReceiverPhone());
		//过滤Emoji的地址信息
		address.setAddress(AppEmojiFilter.filterEmoji(addAddress.getLocation()));
		address.setZipCode(addAddress.getZipCode());
		address.setAsDefault(addAddress.getAsDefault());
		address.getDistrict().setId(addAddress.getDistrictId());
		// 环线信息
		if (addAddress.getRingId() != null) {
			Ring ring = _ringService.getById(addAddress.getRingId());
			if (ring != null) {
				address.setCircuitDesc(ring.name);
			}
		}
		if (address.getAsDefault() && memberId != null) {
			// 如果新地址要设为默认地址，则原默认地址要修改
			com.thebeastshop.member.vo.MemberAddressVO memberAddress = this.addressService.getDefaultByOwnerId(memberId);
			if (memberAddress != null) {
				memberAddress.setAsDefault(false);
				this.addressService.update(memberAddress);
			}
		}
		// 创建新地址
		address = addressService.create(address);
		MemberAddressVO addressVo = memberAddressAdapter.getMemberAddress(address, member,addAddress.getRingId());
		return JsonApiResult.ofSuccessResult(addressVo);
	}

	/**
	 * 修改一个地址
	 * 
	 * @param code
	 * @param data
	 */
	@ApiAuth
	@RequestMapping(value = "/edit")
	public JsonApiResult updateAddress(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
		JSONObject jsonData = JSONObject.parseObject(data);
		Long memberId = jsonData.getLong("memberId");
		Long addressId = jsonData.getLong("addressId");
		MemberAddressDTO editAddress = jsonData.getObject("editAddress", MemberAddressDTO.class);
		if (memberId == null && addressId == null || editAddress == null) {
			logger.error(ErrorCode.DEFAULT_EXCEPTION.getCode() + ":" + ErrorCode.DEFAULT_EXCEPTION.getMesage());
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(),
					ErrorCode.PARAMS_ILLEGAL.getMesage());
		}

		MemberVO member = memberQueryService.getById(memberId);
		com.thebeastshop.member.vo.MemberAddressVO address = addressService.getById(addressId);
		if (address != null && address.getMemberId().equals(memberId)) {
			address.setReceiver(AppEmojiFilter.filterEmoji(editAddress.getReceiverName()));
			address.setReceiverPhone(editAddress.getReceiverPhone());
			address.setAsDefault(editAddress.getAsDefault());
			address.setAddress(AppEmojiFilter.filterEmoji(editAddress.getLocation()));
			address.setZipCode(editAddress.getZipCode());
			address.getDistrict().setId(editAddress.getDistrictId());
			// 环线信息
			address.setCircuitDesc("");
			if (editAddress.getRingId() != null) {
				Ring ring = _ringService.getById(editAddress.getRingId());
				if (ring != null) {
					address.setCircuitDesc(ring.name);
				}
			}
			if (address.getAsDefault()) {
				// 如果更新地址要设为默认地址，则原默认地址要修改
				com.thebeastshop.member.vo.MemberAddressVO memberAddress = this.addressService.getDefaultByOwnerId(memberId);
				if (memberAddress != null) {
					memberAddress.setAsDefault(false);
					this.addressService.update(memberAddress);

				}
			}
		}
		// 更新地址
		addressService.update(address);
		MemberAddressVO addressVo = memberAddressAdapter.getMemberAddress(address, member);
		return JsonApiResult.ofSuccessResult(addressVo);
	}

	/**
	 * 删除一个地址
	 * 
	 * @param code
	 */
	@ApiAuth
	@RequestMapping(value = "/delete")
	public JsonApiResult deleteAddress(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
		JSONObject jsonData = JSONObject.parseObject(data);
		Long memberId = jsonData.getLong("memberId");
		Long addressId = jsonData.getLong("addressId");
		if (memberId == null || addressId == null) {
			logger.error(ErrorCode.DEFAULT_EXCEPTION.getCode() + ":" + ErrorCode.DEFAULT_EXCEPTION.getMesage());
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(),
					ErrorCode.PARAMS_ILLEGAL.getMesage());
		}

		addressService.delete(addressId);
		return JsonApiResult.ofSuccessResult(true);
	}

	/**
	 * 获取某会员的地址列表
	 * 
	 * @return
	 */
	@ApiAuth
	@RequestMapping(value = "/list")
	public JsonApiResult listAddress(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
		JSONObject jsonData = JSONObject.parseObject(data);
		Long memberId = jsonData.getLong("memberId");
		if (memberId == null) {
			logger.error(ErrorCode.DEFAULT_EXCEPTION.getCode() + ":" + ErrorCode.DEFAULT_EXCEPTION.getMesage());
			return JsonApiResult.ofErrorResult(ErrorCode.PARAMS_ILLEGAL.getCode(),
					ErrorCode.PARAMS_ILLEGAL.getMesage());
		}

		MemberVO member = memberQueryService.getById(memberId);
		List<com.thebeastshop.member.vo.MemberAddressVO> addresses = addressService.list(memberId);
		List<MemberAddressVO> list = memberAddressAdapter.getMemberAddress(addresses, member);
		return JsonApiResult.ofSuccessResult(list);
	}

}
