package com.thebeastshop.pegasus.channelservice.controller.activity;


import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.configuration.service.ReceiveCouponService;
import com.thebeastshop.configuration.vo.ReceiveCouponVO;
import com.thebeastshop.coupon.service.CpCouponService;
import com.thebeastshop.coupon.service.CpCouponSetService;
import com.thebeastshop.coupon.service.CpCouponThirdPartyService;
import com.thebeastshop.coupon.vo.CpCouponVO;
import com.thebeastshop.coupon.vo.CpSendCouponUserVO;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.JsonApiResult;
import com.thebeastshop.pegasus.channelservice.adapter.redenvelope.RedEnvelopeReceivedAdapter;
import com.thebeastshop.support.exception.WrongArgException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import java.util.List;


/**
 * 优惠券活动 Controller
 */
@Controller("activityCouponController")
@RequestMapping("/api/activity/coupon")
public class ActivityCouponController {

    private static Logger logger = LoggerFactory.getLogger(ActivityCouponController.class);
    @Autowired
    private RedEnvelopeReceivedAdapter redEnvelopeReceivedAdapter;

    @Resource
    private CpCouponThirdPartyService cpCouponThirdPartyService;

    @Resource
    private CpCouponService cpCouponService;

    @Resource
    private CpCouponSetService cpCouponSetService;

    @Resource
    private ReceiveCouponService receiveCouponService;

    @RequestMapping(value = "/receive", method = RequestMethod.GET)
    @ResponseBody
    public JsonApiResult receiveCoupon(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("[post] receiveCoupon");
        JSONObject jsonData = JSONObject.parseObject(data);
        String mobile = jsonData.getString("mobile");
        //活动id
        String activityCode = jsonData.getString("activityCode");
        if (StringUtils.isBlank(mobile) && StringUtils.isBlank(activityCode)) {
            return JsonApiResult.ofErrorResult(-10, "请输入手机号和活动Code");
        }

        try {
            // 获取手机号对应的会员
        	MemberVO member = redEnvelopeReceivedAdapter.getMemberByMobile(mobile, code);
            if (null == member) {
                return JsonApiResult.ofErrorResult(-10, "注册会员失败");
            }
            ActivityReceiveConfig receiveConfig = this.getActivityInfo(activityCode);
            Boolean flag = Boolean.FALSE;
            String receiveValue = receiveConfig.getReceiveValue();
            switch (receiveConfig.getReceiveType()){
                case COUPON:
                    flag = this.sendPlatformCoupons(member.getId(),receiveValue);
                    break;
                case GROUP_COUPON:
                    flag = this.sendGroupCoupons(member.getId(),receiveValue);
                    break;
                case THIRD_COUPON:
                    flag = this.sendThirdCoupons(member.getId(),receiveValue);
                    break;
            }
            if (flag) {
                return JsonApiResult.ofSuccessResult(true);
            } else {
                return JsonApiResult.ofErrorResult(3, "发送优惠券失败");
            }
        } catch (WrongArgException wae) {
            logger.error(wae.toString());
            return JsonApiResult.ofErrorResult(3, wae.getMessage());
        } catch (Exception e) {
            logger.error(e.getMessage());
            return JsonApiResult.ofErrorResult(3, e.getMessage());
        }
    }

    /**
     * 根据活动code 获取优惠全发放配置
     * @param activityCode
     * @return
     */
    private ActivityReceiveConfig getActivityInfo(String activityCode) {
        logger.info("[getActivityInfo] accivityCode{}", activityCode);
        ActivityReceiveConfig conf = new ActivityReceiveConfig();
        ServiceResp<ReceiveCouponVO> resp = receiveCouponService.getReceiveCouponByEncryptionId(activityCode);
        if(resp != null && resp.isSuccess()){
            ReceiveCouponVO receiveCoupon = resp.getBean();
            if(receiveCoupon.getRelatedTypeId()==1){
                conf.build(ActivityReceiveConfig.ReceiveType.COUPON,receiveCoupon.getRelatedId().toString());
            }else if(receiveCoupon.getRelatedTypeId() ==4){
                conf.build(ActivityReceiveConfig.ReceiveType.GROUP_COUPON,receiveCoupon.getRelatedId().toString());
            }else if(receiveCoupon.getRelatedTypeId() == 5){
                conf.build(ActivityReceiveConfig.ReceiveType.THIRD_COUPON,receiveCoupon.getRelatedId().toString());
            }else{
                logger.error("查询活动配置信息失败：{}",resp.toString());
                throw new WrongArgException("查询配置信息失败,不支持的优惠券类型");
            }
        }else {
            logger.error("查询活动配置信息失败：{}",resp == null ?"返回null":resp.toString());
            throw new WrongArgException("活动已过期");
        }

        return conf;
    }


    /**
     * 发送第三方券
     *
     * @param memberId
     * @param cstpId
     * @return
     */
    private Boolean sendThirdCoupons(Long memberId, String cstpId) {
        logger.info("[sendThirdCoupons] memberId: {} cstpId: {}", memberId, cstpId);
        ServiceResp<Boolean> resp = cpCouponThirdPartyService.exchangeToMember(Long.parseLong(cstpId), memberId);
        if(resp == null ){
            logger.error("第三方优惠券发放返回null");
            throw new WrongArgException("第三方优惠券发放系统返回null");
        }else if(resp.isFailure()){
            logger.error("第三方优惠券发放失败：{}",resp.toString());
            throw new WrongArgException(resp.getRespMsg());
        }
        logger.info("发送第三方优惠券成功");
        return true;
    }


    /**
     * 发送线下券
     *
     * @param memberId
     * @param cstpId
     * @return
     */
    private Boolean sendPlatformCoupons(Long memberId, String cstpId) {
        logger.info("[sendPlatformCoupons] memberId: {} cstpId: {}", memberId, cstpId);
        ServiceResp<CpCouponVO> resp = cpCouponService.send(Long.parseLong(cstpId),memberId);
        if(resp == null ){
            logger.error("优惠券发放系统返回null");
            throw new WrongArgException("优惠券发放系统返回null");
        }else if(resp.isFailure()){
            logger.error("优惠券发放失败：{}",resp.toString());
            throw new WrongArgException(resp.getRespMsg());
        }
        logger.info("发送优惠券成功");
        return true;
    }

    /**
     * 发送组合券
     *
     * @param memberId
     * @param groupId
     * @return
     */
    private Boolean sendGroupCoupons(Long memberId, String groupId) {
        logger.info("[sendGroupCoupons] memberId: {} groupId: {}", memberId, groupId);
        List<Long> memberIds =  Lists.newArrayList(memberId);
        List<Long> groupIds = Lists.newArrayList(Long.parseLong(groupId));
        ServiceResp<Boolean> resp = cpCouponSetService.sendCouponsToMemberIds(groupIds, memberIds);
        if(resp == null ){
            logger.error("组合优惠券发放返回null");
            throw new WrongArgException("优惠券系统返回null");
        }else if(resp.isFailure()){
            logger.error("组合优惠券发放失败：{}",resp.toString());
            throw new WrongArgException(resp.getRespMsg());
        }
        logger.info("发送组合优惠券成功");
        return true;
    }

    @RequestMapping(value = "/receive/MemberAndMobile")
    @ResponseBody
    public JsonApiResult receiveCouponByMemberIdAndMobile(@RequestParam("CODE") String code, @RequestParam("DATA") String data) {
        logger.info("[post] /receive/MemberAndMobile code:{} date:{}",code,data);
        JSONObject jsonData = JSONObject.parseObject(data);
        String mobile = jsonData.getString("mobile");
        //活动id
        Long memberId = jsonData.getLong("memberId");
        Long couponSampleId = jsonData.getLong("sampleId");
        try {
            // 获取手机号对应的会员
            MemberVO member = redEnvelopeReceivedAdapter.getMemberByMobile(mobile, code);
            if (null == member) {
                return JsonApiResult.ofErrorResult(-10, "注册会员失败");
            }
            List<Long> memberIds = Lists.newArrayListWithCapacity(2);
            memberIds.add(memberId);
            if(!member.getId().equals(memberId)){
                memberIds.add(member.getId());
            }
            CpSendCouponUserVO cpSendCouponUserVO = new CpSendCouponUserVO();
            cpSendCouponUserVO.setCouponSampleId(couponSampleId);
            cpSendCouponUserVO.setMemberIds(memberIds);
            ServiceResp<Boolean> resp = cpCouponService.sendCouponUsers(cpSendCouponUserVO);

            if(resp.isSuccess() && resp.getBean()) {
                return JsonApiResult.ofSuccessResult(true);
            }else{
                logger.info("发放优惠券失败:{}",cpSendCouponUserVO);
                return JsonApiResult.ofErrorResult(3,resp.getRespCode()+":"+resp.getRespMsg());

            }

        } catch (Exception e) {
            logger.error(e.getMessage());
            return JsonApiResult.ofErrorResult(3, e.getMessage());
        }
    }



}


/**
 * 活动领取配置
 * receiveType `enum.<COUPON|THIRD_COUPON|GROUP_COUPON>` 平台优惠券|第三方优惠券|组合优惠券
 * receiveValue `String` 活动发放配置绑定值
 */
class ActivityReceiveConfig{
    public enum ReceiveType {COUPON,THIRD_COUPON,GROUP_COUPON};

    private ReceiveType receiveType;
    private String receiveValue;

    public ReceiveType getReceiveType() {
        return receiveType;
    }

    public void setReceiveType(ReceiveType receiveType) {
        this.receiveType = receiveType;
    }

    public String getReceiveValue() {
        return receiveValue;
    }

    public void setReceiveValue(String receiveValue) {
        this.receiveValue = receiveValue;
    }

    /**
     *  构建领取配置对象
     * @param receiveType   `enum.<COUPON|THIRD_COUPON|GROUP_COUPON>` 平台优惠券|第三方优惠券|组合优惠券
     * @param receiveValue 活动发放配置绑定值
     */
    public void build(ReceiveType receiveType,String receiveValue){
        this.setReceiveType(receiveType);
        this.setReceiveValue(receiveValue);
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
                .append("receiveType", receiveType)
                .append("receiveValue", receiveValue)
                .toString();
    }
}