package com.thebeastshop.pegasus.channelservice.aop;

import java.lang.reflect.Method;

import com.thebeastshop.support.exception.BaseException;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.beast.clog.agent.TagBuilder;
import com.beast.clog.agent.log.ILog;
import com.beast.clog.agent.log.LogManager;
import com.beast.clog.agent.metircs.IMetrics;
import com.beast.clog.agent.metircs.MetricsManager;
import com.beast.clog.common.utils.HostUtil;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.thebeastshop.common.utils.GZIPHelper;

/**
 * @Author: alexzchen
 * @Date: Created in 2018/9/18
 * @Description:
 */

@Configuration
@EnableAspectJAutoProxy(proxyTargetClass = true)
@ComponentScan("com.thebeastshop.pegasus.channelservice.controller")
@Aspect
@Component
public class ControllerAspect {

    private static ILog logger = LogManager.getLogger(ControllerAspect.class);
    private static IMetrics metrics = MetricsManager.getMetrics();
    private static String hostName = HostUtil.getHostName();

    @Around("execution(* com.thebeastshop.pegasus.channelservice.controller..*.*(..))")
    public Object doAround(ProceedingJoinPoint pJoinPoint) throws Throwable {

        Class<?> clazz = pJoinPoint.getTarget().getClass();
        String className = clazz.getSimpleName();
        Signature signature = pJoinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        String methodName = method.getName();

        long start = System.currentTimeMillis();

        try {
            Object result = pJoinPoint.proceed();
            long end = System.currentTimeMillis();
            metrics.record("beast.cs.controller.request", end - start, TagBuilder.create().append("controller", className).append("method", methodName).append("success", "true").append("host", hostName).build());
            return result;

        } catch (BaseException ex) {
            long end = System.currentTimeMillis();
            logger.warn("自定义业务异常: ", ex, TagBuilder.create().append("controller", className).append("method", methodName).append("executeTime", end - start).build());
            metrics.record("beast.cs.controller.request", end - start, TagBuilder.create().append("controller", className).append("method", methodName).append("success", "true").append("host", hostName).build());
            throw ex;
        } catch (Throwable ex) {
            long end = System.currentTimeMillis();
            logger.error("系统异常: ", ex, TagBuilder.create().append("controller", className).append("method", methodName).append("executeTime", end - start).build());
            metrics.record("beast.cs.controller.request", end - start, TagBuilder.create().append("controller", className).append("method", methodName).append("success", "false").append("host", hostName).build());
            logArgs(pJoinPoint, className, methodName);
            throw ex;
        }

    }

    private String getArgValue(Object arg) {
        String argValue;
        if (arg == null) {
            return "null";
        }
        if (arg instanceof CharSequence) {
            return "\"" + arg.toString() + "\"";
        }
        if (int.class.isAssignableFrom(arg.getClass()) ||
                long.class.isAssignableFrom(arg.getClass()) ||
                short.class.isAssignableFrom(arg.getClass()) ||
                byte.class.isAssignableFrom(arg.getClass()) ||
                boolean.class.isAssignableFrom(arg.getClass())) {
            return String.valueOf(arg);
        }
        argValue = JSON.toJSONString(arg, SerializerFeature.DisableCircularReferenceDetect);
        return argValue;
    }

    private void logArgs(ProceedingJoinPoint pJoinPoint, String className, String methodName) {
        Object arg = null;
        StringBuilder argsBuilder = new StringBuilder();
        int argsLen = pJoinPoint.getArgs().length;
        try {
            for (int i = 0; i < argsLen; i++) {
                arg = pJoinPoint.getArgs()[i];
                int argIndex = i + 1;
                String argValue = getArgValue(arg);
                argsBuilder.append("参数" + argIndex + ": " + argValue);
                if (i < argsLen - 1) {
                    argsBuilder.append(", ");
                }
            }

            logger.error("系统异常参数", argsBuilder.toString(), TagBuilder.create().append("controller", className).append("method", methodName).build());

        } catch (Exception e) {
            logger.error(e);
        }
    }
}
