package com.thebeastshop.pegasus.channelservice.adapter.product;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.adapter.price.PriceAdapter;
import com.thebeastshop.pegasus.merchandise.vo.*;
import com.thebeastshop.price.service.PrsPriceService;
import com.thebeastshop.price.vo.PrsCalcProductDTO;
import com.thebeastshop.price.vo.PrsProductPriceResultVO;
import com.thebeastshop.price.vo.PrsSpvPriceResultVO;
import com.thebeastshop.support.file.FileConstant;
import com.thebeastshop.support.vo.product.ProductDimensionVO;
import com.thebeastshop.support.vo.product.ProductVO;
import com.thebeastshop.support.vo.product.ProductVariantVO;
import com.thebeastshop.support.vo.product.SpvVO;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;

/**
 * 组合搭配适配
 * Created by GFL on 2018/11/29.
 */
@Service
public class ProductCombinationAdapter {

    private static Logger logger = LoggerFactory.getLogger(ProductCombinationAdapter.class);

    @Autowired
    private PriceAdapter priceAdapter;

    @Autowired
    private ProductAdapter productAdapter;

    @Autowired
    private PrsPriceService prsPriceService;

    @Autowired
    private SpvPriceAdapter spvPriceAdapter;


    public List<ProductVO> getCombinedProductDetails(List<PsProductVO> products, MemberVO member, OpChannelVO channel,
                                                     AccessWayEnum accessWay) {

        List<ProductVO> result = Lists.newArrayList();
        if (CollectionUtils.isEmpty(products)) return result;

        PrsCalcProductDTO arg0 = new PrsCalcProductDTO();
        Map<String, List<PsSkuVO>> mapProductId = getProductSkuMap(products);
        priceAdapter.assemblePriceProducts(arg0, products, mapProductId, member, channel, accessWay);
        logger.info("商品列表页批量获取商品价格");
        Map<Long, PrsProductPriceResultVO> priceResultMap = prsPriceService.calculatePrs4ProductList(arg0);
        Map<Long, BigDecimal> spvPriceMap = getSpvPriceMap(priceResultMap.values());
        Map<String, ProductVariantVO> variantMap = this.getProductVariantMap(products, member, channel, spvPriceMap);

        for (PsProductVO product : products) {
            ProductVO productVO = buildCombinationProduct(product, channel, variantMap.get(product.getCode()));
            result.add(productVO);
        }
        return result;
    }

    private ProductVO buildCombinationProduct(PsProductVO psProd, OpChannelVO channel, ProductVariantVO variant) {

        ProductVO productVO = new ProductVO();
        productVO.setId(psProd.getCode());
        productVO.setName(psProd.getNameCn());
        productVO.setSummary(psProd.getDescription());
        productVO.setSoldOut(productAdapter.checkIsSoldOut(psProd, channel));
        productVO.setCombination(BooleanUtils.toBoolean(psProd.getDisplayInCombinedArea()));
        productVO.setVariant(variant);

        String featureImage = psProd.getPicList();
        if (StringUtils.isNotBlank(featureImage) && !featureImage.startsWith(FileConstant.BASE_URL)) {
            featureImage = FileConstant.BASE_URL + featureImage;
        }
        productVO.setFeatureImage(featureImage);

        SpvVO.BrandVO brandVO = new SpvVO.BrandVO();
        if (CollectionUtils.isNotEmpty(psProd.getBrands())) {
            PsBrandVO brand = psProd.getBrands().get(0);
            if (brand != null) {
                brandVO.setId(brand.getBrandId());
                brandVO.setName(brand.getBrandName());
            }
        }
        productVO.setBrand(brandVO);

        List<SpvVO> spvVOS = (ArrayList)variant.getSpvs();
        Collections.sort(spvVOS, new Comparator<SpvVO>() {
            @Override
            public int compare(SpvVO o1, SpvVO o2) {
                int i = o1.getRawPrice().compareTo(o2.getRawPrice());
                if(i==0){
                    i=o1.getPrice().compareTo(o2.getPrice());
                }
                return i;
            }
        });
        SpvVO economizeSpv = spvVOS.get(0);

        productVO.setPrice(economizeSpv.getPrice());
        productVO.setRawPrice(economizeSpv.getRawPrice());
        return productVO;
    }

    /**
     * 获取商品规格信息
     *
     * @param products
     * @param member
     * @param opChannel
     * @param priceMap
     * @return
     */
    private Map<String, ProductVariantVO> getProductVariantMap(List<PsProductVO> products, MemberVO member, OpChannelVO opChannel, Map<Long, BigDecimal> priceMap) {

        Map<String, ProductVariantVO> resultMap = new HashMap<>();
        List<PsSpvVO> spvs = Lists.newArrayList();
        for (PsProductVO product : products) {
            spvs.addAll(product.getSpvList());
        }

        Map<Long, SpvVO> spvMap = spvPriceAdapter.getSpvPrice(spvs, products, member, opChannel, null, priceMap);

        for (PsProductVO product : products) {
            ProductVariantVO variant = new ProductVariantVO();
            Map<String, ProductDimensionVO> pdMap = Maps.newLinkedHashMap(); // 规格信息
            List<SpvVO> spvVOList = Lists.newArrayList();
            for (PsSpvVO spv : product.getSpvList()) {
                SpvVO spvVO = spvMap.get(spv.getSpvId());
                spvVO.setCheap(spv.getCostPrice().compareTo(spvVO.getRawPrice().multiply(new BigDecimal("0.7"))) == 1);
                List<Integer> group = null;
                List<PsAttributeValuePairVO> attrList = spv.getAttributeValuePairList();
                if (CollectionUtils.isNotEmpty(attrList)) {
                    group = new ArrayList<Integer>();
                    // 设置规格信息
                    for (PsAttributeValuePairVO attr : attrList) {
                        if (attr != null && StringUtils.isNotBlank(attr.getAttrName())
                                && StringUtils.isNotBlank(attr.getValue())) {
                            String attrName = attr.getAttrName();
                            String attrValue = attr.getValue();
                            ProductDimensionVO pd = pdMap.get(attrName);
                            if (pd == null) {
                                pd = new ProductDimensionVO();
                                pd.setChoices(new ArrayList<String>());
                                pd.setName(attrName);
                                pdMap.put(attrName, pd);
                            }
                            List<String> choices = pd.getChoices();
                            if (!choices.contains(attrValue))
                                choices.add(attrValue);
                            group.add(choices.indexOf(attrValue));
                        }
                    }
                }
                spvVO.setGroup(group);
                spvVOList.add(spvVO);
            }

            Collection<ProductDimensionVO> dimensionVos = pdMap.values();

            if (CollectionUtils.isNotEmpty(dimensionVos) && dimensionVos.size() == 1) {
                Iterator<ProductDimensionVO> iterator = dimensionVos.iterator();
                ProductDimensionVO vo = iterator.next();
                List<String> choices = vo.getChoices();
                if (CollectionUtils.isNotEmpty(choices) && choices.size() == 1) {
                    dimensionVos = null;
                }
            }

            variant.setDimensions(dimensionVos);
            variant.setSpvs(spvVOList);

            resultMap.put(product.getCode(), variant);
        }
        return resultMap;
    }


    /**
     * 根据价格计算结果获取spv售价map
     *
     * @param priceVo
     * @return
     */
    private Map<Long, BigDecimal> getSpvPriceMap(Collection<PrsProductPriceResultVO> priceVo) {

        Map<Long, BigDecimal> spvPriceMap = new HashMap<>();
        List<PrsSpvPriceResultVO> spvPriceList = Lists.newArrayList();
        for (PrsProductPriceResultVO prsProdPrice : priceVo) {

            if (CollectionUtils.isNotEmpty(prsProdPrice.getSpvPriceResultVOs())) {
                List<PrsSpvPriceResultVO> result = prsProdPrice.getSpvPriceResultVOs();
                for (PrsSpvPriceResultVO re : result) {
                    spvPriceMap.put(re.getSpvId(), re.getDiscountPrice());
                }
            }
        }
        return spvPriceMap;
    }

    /*
     * 获取商品skuMap
     */
    private Map<String, List<PsSkuVO>> getProductSkuMap(List<PsProductVO> products) {
        Map<String, List<PsSkuVO>> map = new HashMap<String, List<PsSkuVO>>();
        if (CollectionUtils.isEmpty(products)) return map;

        for (PsProductVO product : products) {
            if (CollectionUtils.isNotEmpty(product.getSpvList())) {
                for (PsSpvVO spvVO : product.getSpvList()) {
                    if (spvVO.getCombined()) {
                        if (CollectionUtils.isNotEmpty(spvVO.getSkuCombinationList())) {
                            for (PsSkuCombinationVO psSkuCombinationVO : spvVO.getSkuCombinationList()) {
                                if (map.containsKey(product.getId())) {
                                    map.get(product.getId()).add(psSkuCombinationVO.getSubSkuVO());
                                } else {
                                    map.put(product.getId(), Lists.newArrayList(psSkuCombinationVO.getSubSkuVO()));
                                }
                            }
                        }
                    } else {
                        if (null != spvVO.getSkuVO())
                            if (map.containsKey(product.getId())) {
                                map.get(product.getId()).add(spvVO.getSkuVO());
                            } else {
                                map.put(product.getId(), Lists.newArrayList(spvVO.getSkuVO()));
                            }
                    }
                }
            }
        }
        return map;
    }
}
