package com.thebeastshop.pegasus.channelservice.adapter.product;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.campaign.enums.DiscountTypeEnum;
import com.thebeastshop.campaign.enums.MemberTypeEnum;
import com.thebeastshop.campaign.service.CampaignLimitService;
import com.thebeastshop.cart.resp.CartProductPack;
import com.thebeastshop.common.utils.DateUtil;
import com.thebeastshop.member.constant.MemberConstant;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.adapter.price.IposPriceAdapter;
import com.thebeastshop.pegasus.merchandise.service.McPsSpvService;
import com.thebeastshop.pegasus.merchandise.vo.PsAttributeValuePairVO;
import com.thebeastshop.pegasus.merchandise.vo.PsBrandVO;
import com.thebeastshop.pegasus.merchandise.vo.PsCampaignVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuCombinationVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import com.thebeastshop.pegasus.service.operation.vo.OpPresaleVO;
import com.thebeastshop.price.vo.PrsAdditionalPackVO;
import com.thebeastshop.stock.service.SPresaleService;
import com.thebeastshop.support.Label;
import com.thebeastshop.support.YesOrNo;
import com.thebeastshop.support.file.FileConstant;
import com.thebeastshop.support.vo.product.SimpleProductVO.DiscVO;
import com.thebeastshop.support.vo.product.SpvVO;
import com.thebeastshop.support.vo.product.SpvVO.BrandVO;
import com.thebeastshop.support.vo.product.SpvVO.FlashSaleVO;
import com.thebeastshop.support.vo.product.SpvVO.PreDeliverVO;
import com.thebeastshop.support.vo.product.SpvVO.Quota;

/**
 * IPOS SPV 适配器
 * @Class Name IposSpvAdapter
 * @Author Xiang.Gao
 * @Create In 2018-12-26
 */
@Component
public class IposSpvAdapter {
	
	@Autowired
	SpvCommonAdapter spvCommonAdapter;
	
	@Autowired
	private SpvCustomizeAdapter spvCustomizeAdapter;
	
	@Autowired
	CampaignLimitService campaignLimitService;
	
	@Autowired
	private McPsSpvService mcPsSpvService;
	
	@Autowired
	private SPresaleService spresaleService;
	
	@Autowired
	private IposPriceAdapter iposPriceAdapter;
	
	/**
	 * 获取SPV库存MAP
	 * @Methods Name getSpvStockMap
	 */
	public Map<Long, PsSpvVO> getSpvStockMap(Collection<PsProductVO> products, String chnCode) {
		Map<Long, PsSpvVO> psSpvVOMap = Maps.newConcurrentMap();
		if (CollectionUtils.isEmpty(products)) {
			return psSpvVOMap;
		}
		List<Long> spvIds = Lists.newArrayList();
		for (PsProductVO product : products) {
			if (CollectionUtils.isEmpty(product.getSpvList())){
				continue;
			}
			List<PsSpvVO> spvList = product.getSpvList();
			for (PsSpvVO spv : spvList) {
				spvIds.add(spv.getSpvId());
			}
		}
		List<PsSpvVO> psSpvVOList = mcPsSpvService.findBySpvIdsWithStock(spvIds, chnCode);
		for (PsSpvVO psSpvVO : psSpvVOList) {
			psSpvVOMap.put(psSpvVO.getSpvId(), psSpvVO);
		}
		return psSpvVOMap;
	}
	
	/**
	 * 统计本地库存
	 * @param psSpvVO
	 */
	public int countSpvLocalStock(PsSpvVO psSpvVO) {
		int localLeft = 0;
		if (psSpvVO == null) {
			return localLeft;
		}
		if (BooleanUtils.isTrue(psSpvVO.isJit())) {
			return Integer.MAX_VALUE -1; 
		}
		if (psSpvVO.getPsStockVO() != null && psSpvVO.getPsStockVO().getCanUseQuantity() != null) {
			if (psSpvVO.getPsStockVO().getCanUseQuantity() > 0) {
				localLeft =  psSpvVO.getPsStockVO().getCanUseQuantity();
			}
		}
		return localLeft;
	}
	
	/**
	 * 统计仓库库存 = 配送仓 + 预售仓
	 * @param psSpvVO
	 */
	public int countSpvWarehouseStock(PsSpvVO psSpvVO) {
		int warehouseLeft = 0 ;
		if (psSpvVO == null) {
			return warehouseLeft;
		}
		if (BooleanUtils.isTrue(psSpvVO.isJit())) {
			return Integer.MAX_VALUE -1; 
		}
		if (psSpvVO.getPsFurnitureStockVO() != null && psSpvVO.getPsFurnitureStockVO().getCanUseQuantity() != null) {
			if (psSpvVO.getPsFurnitureStockVO().getCanUseQuantity() > 0) {
				warehouseLeft += psSpvVO.getPsFurnitureStockVO().getCanUseQuantity();
			}
		}
		if (psSpvVO.getPsPresaleSaleStockVO() != null && psSpvVO.getPsPresaleSaleStockVO().getAvailQuantity() > 0) {
			warehouseLeft += psSpvVO.getPsPresaleSaleStockVO().getAvailQuantity();
		}
		return warehouseLeft;
	}
	
	/**
	 * 统计 spv 库存 = 本地仓 + 配送仓 + 预售仓
	 * @Methods Name countSpvStock
	 */
	public int countSpvStock(PsSpvVO psSpvVO, boolean containtWh) {
		int left = 0 ;
		if (psSpvVO == null) {
			return left;
		}
		if (BooleanUtils.isTrue(psSpvVO.isJit())) {
			return Integer.MAX_VALUE -1; 
		}
		if (psSpvVO.getPsStockVO() != null && psSpvVO.getPsStockVO().getCanUseQuantity() != null) {
			if (psSpvVO.getPsStockVO().getCanUseQuantity() > 0) {
				left += psSpvVO.getPsStockVO().getCanUseQuantity();
			}
		}
		if (containtWh) { 
			if (psSpvVO.getPsFurnitureStockVO() != null && psSpvVO.getPsFurnitureStockVO().getCanUseQuantity() != null) {
				if (psSpvVO.getPsFurnitureStockVO().getCanUseQuantity() > 0) {
					left += psSpvVO.getPsFurnitureStockVO().getCanUseQuantity();
				}
			}
			if (psSpvVO.getPsPresaleSaleStockVO() != null && psSpvVO.getPsPresaleSaleStockVO().getAvailQuantity() > 0) {
				left += psSpvVO.getPsPresaleSaleStockVO().getAvailQuantity();
			}
		}
		return left;
	}
	
	/**
	 * 统计现货库存 = 本地仓 + 配送仓
	 * @param psSpvVO
	 */
	public int countSpotStock(PsSpvVO psSpvVO) {
		int spotLeft = 0;
		if (psSpvVO.getPsStockVO() != null && psSpvVO.getPsStockVO().getCanUseQuantity() != null) {
			if (psSpvVO.getPsStockVO().getCanUseQuantity() > 0) {
				spotLeft += psSpvVO.getPsStockVO().getCanUseQuantity();
			}
		}
		if (psSpvVO.getPsFurnitureStockVO() != null && psSpvVO.getPsFurnitureStockVO().getCanUseQuantity() != null) {
			if (psSpvVO.getPsFurnitureStockVO().getCanUseQuantity() > 0) {
				spotLeft += psSpvVO.getPsFurnitureStockVO().getCanUseQuantity();
			}
		}
		return spotLeft;
	}
	
	/**
	 * 统计预售库存
	 * @param psSpvVO
	 */
	public int countPreSaleStock(PsSpvVO psSpvVO) {
		int preLeft = 0;
		if (psSpvVO.getPsPresaleSaleStockVO() != null && psSpvVO.getPsPresaleSaleStockVO().getAvailQuantity() > 0) {
			preLeft += psSpvVO.getPsPresaleSaleStockVO().getAvailQuantity();
		}
		return preLeft;
	}
	
	/**
	 * 校验 spv 库存
	 * 
	 * @Methods Name checkSpvStock
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return Boolean
	 */
	public Boolean checkProductStock(PsProductVO product, boolean containWhDeliver,
			Map<Long, PsSpvVO> spvStockMap, int count) {
		List<PsSpvVO> spvList = product.getSpvList();
		for (PsSpvVO psSpvVO : spvList) {
			int left = countSpvStock(spvStockMap.get(psSpvVO.getSpvId()), containWhDeliver);
			if(count <= left) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 获取包裹 SPV 信息
	 * 
	 * @Methods Name getPsSpvVOByPack
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return List<PsSpvVO>
	 */
	public List<PsSpvVO> getPsSpvVOByPack(List<CartProductPack> packs) {
		List<PsSpvVO> psSpvList = Lists.newArrayList();
		for (CartProductPack pack : packs) {
			psSpvList.add(pack.getSpv());
			if (CollectionUtils.isEmpty(pack.getGiftPacks())) {
				continue;
			}
			for (CartProductPack giftPack : pack.getGiftPacks()) {
				psSpvList.add(giftPack.getSpv());
			}
		}
		return psSpvList;
	}
	
	/**
	  * 设置SPV 活动限量信息
	 * 
	 * @param spv
	 * @param spvVO
	 * @param member
	 */
	public void setSpvCampaignLimit(PsSpvVO spv, SpvVO spvVO, MemberVO member) {
		// 参与抢购活动
		PsCampaignVO rushCamp = getRushCamp(spv);
		// 价格折扣限量
		PsCampaignVO priceLimitCamp = getPriceLimitCamp(spv);
		if (rushCamp != null) {
			FlashSaleVO flashSale = new FlashSaleVO();
			Integer rushStock = iposPriceAdapter.getPanicLeftAmount(rushCamp.getCode(), member.getCode(), spv);
			flashSale.setSoldOut(rushStock <= 0);
			if (rushCamp.getExpireTime() != null) {
				String dateExp = DateUtil.formatDate(rushCamp.getExpireTime(), DateUtil.FORMAT_YYYY_MM_DD_HH_MM_SS);
				Timestamp ts = Timestamp.valueOf(dateExp);
				flashSale.setExpiresAt(ts);
			}
			spvVO.setFlashSale(flashSale);
			if(rushStock <= spv.getCanUseQuantityOffLine()){
				spvVO.setLeft(rushStock);
			}
		} else if(priceLimitCamp != null && !MemberConstant.DEFAULT_MEMBER_ID.equals(member.getId())) {
			Integer priceStock = iposPriceAdapter.getPriceUserLeftAmount(priceLimitCamp.getCode(), member.getCode(), spv);
			spvVO.setNewcomerTag(priceStock > 0);
			Quota quta = new Quota();
			quta.setTotal(priceLimitCamp.getLimitAmount());
			quta.setLeft(priceStock);
			spvVO.setQuota(quta);
		}
	}
	
	/**
	 *  构建主品SpvVO
	 * @param product
	 * @param spv
	 * @param member
	 * @return SpvVO
	 */
	public SpvVO buildingRawSpvVO(PsProductVO product, PsSpvVO spv, MemberVO member) {
		SpvVO spvVO = new SpvVO();
		if(BooleanUtils.toBoolean(product.getDisplayInCombinedArea())){
			spvVO.setCombinedSingleSpvs(getSpvCombinedSingleList(spv));
			spvVO.setCombination(Boolean.TRUE);
		}
		spvVO.setId(spv.getSpvId());
		spvVO.setName(product.getNameCn());
		spvVO.setBrand(getBrand(product.getBrands()));
		spvVO.setImage(getFeatureImage(product.getPicList()));
		spvVO.setSpvDesc(getSpvDesc(spv.getAttributeValuePairList()));
		spvVO.setRawPrice(product.getCombined() ? spv.getOriginalPrice() : spv.getSalesPrice());
		spvVO.setProductCode(spv.getProductCode());
		spvVO.setSkuCodes(getSkuCodes(spv));
		spvVO.setCombined(spv.getCombined());
		spvVO.setLeft(spv.getCanUseQuantityOffLine());
		spvVO.setPrice(spv.getDiscountPrice() != null ? spv.getDiscountPrice() : spv.getSalesPrice());
		spvVO.setMinAmount(spv.getMinAmount() != null ? spv.getMinAmount() : 1);
		spvVO.setInstallDesc(getInstallDesc(spv));
		spvVO.setIsOversea(Integer.valueOf(1).equals(product.getCrossBorderFlag()));
		spvVO.setDiscount(getDiscVO(member, product.getAllowVipDiscount(), spv.getCampaignList()));
		// 设置SPV 活动限量信息
		setSpvCampaignLimit(spv, spvVO, member);
		return spvVO;
	}
	
	/**
	 *  构建赠品/换购spvVO
	 * 
	 * @Methods Name getSpvVO
	 * @return SpvVO
	 */
	public SpvVO buildingAddSpvVO(PsProductVO product, PsSpvVO spv, int left) {
		SpvVO spvVO = new SpvVO();
		spvVO.setId(spv.getSpvId());
		spvVO.setName(product.getNameCn());
		spvVO.setBrand(getBrand(product.getBrands()));
		spvVO.setImage(getFeatureImage(product.getPicList()));
		spvVO.setSpvDesc(getSpvDesc(spv.getAttributeValuePairList()));
		spvVO.setRawPrice(product.getCombined() ? spv.getOriginalPrice() : spv.getSalesPrice());
		spvVO.setPrice(spv.getDiscountPrice() != null ? spv.getDiscountPrice() : spv.getSalesPrice());
		spvVO.setProductCode(spv.getProductCode());
		spvVO.setSkuCodes(getSkuCodesBySpv(spv));
		spvVO.setCombined(spv.getCombined());
		spvVO.setIsOversea(Integer.valueOf(1).equals(product.getCrossBorderFlag()));
		spvVO.setLeft(left);
		spvVO.setMinAmount(spv.getMinAmount() != null ? spv.getMinAmount() : 1);
		spvVO.setInstallDesc(getInstallDesc(spv));
		return spvVO;
	}
	
	/**
	 * 商品 安装信息
	 */
	private String getInstallDesc(PsSpvVO spv) {
		List<PsSkuVO> psSkuList = Lists.newArrayList();
		if (spv.getCombined()) {
			List<PsSkuCombinationVO> comSpvList = spv.getSkuCombinationList();
			for (PsSkuCombinationVO com : comSpvList) {
				psSkuList.add(com.getSubSkuVO());
			}
		}else {
			psSkuList.add(spv.getSkuVO());
		}
		for (PsSkuVO sku : psSkuList) {
			if (sku == null || sku.getInstallService() == null || sku.getInstallService() == -1) {
				continue;
			}
			if (sku.getInstallService() == 2) {
				return "需要安装";
			}
			if (sku.getInstallService() == 1) {
				return "用户自行安装";
			}
			if (sku.getInstallService() == 0) {
				return "无需安装";
			}
		}
		return "";
	}
	
	/**
	 * 组合商品单品列表
	 * @param psSpvVO
	 * @return
	 */
	private List<SpvVO> getSpvCombinedSingleList(PsSpvVO psSpvVO){
		List<SpvVO> spvs = Lists.newArrayList();
		if(psSpvVO.getCombined() && CollectionUtils.isNotEmpty(psSpvVO.getSkuCombinationList())){
			for (PsSkuCombinationVO psSkuCombinationVO : psSpvVO.getSkuCombinationList()) {
				PsSkuVO psSkuVO = psSkuCombinationVO.getSubSkuVO();
				SpvVO spvVO = new SpvVO();
				spvVO.setMinAmount(psSkuCombinationVO.getQuantity());
				spvVO.setName(psSkuVO.getSkuNameCN());
				spvVO.setRawPrice(BigDecimal.valueOf(psSkuVO.getSalesPrice()));
				// featureImage
				String image = psSkuVO.getSkuImg();
				if (StringUtils.isNotBlank(image) && !image.startsWith(FileConstant.BASE_URL)) {
					image = FileConstant.BASE_URL + image;
				}
				spvVO.setImage(image);

				SpvVO.BrandVO brand= new BrandVO();
				brand.setName(psSkuVO.getBrandName());
				brand.setId(psSkuVO.getBrandId());
				spvVO.setBrand(brand);

				String skuAttrs = psSkuCombinationVO.getAttrs();
				List<PsAttributeValuePairVO> attrList = psSpvVO.getAttributeValuePairList();
				if(StringUtils.isNotBlank(skuAttrs) && CollectionUtils.isNotEmpty(attrList)){
					StringBuilder sb = new StringBuilder();
					for (PsAttributeValuePairVO attr : attrList) {
						if(skuAttrs.contains(attr.getAttrId()+"")&&skuAttrs.contains(attr.getAttrValueId()+""))
							sb.append(attr.getAttrName()).append(" : ").append(attr.getValue()).append(" ");
					}
					spvVO.setSpvDesc(sb.toString());
				}
				spvVO.setSkuCodes(Lists.newArrayList(psSkuVO.getSkuCode()));
				spvs.add(spvVO);
			}
		}
		// 按价格由低到高排序
		Collections.sort(spvs, new Comparator<SpvVO>() {
			@Override
			public int compare(SpvVO o1, SpvVO o2) {
				return o1.getRawPrice().compareTo(o2.getRawPrice());
			}
		});
		return spvs;
	}
	
	/**
	 * 获取图片
	 * @Methods Name getFeatureImage
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private String getFeatureImage(String featureImage) {
		if (StringUtils.isNotBlank(featureImage) && !featureImage.startsWith(FileConstant.BASE_URL)) {
			featureImage = FileConstant.BASE_URL + featureImage;
		}
		return featureImage;
	}
	
	/**
	 * SPV 下skuCode集合
	 * 
	 * @Methods Name getSkuCodes
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private List<String> getSkuCodes(PsSpvVO spv) {
		List<String> skuCodes = Lists.newArrayList();
		if (spv.getCombined()) {
			for (PsSkuCombinationVO sku : spv.getSkuCombinationList()) {
				skuCodes.add(sku.getSubSkuCode());
			}
		} else {
			skuCodes.add(spv.getSkuCode());
		}
		return skuCodes;
	}
	
	/**
	 * 品牌
	 * @Methods Name getBrand
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private BrandVO getBrand(List<PsBrandVO> brands) {
		BrandVO brandVO = new BrandVO();
		if (CollectionUtils.isNotEmpty(brands)) {
			PsBrandVO brand = brands.get(0);
			if (brand != null) {
				brandVO.setId(brand.getBrandId());
				brandVO.setName(brand.getBrandNameCN());
			}
		}
		return brandVO;
	}
	
	/**
	 * 折扣信息
	 * @Methods Name getDiscVO
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private DiscVO getDiscVO(MemberVO member,Integer prodVipDiscountFlag, List<PsCampaignVO> campList) {
		DiscVO disc = new DiscVO();
		boolean prodVipDiscount = Integer.valueOf(1).equals(prodVipDiscountFlag);
		if (member.getMemberLevel() > 0 && prodVipDiscount) {
			disc.setVipLevel(member.getMemberLevel().toString());
		}
		disc.setBirthday(false);
		if (member.isInBdayDiscountTime() && prodVipDiscount) {
			disc.setBirthday(true);
			disc.setVipLevel(null);
		}
		for (PsCampaignVO camp : campList) {
			if (camp.getDiscountTypeId() == 6 && null != camp.getExpireTime()) {
				disc.setVipLevel(null);
				disc.setBirthday(false);
			}
		}
		return disc;
	}
	
	/**
	 * spv 描述
	 * @Methods Name getSpvDesc
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private String getSpvDesc(List<PsAttributeValuePairVO> attrList) {
		StringBuilder sb = new StringBuilder();
		if (CollectionUtils.isNotEmpty(attrList)) {
			for (PsAttributeValuePairVO attr : attrList) {
				if (attr != null && StringUtils.isNotBlank(attr.getAttrName()) && StringUtils.isNotBlank(attr.getValue()))
					sb.append(attr.getAttrName()).append(" : ").append(attr.getValue()).append(" ");
			}
		}
		return sb.toString();
	}
	
	/**
	 * 获取spv下skuCode
	 * @Methods Name getSkuCodesBySpv
	 * @Create In 2018-12-21 By Xiang.Gao
	 * @return List<String>
	 */
	private List<String> getSkuCodesBySpv(PsSpvVO spv) {
		List<String> skuCodes = Lists.newArrayList();
		if (spv.getCombined()) {
			if (CollectionUtils.isNotEmpty(spv.getSkuCombinationList())) {
				for (PsSkuCombinationVO psc : spv.getSkuCombinationList()) {
					skuCodes.add(psc.getSubSkuCode());
				}
			}
		} else {
			skuCodes.add(spv.getSkuCode());
		}
		return skuCodes;
	}
	
	/**
	 * 获取额外商品SPV
	 * @param signGiftList
	 * @param addProdMap
	 */
	public List<PsSpvVO> getAddSpvByProduct(List<PrsAdditionalPackVO> signGiftList, Map<String, PsProductVO> addProdMap) {
		List<PsSpvVO> spvList = Lists.newArrayList();
		for (PrsAdditionalPackVO add : signGiftList) {
			PsProductVO product = addProdMap.get(add.getProductId().toString());
			if (product == null) {
				continue;
			}
			spvList.addAll(product.getSpvList());
		}
		return spvList;
	}
	
	/**
	 * 获取预计发货信息
	 * @param pack
	 * @param spvPresaleDateMap
	 */
	public PreDeliverVO getPreDeliverVO(CartProductPack pack, Date presaleDate) {
		return this.getPreDeliverVO(pack.getProduct(), pack.getCount(), pack.getSpv(), presaleDate);
	}
	
	/**
	 * 获取预计发货信息
	 * @param pack
	 * @param spvPresaleDateMap
	 */
	public PreDeliverVO getPreDeliverVO(PsProductVO product, int buyCount, PsSpvVO psSpvVO, Date presaleDate) {
		// 过滤 月送鲜花、组合、海淘商品
		if (YesOrNo.YES.getId().equals(product.getIsMonthSend())){
			return null;
		}
		if (Integer.valueOf(1).equals(product.getCrossBorderFlag())) {
			return null;
		}
		if (BooleanUtils.isTrue(product.getCombined())) {
			return null;
		}
		PreDeliverVO preDeliverVO = null;
		// 现货库存 
		int spotStockNum = this.countSpotStock(psSpvVO);
		// 预售库存
		int preSellStockNum = this.countPreSaleStock(psSpvVO);
		
		// 预售 购买数量大于现货 且小于预售数量
		if (presaleDate != null &&  buyCount > spotStockNum && buyCount <= preSellStockNum) {
			preDeliverVO = new PreDeliverVO();
			Calendar c = Calendar.getInstance();
			c.setTime(presaleDate);
			int month = c.get(Calendar.MONTH) + 1;
			int day = c.get(Calendar.DAY_OF_MONTH);
			preDeliverVO.setType(0);
			preDeliverVO.setDeliverAt(c.getTime());
			preDeliverVO.setContent("该商品预售, 预计" + month + "月" + day + "日发货。");
		} 
		
		// 来单 购买数量大于现货
		PsSkuVO sku = psSpvVO.getSkuVO();
		if (Integer.valueOf(1).equals(sku.getIsJit()) && buyCount > spotStockNum) {
			Integer deliverDay = sku.getDeliveryPeriod();
			if (deliverDay == null) {
				deliverDay = 0;
			}
			if (StringUtils.isNotBlank(sku.getCustomizeType())) {
				int days = spvCustomizeAdapter.getCustomizePlanedDeliveryDate(sku);
				if (deliverDay < days) {
					deliverDay = days;
				}
			}
			Calendar c = Calendar.getInstance();
			c.add(Calendar.DATE, deliverDay);
			if (preDeliverVO != null && preDeliverVO.getDeliverAt().after(c.getTime())) {
				return preDeliverVO;
			}
			preDeliverVO = new PreDeliverVO();
			preDeliverVO.setType(1);
			preDeliverVO.setDeliverAt(c.getTime());
			preDeliverVO.setContent("该商品为来单制作，预计将于"+ deliverDay +"个工作日后发货。");
		}
		return preDeliverVO;
	}
	
	/**
	  *  获取spv标签
	 * @param psSpvVO
	 * @return
	 */
	public List<Label> getSpvLabel(PsSpvVO psSpvVO) {
		List<Label> labels = Lists.newArrayList();
		if (psSpvVO.getSkuVO() != null && Integer.valueOf(1).equals(psSpvVO.getSkuVO().getCrossBorderFlag())) {
			labels.add(getOverSeaLabel());
		} else if (psSpvVO.getCombined() && CollectionUtils.isNotEmpty(psSpvVO.getSkuCombinationList())) {
			for (PsSkuCombinationVO psSkuCombinationVO : psSpvVO.getSkuCombinationList()) {
				if (Integer.valueOf(1).equals(psSkuCombinationVO.getSubSkuVO().getCrossBorderFlag())) {
					labels.add(getOverSeaLabel());
					break;
				}
			}
		}
		return labels;
	}
	
	/**
	  *  提取SPV
	 * @param packList
	 * @return
	 */
	public List<PsSpvVO> featchPsSpvVO(List<CartProductPack> packList) {
		List<PsSpvVO> spvs = Lists.newArrayList();
		for (CartProductPack pack : packList) {
			spvs.add(pack.getSpv());
		}
		return spvs;
	}
	
	private Label getOverSeaLabel() {
		return new Label("跨境商品", "#FFFFFF", "#000000", "本商品由香港发货 ，价格已含税", "#333333", "", 0, true);
	}
	
	/**
	  * 获取SKU预售时间
	 */
	public Map<String, Date> mapSkuCodePresale(Collection<PsSpvVO> spvs) {
		Map<String, Date> planedDeliveryDate = Maps.newHashMap();
		if (CollectionUtils.isEmpty(spvs)) {
			return planedDeliveryDate;
		}

		List<String> skuCodes = new ArrayList<>();
		for (PsSpvVO psSpvVo : spvs) {
			skuCodes.add(psSpvVo.getSkuCode());
		}
		planedDeliveryDate = spresaleService.findPlanedDeliveryDateBySkuCodes(skuCodes, OpPresaleVO.CHANNEL_TYPE_OFFLINE);
		return planedDeliveryDate;
	}
	
	/**
	  * 获取抢购活动
	 */
	public PsCampaignVO getRushCamp(PsSpvVO spv) {
		List<PsCampaignVO> campList = spv.getCampaignList();
		for (PsCampaignVO camp : campList) {
			if (camp.getDiscountTypeId() != 6) {
				continue;
			}
			return camp;
		}
		return null;
	}
	
	/**
	   * 获取价格折扣限量活动
	 * @param spv
	 * @return
	 */
	public PsCampaignVO getPriceLimitCamp(PsSpvVO spv) {
		List<PsCampaignVO> campList = spv.getCampaignList();
		for (PsCampaignVO camp : campList) {
			if (!DiscountTypeEnum.PRICE_DISCOUNT.getId().equals(camp.getDiscountTypeId())) {
				continue;
			}
			if (!MemberTypeEnum.NEW_MEMBER.getCode().equals(camp.getMemberType())) {
				continue;
			}
			return camp;
		}
		return null;
	}
	
}
