package com.thebeastshop.pegasus.channelservice.adapter.price;

import com.alibaba.fastjson.JSON;
import com.beast.clog.common.utils.DateUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.campaign.enums.DiscountTypeEnum;
import com.thebeastshop.campaign.enums.MemberTypeEnum;
import com.thebeastshop.campaign.service.CampaignLimitService;
import com.thebeastshop.campaign.vo.CampaignStreamVO;
import com.thebeastshop.cart.enums.CartPackSourceEnum;
import com.thebeastshop.cart.resp.CartProductPack;
import com.thebeastshop.cart.support.ScDeliveryVO;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.common.utils.NumberUtil;
import com.thebeastshop.cooperation.service.ShopVipService;
import com.thebeastshop.cooperation.vo.VipActivityVO;
import com.thebeastshop.coupon.enums.CpCouponTypeEnum;
import com.thebeastshop.coupon.service.CpCouponService;
import com.thebeastshop.coupon.vo.CpCouponVO;
import com.thebeastshop.coupon.vo.CpFixedPriceVO;
import com.thebeastshop.coupon.vo.CpFullCutPriceVO;
import com.thebeastshop.coupon.vo.CpFullDiscountVO;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.adapter.member.MemberAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.product.ProductAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.product.SpvCustomizeAdapter;
import com.thebeastshop.pegasus.component.delivery.constants.PostageConstant;
import com.thebeastshop.pegasus.component.discount.DiscountType;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.vo.*;
import com.thebeastshop.price.enums.PrsCartSettleLabelEnum;
import com.thebeastshop.price.enums.PrsCartSourceEnum;
import com.thebeastshop.price.enums.PrsPriceTypeEnum;
import com.thebeastshop.price.service.PrsPriceService;
import com.thebeastshop.price.vo.*;
import com.thebeastshop.privilege.service.InterestFreeService;
import com.thebeastshop.privilege.vo.InterestFreeCashVO;
import com.thebeastshop.privilege.vo.InterestFreeVO;
import com.thebeastshop.support.Label;
import com.thebeastshop.support.Label.CampaignVO;
import com.thebeastshop.support.Label.LabelMode;
import com.thebeastshop.support.Label.PostageVO;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.util.JsonUtil;
import com.thebeastshop.support.util.PriceUtil;
import com.thebeastshop.support.vo.cart.CartProductPackVO.PackCampaignVO;
import com.thebeastshop.support.vo.cart.CartProductPackVO.PackCampaignVO.InfoVO;
import com.thebeastshop.support.vo.giftcard.GiftCardVO;
import com.thebeastshop.support.vo.market.MarketVipDTO;
import com.thebeastshop.support.vo.order.OrderDetailVO.PriceStepVO;
import com.thebeastshop.support.vo.order.OrderPreviewDTO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;
import java.util.Map.Entry;

/**
 * IPOS 价格计算适配器
 * @Class Name IposPriceAdapter
 * @Author Xiang.Gao
 * @Create In 2018-12-25
 */
@Component
public class IposPriceAdapter {
	
	private Logger logger = LoggerFactory.getLogger(getClass());
	
	@Autowired
	private SpvCustomizeAdapter spvCustomizeAdapter;
	
	@Autowired
	private ProductAdapter productAdapter;
	
	@Autowired
	private ShopVipService shopVipService;
	
	@Autowired
	private McPsProductService mcPsProductService;
	
	@Autowired
	private InterestFreeService interestFreeService;
	
	@Autowired
	private CpCouponService cpCouponService;
	
	@Autowired
	private CampaignLimitService campaignLimitService;
	
	@Autowired
	private PrsPriceService prsPriceService;
	
	@Autowired
	private MemberAdapter memberAdapter;
	
	@Autowired
	private IposCampaignAdapter iposCampaignAdapter;
	
	/**
	 * 价格计算 会员
	 * 
	 * @Methods Name getPrsMember
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private PrsMemberVO getPrsMember(MemberVO member) {
		Boolean isNewMember = memberAdapter.isNewMemberIposCart(member.getId());
		PrsMemberVO membervo = new PrsMemberVO();
		membervo.setBirthday(member.getBirthday());
		membervo.setCode(member.getCode());
		membervo.setId(member.getId());
		membervo.setMemberLevel(member.getMemberLevel());
		membervo.setNickName(member.getNickName());
		membervo.setFishMan(isNewMember);
		return membervo;
	}
	
	/**
	 * 价格计算 礼品卡
	 * 
	 * @Methods Name getPrsGiftCard
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return List<PrsGiftCardVO>
	 */
	private List<PrsGiftCardVO> getPrsGiftCard(List<GiftCardVO> giftCards) {
		List<PrsGiftCardVO> prsCardList = new ArrayList<PrsGiftCardVO>();
		if (CollectionUtils.isEmpty(giftCards)) {
			return prsCardList;
		}
		for (GiftCardVO card : giftCards) {
			PrsGiftCardVO prsCard = new PrsGiftCardVO();
			prsCard.setAmount(card.getAmount());
			prsCard.setCardNo(card.getCardNo());
			// 因礼品卡 失效时间可能为空 暂时默认
			prsCard.setExpireDate(DateUtil.parse("2999-12-31 00:00:00"));
			prsCardList.add(prsCard);
		}
		return prsCardList;
	}
	
	/**
	 * 价格计算 优惠券
	 * 
	 * @Methods Name getPrsCouponVO
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	private PrsCouponVO getPrsCouponVO(CpCouponVO cpCoupon) {
		if (cpCoupon == null) {
			return null;
		}
		PrsCouponSampleVO sample = new PrsCouponSampleVO();
		sample.setAccessWayIds(cpCoupon.getCouponSample().getAccessWayIds());
		sample.setCategoryIds(cpCoupon.getCouponSample().getCategoryIds());
		sample.setCouponTypeId(cpCoupon.getCouponSample().getCouponTypeId());
		sample.setCrossBorder(cpCoupon.getCouponSample().getCrossBorder());

		if (cpCoupon.getCouponSample().getCouponTypeId().equals(CpCouponTypeEnum.CHEAPEN_LINE.getId())) {
			CpFullCutPriceVO fullCutCoupon = (CpFullCutPriceVO) cpCoupon.getCouponSample().getCouponContent();
			sample.setCutPrice(fullCutCoupon.getCutPrice());
			sample.setExcludeProductIds(cpCoupon.getCouponSample().getExcludeProductIds());
			sample.setExpireTime(cpCoupon.getCouponSample().getExpireTime());
			// 是否可优惠累加
			if (fullCutCoupon.getIsTopLimit() != null && fullCutCoupon.getIsTopLimit() == 0) {
				sample.setCumulative(1);
			} else {
				sample.setCumulative(0);
			}
			// 优惠内容
			PrsCouponSectionVO couponSectionVO = new PrsCouponSectionVO();
			couponSectionVO.setLine(fullCutCoupon.getFullPrice());
			couponSectionVO.setFactor(fullCutCoupon.getCutPrice());
			sample.setSectionVOS(Lists.newArrayList(couponSectionVO));
			sample.setFullPrice(fullCutCoupon.getFullPrice());
		} else if (cpCoupon.getCouponSample().getCouponTypeId().equals(CpCouponTypeEnum.DISCOUNT_LINE.getId())) {
			CpFullDiscountVO fullCutCoupon = (CpFullDiscountVO) cpCoupon.getCouponSample().getCouponContent();
			sample.setCutPrice(fullCutCoupon.getDiscount());
			sample.setExcludeProductIds(cpCoupon.getCouponSample().getExcludeProductIds());
			sample.setExpireTime(cpCoupon.getCouponSample().getExpireTime());
			// 是否可优惠累加
			sample.setCumulative(0);
			// 优惠内容
			PrsCouponSectionVO couponSectionVO = new PrsCouponSectionVO();
			couponSectionVO.setLine(fullCutCoupon.getFullPrice());
			couponSectionVO.setFactor(fullCutCoupon.getDiscount());
			sample.setSectionVOS(Lists.newArrayList(couponSectionVO));
			sample.setFullPrice(fullCutCoupon.getFullPrice());
		} else if (cpCoupon.getCouponSample().getCouponTypeId().equals(CpCouponTypeEnum.FIXEDPRICE_LINE.getId())) {
			CpFixedPriceVO fullCutCoupon = (CpFixedPriceVO) cpCoupon.getCouponSample().getCouponContent();
			sample.setCutPrice(fullCutCoupon.getFixedPrice());
			sample.setExcludeProductIds(cpCoupon.getCouponSample().getExcludeProductIds());
			sample.setExpireTime(cpCoupon.getCouponSample().getExpireTime());
			// 是否可优惠累加
			sample.setCumulative(0);
			// 优惠内容
			PrsCouponSectionVO couponSectionVO = new PrsCouponSectionVO();
			couponSectionVO.setLine(BigDecimal.ZERO);
			couponSectionVO.setFactor(fullCutCoupon.getFixedPrice());
			sample.setSectionVOS(Lists.newArrayList(couponSectionVO));
			sample.setFullPrice(BigDecimal.ZERO);
		}
		sample.setId(cpCoupon.getCouponSample().getId());
		sample.setName(cpCoupon.getCouponSample().getName());
		sample.setOfflineStoreIds(cpCoupon.getCouponSample().getOfflineStoreIds());
		sample.setProductIds(cpCoupon.getCouponSample().getProductIds());
		sample.setProductScopeId(cpCoupon.getCouponSample().getProductScopeId());
		sample.setStartTime(cpCoupon.getCouponSample().getStartTime());
		sample.setStateId(cpCoupon.getCouponSample().getStateId());
		sample.setTitle(cpCoupon.getCouponSample().getTitle());
		PrsCouponVO coupon = new PrsCouponVO();
		coupon.setCouponSample(sample);
		coupon.setCouponSampleId(cpCoupon.getCouponSampleId());
		coupon.setExpireTime(cpCoupon.getExpireTime());
		coupon.setId(cpCoupon.getId());
		coupon.setOwnerId(cpCoupon.getOwnerId());
		coupon.setStartTime(cpCoupon.getStartTime());
		coupon.setUsed(cpCoupon.getUsed());
		coupon.setValid(cpCoupon.getValid());
		return coupon;
	}
	
	private List<PrsSkuVO> getPrsPackSkus(PsSpvVO psSpv, List<PsSkuVO> psSkus) {
		// SKU信息
		List<PrsSkuVO> skus = new ArrayList<PrsSkuVO>();
		if (psSpv.getCombined()) {
			Map<String, Integer> subSkuCountMap = new HashMap<String, Integer>();
			List<String> skuCodes = new ArrayList<>();
			if (CollectionUtils.isNotEmpty(psSpv.getSkuCombinationList())) {
				for (PsSkuCombinationVO psSku : psSpv.getSkuCombinationList()) {
					skuCodes.add(psSku.getSubSkuCode());
					subSkuCountMap.put(psSku.getSubSkuCode(), psSku.getQuantity());
				}
			}
			for (String skuCode : skuCodes) {
				for (PsSkuVO psSku : psSkus) {
					if (psSku.getSkuCode().equals(skuCode)) {
						skus.add(new PrsSkuVO(psSpv.getSpvId(), psSku.getSkuCode(), psSku.getSkuNameCN(),
								new BigDecimal(psSku.getSalesPrice()), subSkuCountMap.get(skuCode)));
					}
				}
			}
		} else {
			BigDecimal salesPrice = BigDecimal.ZERO;
			for (PsSkuVO psSku : psSkus) {
				if (psSpv.getSkuCode().equals(psSku.getSkuCode())) {
					salesPrice = psSpv.getSalesPrice();
					skus.add(new PrsSkuVO(psSpv.getSpvId(), psSku.getSkuCode(), psSku.getSkuNameCN(), salesPrice, 1));
				}
			}
		}
		return skus;
	}
	
	/**
	 * 价格计算参数组装
	 * 
	 * @Methods Name getPrsProductPacks
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return List<PrsProductPackVO>
	 */
	private List<PrsProductPackVO> getPrsProductPacks(List<CartProductPack> packs, Boolean birthday, MemberVO member) {
		List<PsProductVO> products = new ArrayList<PsProductVO>();
		List<String> productCodes = Lists.newArrayList();
		for (CartProductPack pack : packs) {
			products.add(pack.getProduct());
			productCodes.add(pack.getProduct().getCode());
		}
		Map<String, List<PsSkuVO>> skuMap = productAdapter.GetSkusByProduct(products);
		List<PrsProductPackVO> packList = new ArrayList<PrsProductPackVO>();
		for (CartProductPack pack : packs) {
			if (CartPackSourceEnum.GIFT.equals(pack.getSource())) {
				continue;
			}
			PsSpvVO psSpv = pack.getSpv();
			PsProductVO psProduct = pack.getProduct();
			
			// 商品活动和spv活动都要取
			List<PsCampaignVO> spvCampaignList = psSpv.getCampaignList();
			List<PsCampaignVO> prodCampaignList = psProduct.getCampaignList();
			List<PsCampaignVO> psCampaignList = new ArrayList<PsCampaignVO>();
			if (CollectionUtils.isNotEmpty(spvCampaignList)) {
				psCampaignList.addAll(spvCampaignList);
			}
			if (CollectionUtils.isNotEmpty(prodCampaignList)) {
				psCampaignList.addAll(prodCampaignList);
			}
			
			PrsProductPackVO packvo = new PrsProductPackVO();
			packvo.setAllowVipDiscount(Integer.valueOf(1).equals(psProduct.getAllowVipDiscount()));
			packvo.setCampaignList(iposCampaignAdapter.priceCampList(psCampaignList));
			packvo.setCategoryId(psProduct.getCategoryId());
			packvo.setCount(pack.getCount());
			packvo.setCrossBorderFlag(psProduct.getCrossBorderFlag());
			packvo.setHasSpecialDiscount(Integer.valueOf(1).equals(psProduct.getIsDiscount()));
			packvo.setId(pack.getId());
			packvo.setIsCustomize(false);
			
			List<PsSkuVO> psSkus = skuMap.get(psProduct.getId());
			if (CollectionUtils.isNotEmpty(psSkus)) {
				for (PsSkuVO psSku : psSkus) {
					if (Integer.valueOf(1).equals(psSku.getCanCustomize())) {
						packvo.setIsCustomize(true);
						packvo.setCustomizePrice(BigDecimal.ZERO);
						BigDecimal customizePrice = spvCustomizeAdapter.getCustomizePriceBySkuCode(pack.getSpv().getSkuCode());
						if (!NumberUtil.isNullOrZero(customizePrice)) {
							packvo.setCustomizePrice(customizePrice);
						}
						break;
					}
				}
			}
			packvo.setAllowVipDiscount(Integer.valueOf(1).equals(psProduct.getAllowVipDiscount()));
			packvo.setUseBirthdayDiscount(birthday && member.isInBdayDiscountTime() 
					&& Integer.valueOf(1).equals(psProduct.getAllowBdDiscount()));
			packvo.setHasSpecialDiscount(Integer.valueOf(1).equals(psProduct.getIsDiscount()));
			packvo.setMinAmount(psSpv.getMinAmount());
			packvo.setProductCode(psProduct.getCode());
			packvo.setProductId(Long.parseLong(psProduct.getId()));
			packvo.setProductName(psProduct.getNameCn());
			packvo.setSalesPrice(pack.getSource().equals(CartPackSourceEnum.ADDIBLE) && psProduct.getCombined()
					? psSpv.getOriginalPrice() : psSpv.getSalesPrice());
			packvo.setOriginalPrice(psSpv.getOriginalPrice());
			packvo.setCombination(psSpv.getCombined());
			
			packvo.setSkuList(getPrsPackSkus(psSpv, psSkus));
			packvo.setSpvId(psSpv.getSpvId());
			packvo.setSource(PrsCartSourceEnum.getEnumByCode(pack.getSource().getCode()));
			packvo.setPoints(0);
			packvo.setPostageDeliverVos(getPrsPackDelivery(pack));
			packList.add(packvo);
		}
		return packList;
	}
	
	/**
	  * 获取可用的免邮权益
	 */
	public InterestFreeVO getInterestFree(List<CartProductPack> packs, Long memberId, 
			AccessWayEnum accessWay, OpChannelVO opChannel) {
		InterestFreeVO interestFreeVO = null;
		List<String> productCodes = Lists.newArrayList();
		for (CartProductPack pack : packs) {
			productCodes.add(pack.getProduct().getCode());
		}
		if (CollectionUtils.isNotEmpty(productCodes)) {
			InterestFreeCashVO interestFreeCashVO = new InterestFreeCashVO();
			interestFreeCashVO.setMemberId(memberId.intValue());
			interestFreeCashVO.setProdCodes(productCodes);
			interestFreeCashVO.setAccessWay(accessWay);
			interestFreeCashVO.setStoreChannel(opChannel.getCode());
			ServiceResp<InterestFreeVO> resp = interestFreeService.getOptimallyUsableInterestFree(interestFreeCashVO);
			if (resp.getBean() != null) {
				interestFreeVO = resp.getBean();
				logger.info("会员member=[{}],有可用的免邮权益 interestFree=[{}]", memberId, JSON.toJSONString(interestFreeVO));
			}
		}
		return interestFreeVO;
	}
	
	/**
	 * 获取优惠券
	 * 
	 * @Methods Name getCoupon
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return CpCouponVO
	 */
	public CpCouponVO getCoupon(Long couponId, Long memberId, String chnCode) {
		if (couponId == null || memberId == null) {
			return null;
		}
		ServiceResp<CpCouponVO> rsp = cpCouponService.checkCoupon(couponId, memberId, chnCode);
		if (rsp == null || rsp.getBean() == null) {
			throw new WrongArgException("优惠券不可用!", "couponId", couponId);
		}
		return rsp.getBean();
	}
	
	/**
	 * 配送信息
	 * 
	 * @Methods Name getPrsPackDelivery
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return List<PrsPostageDeliverVo>
	 */
	public List<PrsPostageDeliverVo> getPrsPackDelivery(CartProductPack pack) {
		List<PrsPostageDeliverVo> deliverList = Lists.newArrayList();
		if (CollectionUtils.isNotEmpty(pack.getDeliverys())) {
			for (ScDeliveryVO delivery : pack.getDeliverys()) {
				deliverList.add(new PrsPostageDeliverVo(delivery.getType(), delivery.getCount(), delivery.getFee()));
			}
		}
		return deliverList;
	}
	
	/**
	 * 商场 VIP 折扣
	 * @Methods Name getMarketVipDiscount
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return BigDecimal
	 */
	public BigDecimal getMarketVipDiscount(MarketVipDTO marketVip) {
		if (marketVip == null) {
			return null;
		}
		ServiceResp<Boolean> resp = shopVipService.isCardUseful(marketVip.getActivityId(), marketVip.getVipCardNo());
		if (!resp.getBean()) {
			throw new WrongArgException("商场VIP卡在本店下单超过限制次数!", "vipCardNo", marketVip.getVipCardNo());
		}
		ServiceResp<VipActivityVO> serviceResp = shopVipService.getVipActivity(marketVip.getActivityId());
		return serviceResp.getBean().getDiscount();
	}
	
	/**
	 * 额外商品信息
	 * 
	 * @Methods Name getAdditionalProductMap
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return Map<String,PsProductVO>
	 */
	public Map<String, PsProductVO> getAdditionalProductMap(List<PrsPriceStepVO> steps) {
		List<String> productIds = Lists.newArrayList();
		for (PrsPriceStepVO step : steps) {
			if (CollectionUtils.isEmpty(step.getAdditionalPackList())) {
				continue;
			}
			for (PrsAdditionalPackVO addpack : step.getAdditionalPackList()) {
				productIds.add(addpack.getProductId().toString());
			}
		}
		Map<String, PsProductVO> psProductMap = mcPsProductService.mapByProdIds(productIds);
		return psProductMap == null ? new HashMap<String, PsProductVO> (): psProductMap; 
	}
	
	/**
	  * 获取活动Map
	 * @Methods Name getCampaignMap
	 * @Create In 2018-12-26 By Xiang.Gao
	 * @param steps
	 * @return Map<PrsCampaignVO,List<PrsAdditionalPackVO>>
	 */
	public Map<PrsCampaignVO, List<PrsAdditionalPackVO>> getCampaignMap(List<PrsPriceStepVO> steps) {
		Map<PrsCampaignVO, List<PrsAdditionalPackVO>> campaignMap = Maps.newHashMap();
		for (PrsPriceStepVO step : steps) {
			if (CollectionUtils.isNotEmpty(step.getAdditionalPackList())) {
				List<PrsAdditionalPackVO> campAddList = new ArrayList<PrsAdditionalPackVO>();
				if (campaignMap.containsKey(step.getCampaign())) {
					campAddList.addAll(campaignMap.get(step.getCampaign()));
				}
				campAddList.addAll(step.getAdditionalPackList());
				campaignMap.put(step.getCampaign(), campAddList);
			}
		}
		Map<PrsCampaignVO, List<PrsAdditionalPackVO>> resultMap = Maps.newHashMap();
		// 筛选赠一赠多
		for (Map.Entry<PrsCampaignVO, List<PrsAdditionalPackVO>> entry : campaignMap.entrySet()) {
			PrsCampaignVO campaignVO = entry.getKey();
			List<PrsAdditionalPackVO> packs = entry.getValue();
			if (CollectionUtils.isEmpty(packs)) {
				resultMap.put(campaignVO, packs);
				continue;
			}
			if (Integer.valueOf(0).equals(campaignVO.getOptionFlag())) {
				resultMap.put(campaignVO, packs);
				continue;
			}
			Map<Long, List<PrsAdditionalPackVO>> sectionMap = new HashMap<Long, List<PrsAdditionalPackVO>>();
			for (PrsAdditionalPackVO pack : packs) {
				List<PrsAdditionalPackVO> groupList = new ArrayList<PrsAdditionalPackVO>();
				if (sectionMap.containsKey(pack.getGroupId())) {
					groupList = sectionMap.get(pack.getGroupId());
				}
				groupList.add(pack);
				sectionMap.put(pack.getGroupId(), groupList);
			}
			for (Map.Entry<Long, List<PrsAdditionalPackVO>> sectionEntry : sectionMap.entrySet()) {
				List<PrsAdditionalPackVO> selectList = new ArrayList<PrsAdditionalPackVO>();
				if (resultMap.containsKey(campaignVO)) {
					selectList = resultMap.get(campaignVO);
				}
				List<PrsAdditionalPackVO> packList = sectionEntry.getValue();
				// 排序 根据sort
				Collections.sort(packList, new Comparator<PrsAdditionalPackVO>() {
					@Override
					public int compare(PrsAdditionalPackVO o1, PrsAdditionalPackVO o2) {
						int a = o1.getSort();
						int b = o2.getSort();
						return (a - b);
					}
				});
				for (PrsAdditionalPackVO pack : packList) {
					selectList.add(pack);
				}
				if (CollectionUtils.isNotEmpty(selectList)) {
					resultMap.put(campaignVO, selectList);
				}
			}
		}
		logger.info("getCampaignMap :{}", JsonUtil.toJson(resultMap));
		return resultMap;
	}
	
	/**
	 * 过滤库存不足、限量不足的商品
	 * 
	 * @Methods Name exceptLimit
	 * @Create In 2018-12-28 By Xiang.Gao
	 */
	public Map<PrsCampaignVO, List<PrsAdditionalPackVO>> exceptLimit(boolean containWhDeliver, 
			Map<PrsCampaignVO, List<PrsAdditionalPackVO>> campaignMap, 
			Map<String, Integer> checkCspLimitMap, Map<Long, PsSpvVO> spvStockMap, 
			Map<String, PsProductVO> psProductVOMap) {
		Map<PrsCampaignVO, List<PrsAdditionalPackVO>> resultMap  = Maps.newHashMap();
		for (Map.Entry<PrsCampaignVO, List<PrsAdditionalPackVO>> entry : campaignMap.entrySet()) {
			PrsCampaignVO campaignVO = entry.getKey();
			List<PrsAdditionalPackVO> packs = entry.getValue();
			if (CollectionUtils.isEmpty(packs)) {
				continue;
			}
			List<PrsAdditionalPackVO> newPacks = new ArrayList<PrsAdditionalPackVO>();
			for (PrsAdditionalPackVO pack : packs) {
				PsProductVO product = psProductVOMap.get(pack.getProductId().toString());
				if (null == product) {
					logger.warn("exceptLimit productIsNull continue {}:", pack.getProductId().toString());
					continue;
				}
				newPacks.add(pack);
			}
			if (CollectionUtils.isNotEmpty(newPacks)) {
				resultMap.put(campaignVO, newPacks);
			}
		}
		logger.info("exceptLimit resultMap :{}", JSON.toJSONString(resultMap));
		return resultMap;
	}
	
	/**
	 * 订单预览标签展示
	 * 
	 * @Methods Name getOrderPreviewLabel
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	public List<Label> getOrderPreviewLabel(List<PrsCartSettleLabelVO> prsCartSetLebels) {
		List<Label> labels = Lists.newArrayList();
		if (CollectionUtils.isEmpty(prsCartSetLebels)) {
			return labels;
		}
		for (PrsCartSettleLabelVO label : prsCartSetLebels) {
			// 满赠换购不显示
			if (label.isFit() && null != label.getDiscountType() 
					&& !label.getDiscountType().getId().equals(DiscountType.GIFT.id)
					&& !label.getDiscountType().getId().equals(DiscountType.CHEAPEN_OTHER.id)) {
				labels.add(setupLabel(label));
			}
		}
		return labels;
	}
	
	/**
	 * 根据活动组装标签
	 * 
	 * @param prsLabel
	 * @return
	 */
	public Label setupLabel(PrsCartSettleLabelVO prsLabel) {
		Label label = new Label();
		label.setType(prsLabel.getCartSettleLabel().getName());
		label.setSummary(prsLabel.getLabelDesc());
		if (prsLabel.getCartSettleLabel().equals(PrsCartSettleLabelEnum.POSTAGE)) {
			label.setMode(LabelMode.POSTAGE);
		} else {
			label.setMode(LabelMode.CAMPAIGN);
		}
		CampaignVO campaignVO = new CampaignVO();
		campaignVO.setId(prsLabel.getCampaignId());
		campaignVO.setName(prsLabel.getCampaignTitle());

		campaignVO.setFulfilled(false);
		if (prsLabel.getHintWords().equals("已满足")) {
			campaignVO.setFulfilled(true);
		}
		campaignVO.setLinkDesc(prsLabel.getHintWords());
		label.setCampaign(campaignVO);
		return label;
	}
	
	/**
	  * 活动限量信息
	 * @param packs
	 * @param member
	 * @return
	 */
	public List<PrsCampaignLimitVO> prsCampaignLimitList(List<CartProductPack> packs, MemberVO member) {
		List<PrsCampaignLimitVO> prsCampaignLimitList = Lists.newArrayList();
		
		for (CartProductPack pack : packs) {
			if (CartPackSourceEnum.GIFT.equals(pack.getSource())) {
				continue;
			}
			PsSpvVO psSpv = pack.getSpv();
			if (psSpv == null) {
				continue;
			}
			List<PsCampaignVO> psCampaignList = pack.getSpv().getCampaignList();
			for (PsCampaignVO camp : psCampaignList) {
				if (!DiscountTypeEnum.PRICE_DISCOUNT.getId().equals(camp.getDiscountTypeId())) {
					continue;
				}
				if (!MemberTypeEnum.NEW_MEMBER.getCode().equals(camp.getMemberType())) {
					continue;
				}
				CampaignStreamVO campaignStream = new CampaignStreamVO();
				campaignStream.setCampaignCode(camp.getCode());
				campaignStream.setSkuCode(psSpv.getSkuCode());
				campaignStream.setProductCode(psSpv.getProductCode());
				campaignStream.setMemberCode(member.getCode());
				ServiceResp<Integer> resp  = campaignLimitService.getPriceDiscountLeftAmount(campaignStream);
				if(resp.isSuccess()) {
					PrsCampaignLimitVO prsCampaignLimit = new PrsCampaignLimitVO();
					prsCampaignLimit.setSpvId(psSpv.getSpvId());
					prsCampaignLimit.setCampaignId(camp.getId());
					prsCampaignLimit.setAvailCnt(resp.getBean());
					prsCampaignLimitList.add(prsCampaignLimit);
				}
			}
		}
		return prsCampaignLimitList;
	}
	
	/**
	 * 购物车预览 价格计算
	 * 
	 * @Methods Name calculatePrsCart
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return PrsCalPackDto
	 */
	public PrsPacksPriceResultVO calculatePrsCart(List<CartProductPack> packs, MemberVO member,
			OpChannelVO opChannel, AccessWayEnum accessWay) {
		
		InterestFreeVO interestFree = getInterestFree(packs, member.getId(), accessWay, opChannel);
		PrsCalPackDto prsCalPack = new PrsCalPackDto();
		prsCalPack.setAccessWay(accessWay);
		prsCalPack.setChannelCode(opChannel.getCode());
		prsCalPack.setMember(getPrsMember(member));
		prsCalPack.setPackList(getPrsProductPacks(packs, true, member));
		prsCalPack.setInterestFreeId(interestFree != null ? interestFree.getProvideId() : null);
		prsCalPack.setCampaignLimitList(prsCampaignLimitList(packs, member));
		
		PrsPacksPriceResultVO priceResult = prsPriceService.calculatePrs4Cart(prsCalPack);
		logger.info("购物车价格计算结果：{}", priceResult.toJson());
		return priceResult;
	}
	
	/**
	 * 订单预览 价格计算 参数组装
	 * 
	 * @Methods Name getPrsCalPackDtoWithOrder
	 * @Create In 2018-12-29 By Xiang.Gao
	 * @return PrsCalPackDto
	 */
	public PrsCalPackDto getPrsCalPackDtoWithOrder(List<CartProductPack> packs, MemberVO member,
			OpChannelVO opChannel, AccessWayEnum accessWay, CpCouponVO cpCoupon, 
			InterestFreeVO interestFree, OrderPreviewDTO orderPreview) {
		
		// 礼品卡
		List<PrsGiftCardVO> prsGiftCardVO = getPrsGiftCard(orderPreview.getGiftCards());
		// 商场VIP折扣
		BigDecimal marketVipDiscount = getMarketVipDiscount(orderPreview.getMarketVip());
		// 自定义运费
		BigDecimal customPostage  = BigDecimal.ZERO;
		if (null != orderPreview.getPostage() && null != orderPreview.getPostage().getNormal()) {
			customPostage = orderPreview.getPostage().getNormal();
		}
		// 免邮权益
		Integer interestFreeId = interestFree != null ? interestFree.getProvideId() : null;
		// 生日折扣
		boolean birthday = orderPreview.isUseBdayDiscount() != null && orderPreview.isUseBdayDiscount();
		PrsCalPackDto prsCalPack = new PrsCalPackDto();
		prsCalPack.setAccessWay(accessWay);
		prsCalPack.setChannelCode(opChannel.getCode());
		prsCalPack.setGiftCardList(prsGiftCardVO);
		prsCalPack.setUseGiftCard(CollectionUtils.isNotEmpty(orderPreview.getGiftCards()));
		prsCalPack.setCoupon(getPrsCouponVO(cpCoupon));
		prsCalPack.setMarketDiscount(marketVipDiscount);
		prsCalPack.setMember(getPrsMember(member));
		prsCalPack.setPackList(getPrsProductPacks(packs, birthday, member));
		prsCalPack.setInterestFreeId(interestFreeId);
		prsCalPack.setCustomPostage(customPostage);
		prsCalPack.setCampaignLimitList(prsCampaignLimitList(packs, member));
		
		return prsCalPack;
	}
	
	/**
	 * 取活动库存限量
	 * @Methods Name getCspLimitMap
	 * @Create In 2018-12-26 By Xiang.Gao
	 * @return Map<String,Integer>
	 */
	public Map<String, Integer> getCspLimitMap(List<PrsAdditionalPackVO> addtionPackList, 
			Map<String, PrsCampaignVO> campaignMap, Map<String, PsProductVO> addProdMap) {
		// 商品信息
		List<CampaignStreamVO> csList = new ArrayList<>();
		for (PrsAdditionalPackVO addPack : addtionPackList) {
			PsProductVO product = addProdMap.get(addPack.getProductId().toString());
			if (product == null) {
				continue;
			}
			PrsCampaignVO campaignVO = campaignMap.get(addPack.getGroupId() + product.getId());
			if (campaignVO == null) {
				continue;
			}
			CampaignStreamVO campaignStreamVO = new CampaignStreamVO();
			campaignStreamVO.setCampaignCode(campaignVO.getCode());
			campaignStreamVO.setProductCode(product.getCode());
			campaignStreamVO.setSkuCode(product.getSkuCodes().get(0));
			csList.add(campaignStreamVO);
		}
		Map<String, Integer> checkCspLimitMap = new HashMap<>();
		if (CollectionUtils.isEmpty(csList)) {
			return checkCspLimitMap;
		}
		// 验证赠品限量库存
		ServiceResp<Map<String, Integer>> srp = campaignLimitService.mapLeftCspLimitAmount(csList);
		if (srp.isSuccess() && srp.getBean() != null) {
			checkCspLimitMap.putAll(srp.getBean());
		}
		return checkCspLimitMap;
	}
	
	/**
	 * 获取活动信息
	 * 
	 * @Methods Name getPackCampaign
	 * @Create In 2018-12-21 By Xiang.Gao
	 * @return PackCampaignVO
	 */
	public PackCampaignVO getPackCampaign(PrsCampaignVO camp, boolean sellOut) {
		PackCampaignVO campaignVO = new PackCampaignVO();
		campaignVO.setId(camp.getId());
		campaignVO.setName(camp.getName());
		InfoVO info = new InfoVO();
		if (DiscountTypeEnum.FULL_GIFT.getId().equals(camp.getDiscountTypeId())) {
			info.setType("赠品");
		}else {
			info.setType(DiscountTypeEnum.getEnumById(camp.getDiscountTypeId()).getName());
		}
		info.setSummaryColor(sellOut ? "#233333": "#333333");
		campaignVO.setInfo(info);
		return campaignVO;
	}
	
	/**
	 * 设置SPV活动价
	 * @param paramPacks
	 * @param spvMap
	 */
	public void assemblePackWithPrice(List<CartProductPack> paramPacks, Map<Long, BigDecimal> spvMap) {
		if (CollectionUtils.isEmpty(paramPacks)) {
			return ;
		}
		Iterator<CartProductPack> iterator = paramPacks.iterator();
		while (iterator.hasNext()) {
			CartProductPack pack = iterator.next();
			PsSpvVO spv = pack.getSpv();
			spv.setDiscountPrice(spvMap.get(pack.getSpvId()));
			pack.setSpv(spv);
		}
	}

	/**
	 * 获取价格步骤
	 * @param priceResult
	 * @param coupon
	 * @param interestFree
	 * @return
	 */
	public Collection<PriceStepVO> getPriceSteps(PrsPacksPriceResultVO priceResult, 
			CpCouponVO coupon, InterestFreeVO interestFree) {
		
		List<PriceStepVO> priceSteps = new ArrayList<PriceStepVO>();
		if (CollectionUtils.isEmpty(priceResult.getPriceStepVOs())) {
			return priceSteps;
		}
		
		String couponTitle = coupon!= null ? coupon.getCouponSample().getTitle() : null;
		String interestFreeName = interestFree != null ? interestFree.getInterestFreeName() : null;
		
		priceSteps.add(new PriceStepVO("商品总额", priceResult.getOriginalTotalPrice()));
		for (PrsPriceStepVO stepList : priceResult.getPriceStepVOs()) {
			if (stepList.getPriceChange().compareTo(BigDecimal.ZERO) == 0) {
				continue;
			}
			if (stepList.getPriceType().equals(PrsPriceTypeEnum.INTEREST_MEM_POSTAGE_FREE)) {
				if (StringUtils.isNotBlank(interestFreeName)) {
					priceSteps.add(new PriceStepVO(interestFreeName, stepList.getPriceChange()));
				} else {
					priceSteps.add(new PriceStepVO(stepList.getStepDesc(), stepList.getPriceChange()));
				}
			} else if (stepList.getPriceType().equals(PrsPriceTypeEnum.COUPON)) {
				if (StringUtils.isNoneBlank(couponTitle)) {
					priceSteps.add(new PriceStepVO(couponTitle, stepList.getPriceChange()));
				} else {
					priceSteps.add(new PriceStepVO(stepList.getStepDesc(), stepList.getPriceChange()));
				}
			} else {
				priceSteps.add(new PriceStepVO(stepList.getStepDesc(), stepList.getPriceChange()));
			}
		}
		return priceSteps;
	}
	
	public List<PrsAdditionalPackVO> getAddtionPackList(Map<PrsCampaignVO, List<PrsAdditionalPackVO>> additionalMap) {
		List<PrsAdditionalPackVO> addtionPackList = Lists.newArrayList();
		for (Entry<PrsCampaignVO, List<PrsAdditionalPackVO>> entry :  additionalMap.entrySet()) {
			addtionPackList.addAll(entry.getValue());
		}
		return addtionPackList;
	}
	
	/**
	  * 获取活动map
	 * key: groupId-productId
	 * @Methods Name getProdCampMap
	 * @Create In 2018-12-29 By Xiang.Gao
	 */
	public Map<String, PrsCampaignVO> getProdCampMap(List<PrsPriceStepVO> steps) {
		Map<String, PrsCampaignVO> campaignMap = new HashMap<>();
		if (CollectionUtils.isEmpty(steps)) {
			return campaignMap;
		}
		for (PrsPriceStepVO step : steps) {
			if (CollectionUtils.isEmpty(step.getAdditionalPackList())) {
				continue;
			}
			for (PrsAdditionalPackVO addpack : step.getAdditionalPackList()) {
				// 同一商品存在多个活动 key = sectionId + productId
				campaignMap.put(addpack.getGroupId() + "-" + addpack.getProductId(), step.getCampaign());
			}
		}
		return campaignMap;
	}
	 
	/** 抢购限量
	 * @return
	 */
	public Integer getPanicLeftAmount(String campCode, String memberCode, PsSpvVO spv) {
		CampaignStreamVO campaignStreamVO = new CampaignStreamVO();
		campaignStreamVO.setCampaignCode(campCode);
		campaignStreamVO.setSkuCode(spv.getSkuCode());
		campaignStreamVO.setMemberCode(memberCode);
		ServiceResp<Integer> resp = campaignLimitService.getPanicLeftAmount(campaignStreamVO);
		if (resp == null) {
			return 0;
		}
		// 库存为-1时表示不限量，则以原有库存为限量数
		return resp.getBean() == -1 ? spv.getCanUseQuantityOffLine() : resp.getBean();
	}
	
	/**
	 * 价格折扣限量
	 * @param campCode
	 * @param spv
	 * @return
	 */
	public Integer getPriceLeftAmount(String campCode, PsSpvVO spv) {
		CampaignStreamVO campaignStream = new CampaignStreamVO();
        campaignStream.setCampaignCode(campCode);
        campaignStream.setSkuCode(spv.getSkuCode());
        ServiceResp<Integer> resp = campaignLimitService.getPriceDiscountLeftAmount(campaignStream);
        if (resp == null || resp.isFailure()) {
        	return 0;
        }
        return resp.getBean() == -1 ? spv.getCanUseQuantityOffLine() : resp.getBean();
	}
	
	/**
	 *  价格折扣限量单个用户
	 * @param campCode
	 * @param memberCode
	 * @param spv
	 * @return
	 */
	public Integer getPriceUserLeftAmount(String campCode, String memberCode, PsSpvVO spv) {
		CampaignStreamVO campaignStream = new CampaignStreamVO();
        campaignStream.setCampaignCode(campCode);
        campaignStream.setMemberCode(memberCode);
        campaignStream.setSkuCode(spv.getSkuCode());
        ServiceResp<Integer> resp = campaignLimitService.getPriceDiscountPerUserLeftAmount(campaignStream);
        if (resp == null || resp.isFailure()) {
        	return 0;
        }
        return resp.getBean() == -1 ? spv.getCanUseQuantityOffLine() : resp.getBean();
	}
	
	/**
	 * 运费标签
	 * @return
	 */
	public List<Label> getPostageLabel(BigDecimal price, boolean hasOverSea, AccessWayEnum accessWay) {
		final List<Label> labels = Lists.newArrayList();
		if (price != null && !hasOverSea) {
			final Label label = new Label();
			label.setMode(LabelMode.POSTAGE);
			label.setType("免邮");
			label.setSummary("满" + PostageConstant.MEET_PRICE + "元免邮");
			label.setPostage(getPostage(price));
			labels.add(label);
		}
		return labels;
	}
	
	private PostageVO getPostage(BigDecimal price) {
		PostageVO postageVO = null;
		if (price != null) {
			BigDecimal postagePrice = PostageConstant.MEET_PRICE;
			boolean reach = price.compareTo(postagePrice) >= 0 ? true : false;
			StringBuilder linkDesc = new StringBuilder();
			if (reach) {
				linkDesc.append("已满足");
			} else {
				linkDesc.append("还差").append(PriceUtil.keepToCent(postagePrice.subtract(price))).append("元，去凑单");
			}
			postageVO = new PostageVO();
			postageVO.setFulfilled(reach);
			postageVO.setLinkDesc(linkDesc.toString());
		}
		return postageVO;
	}
	
	/**
	 * 新人价 SPV
	 * @param spvPriceSteps
	 * @return
	 */
	public List<Long> getNewcomerSpvIds(List<PrsSpvPriceResultVO> spvPriceSteps) {
		List<Long> newcomerSpvList = Lists.newArrayList();
		
		if (CollectionUtils.isEmpty(spvPriceSteps)) {
			return newcomerSpvList;
		}
		for (PrsSpvPriceResultVO prsPriceStep : spvPriceSteps) {
			List<PrsSpvPriceStepVO> priceStepVO = prsPriceStep.getSpvPriceStepVOS();
			if (CollectionUtils.isEmpty(priceStepVO)) {
				continue;
			}
			for (PrsSpvPriceStepVO step : priceStepVO) {
				if (PrsPriceTypeEnum.FISH_MAN.equals(step.getPriceType())) {
					newcomerSpvList.add(prsPriceStep.getSpvId());
				}
			}
		}
		return newcomerSpvList;
	}
	
	
}
