package com.thebeastshop.pegasus.channelservice.adapter.lesson;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.common.utils.DateUtil;
import com.thebeastshop.course.enums.FrontCourseSessionStatusEnum;
import com.thebeastshop.course.service.FrontBeastCourseService;
import com.thebeastshop.course.vo.FrontBeastCourseVO;
import com.thebeastshop.course.vo.FrontBeastSessionVO;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.adapter.price.PriceAdapter;
import com.thebeastshop.pegasus.channelservice.adapter.product.ProductAdapter;
import com.thebeastshop.pegasus.channelservice.exception.ChannelException;
import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.service.McPsSkuService;
import com.thebeastshop.pegasus.merchandise.vo.OpChannelVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import com.thebeastshop.price.service.PrsPriceService;
import com.thebeastshop.price.vo.*;
import com.thebeastshop.support.enums.lesson.CourseStatus;
import com.thebeastshop.support.enums.lesson.CourseTimeStatus;
import com.thebeastshop.support.file.FileConstant;
import com.thebeastshop.support.vo.lesson.*;
import com.thebeastshop.support.vo.product.ProductShareVO;
import com.thebeastshop.support.vo.product.VideoVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;

/**
 * Created by GFL on 2019/3/7.
 */
@Component
public class BeastLessonAdapter {

    @Autowired
    private McPsProductService mcPsProductService;

    @Autowired
    private McPsSkuService mcPsSkuService;

    @Autowired
    private PriceAdapter priceAdapter;

    @Autowired
    private PrsPriceService prsPriceService;

    @Autowired
    ProductAdapter productAdapter;
    


    /**
     * 获取spv和对应CourseSessionMap
     *
     * @param beastCourse
     * @return
     */
    public Map<Long, FrontBeastSessionVO> getSpvAndSessionMap(FrontBeastCourseVO beastCourse) {
        Map<Long, FrontBeastSessionVO> spvAndSessionMap = new HashMap<>();
        for (FrontBeastSessionVO frontBeastSessionVO : beastCourse.getBeastSessionList()) {
            spvAndSessionMap.put(frontBeastSessionVO.getSpvId(), frontBeastSessionVO);
        }
        return spvAndSessionMap;
    }

    public CourseInfo getCourseInfoByProduct(FrontBeastCourseVO beastCourse, String productCode, MemberVO member, OpChannelVO opChannel, AccessWayEnum accessWay) {


        Map<Long, FrontBeastSessionVO> sessionMap = getSpvAndSessionMap(beastCourse);
        MemberLevelEnum memberLevel = MemberLevelEnum.getEnumByCode(member.getMemberLevel());
        PsMatchCampaignCond cond = new PsMatchCampaignCond();
        cond.setMatchCampaign(true);
        cond.setChannelCode(opChannel.getCode());
        cond.setAccessWay(accessWay.getCode());
        cond.setMemberLevel(memberLevel.getCode());
        cond.setDiscount(memberLevel.getLevelAmountCondition());
        cond.setBdDiscount(member.getBdayDiscountRatio());
        PsProductVO product = mcPsProductService.getProductDetailByCode(opChannel.getCode(), productCode, cond);
        if (null == product) {
            throw new ChannelException(400, "商品不存在");
        }

        List<PsSkuVO> skus = mcPsSkuService.findBySkuCodes(product.getSkuCodes());
        PrsCalcProductDTO arg0 = priceAdapter.assemblePriceProduct(product, skus, member, opChannel, accessWay);

        PrsProductPriceResultVO priceResult = getPointExchangePriceResult(arg0, sessionMap, Boolean.FALSE);
        PrsProductPriceResultVO pointDeductResult = getPointExchangePriceResult(arg0, sessionMap, Boolean.TRUE);
        Map<Long,CoursePrice> spvCoursePriceMap = this.getCoursePriceMapBySpvID(product,beastCourse,priceResult,pointDeductResult);
        // 设置基础信息
        CourseInfo courseInfo = this.buildCourseInfo(product,beastCourse,spvCoursePriceMap);
        return courseInfo;
    }


    /**
     * 构建课程对象
     * @param product
     * @param beastCourse
     * @param spvCoursePriceMap
     * @return
     */
    private CourseInfo buildCourseInfo(PsProductVO product, FrontBeastCourseVO beastCourse,Map<Long,CoursePrice> spvCoursePriceMap) {
        // 获取限制对象
        CourseQuota courseQuota = new CourseQuota(beastCourse.getPerMemberLimitMax(), beastCourse.getPerMemberLimitMax() - beastCourse.getMemberReservedNum());
        List<CourseTime> courseTimes = this.getAllCourseTime4FrontBeastCourseVO(beastCourse, courseQuota,spvCoursePriceMap);
        String registerEndTime = this.getRegisterEndTime4FrontBeastCourseVO(beastCourse);
        CourseStock courseStock = this.getTotalStock(beastCourse);

        CourseInfo courseInfo = new CourseInfo();
        courseInfo.setCode(beastCourse.getFrontBeastCourseId().toString());
        courseInfo.setProductCode(product.getCode());
        courseInfo.setName(product.getNameCn());
        courseInfo.setVipLevel(beastCourse.getMemberLevels());
        courseInfo.setTips("课程类活动，报名结束后若不足开课人数，课程将取消并退款");
        this.setCityNameAndOpenTime(courseInfo,beastCourse);
        this.setCourseInfoStatusAndRemind(courseInfo, beastCourse);
        this.setCourseInfoImageAndVideo(courseInfo, product);
        courseInfo.setQuota(courseQuota);
        courseInfo.setStock(courseStock);
        courseInfo.setTimes(courseTimes);
        courseInfo.setRegisterEndTime(registerEndTime);
        courseInfo.setVariant(this.getBeastCourseVariant(courseTimes));
        courseInfo.setShare(getProductShareVO(courseInfo,product));
        return  courseInfo;
    }

    /**
     * 获取报名结束时间
     * @param beastCourse
     * @return
     */
    private String getRegisterEndTime4FrontBeastCourseVO(FrontBeastCourseVO beastCourse) {
        if(CollectionUtils.isEmpty(beastCourse.getBeastSessionList())){
            return "";
        }
        Date registerEndTime = null;
        for (FrontBeastSessionVO beastSession : beastCourse.getBeastSessionList()) {
            if(registerEndTime == null || registerEndTime.before(beastSession.getReserveEndTime())){
                registerEndTime = beastSession.getReserveEndTime();
            }
        }
        return DateUtil.format(registerEndTime, "MM月dd日 HH:mm");
    }

    /**
     * 设置城市和开始时间
     * @param courseInfo
     * @param beastCourse
     */
    private void setCityNameAndOpenTime(CourseInfo courseInfo, FrontBeastCourseVO beastCourse) {
        List<String> cityNames = Lists.newArrayList();
        List<String> openTimes = Lists.newArrayList();
        for (FrontBeastSessionVO beastSession : beastCourse.getBeastSessionList()) {
            if(!cityNames.contains(beastSession.getCityName())){
                cityNames.add(beastSession.getCityName());
            }
            String openTime = DateUtil.format(beastSession.getSessionStartTime(), "MM月dd日");
            if(!openTimes.contains(openTime)){
                openTimes.add(openTime);
            }
        }
        courseInfo.setCityNames(cityNames);
        courseInfo.setDate(openTimes);
    }

    /**
     * 获取课程所有场次
     * @param beastCourse
     * @param courseQuota
     * @param spvCoursePriceMap
     * @return
     */
    private List<CourseTime> getAllCourseTime4FrontBeastCourseVO(FrontBeastCourseVO beastCourse, CourseQuota courseQuota,Map<Long,CoursePrice> spvCoursePriceMap) {
        List<CourseTime> courseTimes = Lists.newArrayList();
        for (FrontBeastSessionVO beastSessionVO : beastCourse.getBeastSessionList()) {
            CourseTime courseTime = convertBeastSession2CourseTime(beastSessionVO, courseQuota);
            courseTime.setPrice(spvCoursePriceMap.get(beastSessionVO.getSpvId()));
            courseTimes.add(courseTime);
        }
        return courseTimes;
    }

    /**
     * 获取场次spvid对应价格
     * @param product
     * @param beastCourse
     * @param priceResult
     * @param pointDeductPriceResult
     * @return
     */
    private Map<Long,CoursePrice> getCoursePriceMapBySpvID(PsProductVO product ,FrontBeastCourseVO beastCourse ,PrsProductPriceResultVO priceResult,PrsProductPriceResultVO pointDeductPriceResult){
        Map<Long,BigDecimal> spvPriceMap = new HashMap<>();
        for (PrsSpvPriceResultVO prsSpvPriceResultVO : priceResult.getSpvPriceResultVOs()) {
            spvPriceMap.put(prsSpvPriceResultVO.getSpvId(),prsSpvPriceResultVO.getDiscountPrice());
        }
        Map<Long,BigDecimal> spvRawPriceMap = new HashMap<>();
        for (PsSpvVO spvVO : product.getSpvList()) {
            spvRawPriceMap.put(spvVO.getSpvId(),spvVO.getSalesPrice());
        }
        Map<Long,BigDecimal> spvPointPriceMap = new HashMap<>();
        for (PrsSpvPriceResultVO prsSpvPriceResultVO : pointDeductPriceResult.getSpvPriceResultVOs()) {
            spvPointPriceMap.put(prsSpvPriceResultVO.getSpvId(),prsSpvPriceResultVO.getDiscountPrice());
        }

        Map<Long,CoursePrice> resultMap = new HashMap<>();
        for (FrontBeastSessionVO beastSession : beastCourse.getBeastSessionList()) {
            CoursePrice coursePrice = new CoursePrice();
            coursePrice.setPoint(beastSession.getPoint().intValue());
            coursePrice.setDeductibleAmount(beastSession.getPointDeductPrice());
            coursePrice.setPointPrice(spvPointPriceMap.get(beastSession.getSpvId()));
            coursePrice.setRawPrice(spvRawPriceMap.get(beastSession.getSpvId()));
            coursePrice.setPrice(spvPriceMap.get(beastSession.getSpvId()));
            resultMap.put(beastSession.getSpvId(),coursePrice);
        }
        return  resultMap;
    }

    /**
     * 获取总库存
     *
     * @param beastCourse
     * @return
     */
    private CourseStock getTotalStock(FrontBeastCourseVO beastCourse) {
        Integer total = 0;
        Integer left = 0;
        for (FrontBeastSessionVO beastSessionVO : beastCourse.getBeastSessionList()) {

            total += beastSessionVO.getMaxMemberNum();
            switch (beastSessionVO.getCourseSessionStatus()){
                case PREVIEW:
                case ENROLLED_NOT_FULLY:
                case ENROLLED_FULLY:
                    left += beastSessionVO.getSessionRemainNum();
                    break;
            }
        }
        return new CourseStock(total, left);
    }

    /**
     * 设置图片和视频信息
     * @param courseInfo
     * @param product
     */
    private void setCourseInfoImageAndVideo(CourseInfo courseInfo, PsProductVO product) {
        // featureImage
        String featureImage = product.getPicList();
        if (StringUtils.isNotBlank(featureImage) && !featureImage.startsWith(FileConstant.BASE_URL)) {
            featureImage = FileConstant.BASE_URL + featureImage;
        }
        courseInfo.setFeatureImage(featureImage);
        // images
        List<String> images = Lists.newArrayList();
        for (String s : product.getPicDetailMuls()) {
            if (StringUtils.isNotBlank(s) && !s.startsWith(FileConstant.BASE_URL)) {
                String image = FileConstant.BASE_URL + s;
                images.add(image);
            }
        }
        courseInfo.setImages(images);
        // 视频
        List<VideoVO> videos = new ArrayList<VideoVO>();
        VideoVO video = new VideoVO();

        if (StringUtils.isNotEmpty(product.getVideo())) {
            video.setUrl(FileConstant.BASE_URL + product.getVideo());
        }
        if (StringUtils.isNotEmpty(product.getVideoCover())) {
            // 图片服务拼接
            video.setThumbnailUrl(
                    FileConstant.BASE_URL + product.getVideoCover() + "@4e_0o_0l_721h_690w_90q.jpg");
        }

        if (StringUtils.isNotEmpty(product.getVideo()) && StringUtils.isNotEmpty(product.getVideoCover())) {
            videos.add(video);
        }
        courseInfo.setVideos(videos);
    }

    /**
     * 转换对象为课程场次
     *
     * @param beastSession
     * @param courseQuota
     * @return
     */
    private CourseTime convertBeastSession2CourseTime(FrontBeastSessionVO beastSession, CourseQuota courseQuota) {
        CourseTime courseTime = new CourseTime();
        courseTime.setCode(beastSession.getBeastSessionId().toString());
        courseTime.setName(beastSession.getProdName());
        courseTime.setCityName(beastSession.getCityName());
        courseTime.setAddress(beastSession.getDetailAddress());
        StringBuffer sb = new StringBuffer();
        sb.append(DateUtil.format(beastSession.getSessionStartTime(), "YYYY.MM.dd HH:mm"));
        sb.append("~");
        sb.append(DateUtil.format(beastSession.getSessionEndTime(), "HH:mm"));
        courseTime.setOpenTime(sb.toString());
        courseTime.setOpenSill(beastSession.getMinMemberNum());
        courseTime.setRegisterTime(DateUtil.format(beastSession.getReserveStartTime(), "MM月dd日 HH:mm"));
        courseTime.setStock(new CourseStock(beastSession.getMaxMemberNum(), beastSession.getSessionRemainNum()));
        courseTime.setSpvId(beastSession.getSpvId());
        courseTime.setSkuCode(beastSession.getSkuCode());
        courseTime.setQuota(courseQuota);
        courseTime.setStatus(convertCourseTimeStatus4FrontCourseSessionStatus(beastSession.getCourseSessionStatus()));
        return courseTime;
    }

    /**
     * 场次状态转换
     *
     * @param frontCourseSessionStatusEnum
     * @return
     */
    private CourseTimeStatus convertCourseTimeStatus4FrontCourseSessionStatus(FrontCourseSessionStatusEnum frontCourseSessionStatusEnum) {
        CourseTimeStatus courseTimeStatus = null;
        switch (frontCourseSessionStatusEnum) {
            case PREVIEW:
                courseTimeStatus = CourseTimeStatus.NOTICE;
                break;
            case ENROLLED_FULLY:
            case ENROLLED_NOT_FULLY:
                courseTimeStatus = CourseTimeStatus.REGISTER;
                break;
            case END:
                courseTimeStatus = CourseTimeStatus.WAITING_CLASS;
                break;
        }
        return courseTimeStatus;
    }

    /**
     * 设置课程状态和提醒信息
     *
     * @param courseInfo
     * @param beastCourse
     */
    private void setCourseInfoStatusAndRemind(CourseInfo courseInfo, FrontBeastCourseVO beastCourse) {
        CourseStatus courseStatus = null;
        Boolean remind = Boolean.FALSE;
        switch (beastCourse.getCourseListStatus()) {
            case ADVANCE_REMINDER_OPENED:
                remind = Boolean.TRUE;
                courseStatus = CourseStatus.NOTICE;
                break;
            case ADVANCE_REMINDER_UNOPENED:
                remind = Boolean.FALSE;
                courseStatus = CourseStatus.NOTICE;
                break;
            case ENROLLED_FULL:
                courseStatus = CourseStatus.REGISTER;
                break;
            case ENROLLED_NOFULL:
                courseStatus = CourseStatus.REGISTER;
                break;
            case REVIEW:
                courseStatus = CourseStatus.OVER;
                break;
        }
        courseInfo.setStatus(courseStatus);
        courseInfo.setRemind(remind);
    }

    private PrsProductPriceResultVO getPointExchangePriceResult(PrsCalcProductDTO arg0, Map<Long, FrontBeastSessionVO> sessionMap, Boolean pointDeduct) {
        PrsProductVO prsProductVO = arg0.getProductVOList().get(0);
        for (PrsSpvVO prsSpvVO : prsProductVO.getSpvVOList()) {
            FrontBeastSessionVO beastSession = sessionMap.get(prsSpvVO.getSpvId());
            prsSpvVO.setPointExchange(pointDeduct);
            prsSpvVO.setPoints(beastSession.getPoint().intValue());
            prsSpvVO.setPointDeductPrice(beastSession.getPointDeductPrice());
        }
        PrsProductPriceResultVO prsProductPriceResultVO = prsPriceService.calculatePointDeduct4ProductDetail(arg0);
        return prsProductPriceResultVO;
    }

    /**
     * 获取规格信息
     * @param times
     * @return
     */
    private  List<CityVariant>  getBeastCourseVariant(List<CourseTime> times){

        if(CollectionUtils.isEmpty(times)){
            return Lists.newArrayList();
        }

        Map<String ,CityVariant> cityMap = Maps.newHashMap();
        for (CourseTime time : times) {
            CityVariant cityVariant = cityMap.get(time.getCityName());
            if(cityVariant ==null){
                cityVariant = new CityVariant();
                cityVariant.setCityName(time.getCityName());
                OpenTimeVariant openTimeVariant = new OpenTimeVariant(time.getOpenTime(),Lists.newArrayList(new AddressVariant(time.getAddress(),time.getSpvId())));
                cityVariant.setData(Lists.newArrayList(openTimeVariant));
                cityMap.put(time.getCityName(),cityVariant);
            }else {
                Boolean isNewOpenTime = Boolean.TRUE;
                for (OpenTimeVariant openTimeVariant : cityVariant.getData()) {
                    if(openTimeVariant.getOpenTime().equalsIgnoreCase(time.getOpenTime())){
                        isNewOpenTime = Boolean.FALSE;
                        openTimeVariant.getData().add(new AddressVariant(time.getAddress(),time.getSpvId()));
                    }
                }
                if(isNewOpenTime){
                    OpenTimeVariant openTimeVariant = new OpenTimeVariant(time.getOpenTime(),Lists.newArrayList(new AddressVariant(time.getAddress(),time.getSpvId())));
                    cityVariant.getData().add(openTimeVariant);
                }
            }
        }

        List<CityVariant> cityVariants = Lists.newArrayList();
        List<String> cityNames = Lists.newArrayList();
        for (CourseTime time : times) {
            if(!cityNames.contains(time.getCityName())) {
                cityNames.add(time.getCityName());
                cityVariants.add(cityMap.get(time.getCityName()));
            }
        }
        return cityVariants;

    }

    /**
     * 获取商品分享信息
     * @param courseInfo
     * @param product
     * @return
     */
    private ProductShareVO getProductShareVO( CourseInfo courseInfo , PsProductVO product) {
        ProductShareVO productShareVO = new ProductShareVO();
        // name
        String name =product.getNameCn();
        // summary
        String summary = product.getDescription();
        productShareVO.setTitle((StringUtils.isNotBlank(summary) ? "「" + summary + "」" : "") + name);
        String desc = productAdapter.getRecommendDesc(Long.parseLong(product.getId()));
        productShareVO.setContent((StringUtils.isNotBlank(desc) ? desc : ""));
        if (null != product && StringUtils.isNotBlank(product.getCode())) {
            productShareVO.setLink("https://m.thebeastshop.com/product/" + product.getCode());
        } else {
            productShareVO.setLink("");
        }
        productShareVO.setImage(courseInfo.getFeatureImage());

        return productShareVO;
    }

}
