package com.thebeastshop.pegasus.channelservice.adapter.interest;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.thebeastshop.common.utils.NumberUtil;
import com.thebeastshop.support.vo.cart.CartBenefitVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.thebeastshop.common.enums.AccessWayEnum;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.pegasus.merchandise.IService.IInterestGoodsService;
import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.exception.InterestException;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.service.McPsSpvService;
import com.thebeastshop.pegasus.merchandise.vo.PsAttributeValuePairVO;
import com.thebeastshop.pegasus.merchandise.vo.PsBrandVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSkuCombinationVO;
import com.thebeastshop.pegasus.merchandise.vo.PsSpvVO;
import com.thebeastshop.scm.cond.InterestGoodsCond;
import com.thebeastshop.scm.po.InterestGoods;
import com.thebeastshop.scm.po.InterestGoodsSku;
import com.thebeastshop.scm.vo.interest.InterestGoodsGetVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsProvideVO;
import com.thebeastshop.scm.vo.interest.InterestGoodsVO;
import com.thebeastshop.support.exception.WrongArgException;
import com.thebeastshop.support.file.FileConstant;
import com.thebeastshop.support.util.PriceUtil;
import com.thebeastshop.support.vo.product.ProductDimensionVO;
import com.thebeastshop.support.vo.product.ProductVariantVO;
import com.thebeastshop.support.vo.product.SpvVO;

/**
 * 权益适配 Created by feilong.gao on 2017/1/16.
 */
@Service
public class InterestAdapter {

	private final static Logger logger = LoggerFactory.getLogger(InterestAdapter.class);

	@Autowired
	private IInterestGoodsService iInterestGoodsService;

	@Autowired
	private McPsProductService mcPsProductService;

	@Autowired
	private McPsSpvService mcPsSpvService;

	/**
	 * 获取权益信息
	 * 
	 * @param interestId
	 * @return
	 */
	public InterestGoods getInterestInfo(String interestId) {
		InterestGoods interestGoods = iInterestGoodsService.selectInterestById(Integer.parseInt(interestId));
		return interestGoods;
	}

	/**
	 * 获购物车权益信息
	 *
	 * @param interestId
	 * @return
	 */
	public CartBenefitVO getCartBenefitVO(String interestId) {
		InterestGoodsVO interestVO = iInterestGoodsService.selectInterestGoodsSku(Integer.parseInt(interestId));
		if (null == interestVO) {
			throw new WrongArgException("权益信息为空");
		}
		CartBenefitVO cartBenefitVO = new CartBenefitVO();
		cartBenefitVO.setId(interestId);
		cartBenefitVO.setCashStartTime(interestVO.getInterestGoods().getCashStartTime());
		cartBenefitVO.setCashEndTime(interestVO.getInterestGoods().getCashEndTime());
		cartBenefitVO.setThresholdPrice(interestVO.getInterestGoods().getCashThreshold());
		cartBenefitVO.setIsThreshold(!NumberUtil.isNullOrZero(cartBenefitVO.getThresholdPrice()));
		if (CollectionUtils.isNotEmpty(interestVO.getInterestGoodsSkuList())) {
			cartBenefitVO.setDeductionPrice(interestVO.getInterestGoodsSkuList().get(0).getDeductionPrice());
		}

		return cartBenefitVO;
	}

	/**
	 * 领取权益
	 *
	 * @param interestId
	 */
	public InterestGoodsGetVO receiveInterest(Long memberId, String interestId) {
		logger.info("会员：{} 领取权益：{}", memberId, interestId);
		InterestGoodsGetVO interestGoodsGetVO = null;
		try {
			interestGoodsGetVO = iInterestGoodsService.getInterestGoods(Integer.parseInt(memberId.toString()),
					Integer.parseInt(interestId));
		} catch (InterestException e) {
			throw new WrongArgException(e.getLocalizedMessage());
		} catch (Exception e) {
			throw new WrongArgException("领取权益异常");
		}
		return interestGoodsGetVO;
	}

	/**
	 * 修改权益sku
	 *
	 * @param memberId
	 * @param interestId
	 * @param oldSpvId
	 * @param newSpvId
	 * @return
	 */
	public Boolean updInterestSku(Long memberId, String interestId, Long oldSpvId, Long newSpvId) {
		Map<String, InterestGoodsSku> spvMap = getSpvByInterest(interestId);
		InterestGoodsSku oldInterestSku = spvMap.get(oldSpvId.toString());
		InterestGoodsSku newInterestSku = spvMap.get(newSpvId.toString());
		if (null == oldInterestSku || null == newInterestSku) {
			throw new WrongArgException("非法的权益规格");
		}
		boolean flag = iInterestGoodsService.changeInterestGoodsSku(Integer.parseInt(memberId.toString()),
				oldInterestSku.getId(), newInterestSku.getId());
		if (!flag) {
			throw new WrongArgException("修改权益规格失败");
		}
		return true;
	}

	/**
	 * 权益是否可选规格
	 *
	 * @param interestId
	 * @return
	 */
	public Boolean hasVariants(String interestId) {
		InterestGoodsVO interestVO = iInterestGoodsService.selectInterestGoodsSku(Integer.parseInt(interestId));
		if (null == interestVO) {
			throw new WrongArgException("权益信息为空");
		}
		if (CollectionUtils.isNotEmpty(interestVO.getInterestGoodsSkuList())) {
			return interestVO.getInterestGoodsSkuList().size() > 1;
		}
		return false;
	}

	/**
	 * 获取权益商品规格
	 *
	 * @param interestId
	 * @return
	 */
	public ProductVariantVO getInterestVariant(String chnCode, String interestId, AccessWayEnum accessWay) {
		ProductVariantVO result = new ProductVariantVO();
		InterestGoodsVO interestGoodsVO = iInterestGoodsService.selectInterestGoodsSku(Integer.parseInt(interestId));
		String productCode = interestGoodsVO.getInterestGoods().getProdCode();
		PsMatchCampaignCond cond = PsMatchCampaignCond.buildCond(accessWay.getCode(), MemberLevelEnum.NORMAL.getCode(),
				chnCode);
		cond.setDiscount(MemberLevelEnum.NORMAL.getLevelAmountCondition());
		PsProductVO product = mcPsProductService.findProdByCode(productCode);
		if (product != null) {
			List<InterestGoodsSku> interestSkus = interestGoodsVO.getInterestGoodsSkuList();
			Map<String, Integer> skuCodes = new HashMap<>();
			List<PsSpvVO> psSpvs = Lists.newArrayList();
			List<PsSpvVO> allSpvs = mcPsSpvService.findByProdId(Long.parseLong(product.getId()));
			for (InterestGoodsSku interestSku : interestSkus) {
				skuCodes.put(interestSku.getSkuCode(), (interestSku.getQuantity() - interestSku.getReceiveQuantity()));
			}
			for (PsSpvVO psSpvVO : allSpvs) {
				if (skuCodes.containsKey(psSpvVO.getSkuCode())) {
					psSpvs.add(psSpvVO);
				}
			}
			if (CollectionUtils.isNotEmpty(psSpvs)) {
				Map<String, ProductDimensionVO> pdMap = Maps.newLinkedHashMap(); // 规格信息
				List<SpvVO> spvVOs = Lists.newArrayList();
				for (PsSpvVO spv : psSpvs) {
					SpvVO spvVO = this.convertSpvVO4PsSpvVO(spv, product);
					spvVO.setLeft(skuCodes.get(spv.getSkuCode()));
					List<Integer> group = Lists.newArrayList();
					List<PsAttributeValuePairVO> attrList = spv.getAttributeValuePairList();
					if (CollectionUtils.isNotEmpty(attrList)) {
						// 设置规格信息
						for (PsAttributeValuePairVO attr : attrList) {
							if (attr != null && StringUtils.isNotBlank(attr.getAttrName())
									&& StringUtils.isNotBlank(attr.getValue())) {
								String attrName = attr.getAttrName();
								String attrValue = attr.getValue();
								ProductDimensionVO pd = pdMap.get(attrName);
								if (pd == null) {
									pd = new ProductDimensionVO();
									pd.setChoices(new ArrayList<String>());
									pd.setName(attrName);
									pdMap.put(attrName, pd);
								}
								List<String> choices = pd.getChoices();
								if (!choices.contains(attrValue)) {
									choices.add(attrValue);
								}
								group.add(choices.indexOf(attrValue));
							}
						}
					}
					spvVO.setGroup(group);
					spvVOs.add(spvVO);
				}

				Collection<ProductDimensionVO> dimensionVos = pdMap.values();
				// 当规格为一个的时候不显示规格信息
				if (CollectionUtils.isNotEmpty(dimensionVos) && dimensionVos.size() == 1) {
					Iterator<ProductDimensionVO> iterator = dimensionVos.iterator();
					ProductDimensionVO vo = iterator.next();
					List<String> choices = vo.getChoices();
					if (CollectionUtils.isNotEmpty(choices) && choices.size() == 1) {
						dimensionVos = null;
					}
				}

				result.setSpvs(spvVOs);
				result.setDimensions(dimensionVos);
			}
		}
		return result;
	}

	/**
	 * 获取spv对应的权益对象
	 *
	 * @param interestId
	 * @return
	 */
	private Map<String, InterestGoodsSku> getSpvByInterest(String interestId) {
		Map<String, InterestGoodsSku> spvInterestMap = null;
		InterestGoodsVO interestGoodsVO = iInterestGoodsService.selectInterestGoodsSku(Integer.parseInt(interestId));
		PsProductVO product = mcPsProductService.findProdByCode(interestGoodsVO.getInterestGoods().getProdCode());
		if (product != null) {
			spvInterestMap = new HashMap<>();
			Map<String, InterestGoodsSku> skuMaps = new HashMap<>();
			for (InterestGoodsSku interestSku : interestGoodsVO.getInterestGoodsSkuList()) {
				skuMaps.put(interestSku.getSkuCode(), interestSku);
			}
			for (PsSpvVO psSpvVO : product.getSpvList()) {
				spvInterestMap.put(psSpvVO.getSpvId().toString(), skuMaps.get(psSpvVO.getSkuCode()));
			}
		}
		return spvInterestMap;
	}

	private SpvVO convertSpvVO4PsSpvVO(PsSpvVO spv, PsProductVO product) {
		SpvVO spvVo = new SpvVO();
		spvVo.setId(spv.getSpvId());
		// featureImage
		String featureImage = product.getPicList();
		if (org.apache.commons.lang.StringUtils.isNotBlank(featureImage)
				&& !featureImage.startsWith(FileConstant.BASE_URL)) {
			featureImage = FileConstant.BASE_URL + featureImage;
		}
		spvVo.setImage(featureImage);
		// brand
		List<PsBrandVO> brands = product.getBrands();
		if (CollectionUtils.isNotEmpty(brands)) {
			PsBrandVO brand = brands.get(0);
			if (brand != null) {
				SpvVO.BrandVO brandVO = new SpvVO.BrandVO();
				brandVO.setId(brand.getBrandId());
				brandVO.setName(brand.getBrandNameCN());
				spvVo.setBrand(brandVO);
			}
		}
		// name
		spvVo.setName(product.getNameCn());
		// description
		StringBuilder sb = new StringBuilder();
		List<PsAttributeValuePairVO> attrList = spv.getAttributeValuePairList();
		if (CollectionUtils.isNotEmpty(attrList)) {
			for (PsAttributeValuePairVO attr : attrList) {
				if (attr != null && org.apache.commons.lang.StringUtils.isNotBlank(attr.getAttrName())
						&& org.apache.commons.lang.StringUtils.isNotBlank(attr.getValue())) {
					sb.append(attr.getAttrName()).append(" : ").append(attr.getValue()).append(" ");
				}
			}
		}
		spvVo.setSpvDesc(sb.toString());
		// price
		spvVo.setRawPrice(spv.getSalesPrice());
		spvVo.setPrice(BigDecimal.ZERO);
		// product code
		spvVo.setProductCode(spv.getProductCode());
		// skuCodes
		List<String> skuCodes = Lists.newArrayList();
		if (spv.getCombined()) {
			for (PsSkuCombinationVO sku : spv.getSkuCombinationList()) {
				skuCodes.add(sku.getSubSkuCode());
			}
		} else {
			skuCodes.add(spv.getSkuCode());
		}
		spvVo.setSkuCodes(skuCodes);
		// 最小起售数
		spvVo.setMinAmount(1);
		// XXX:预售信息()海淘商品或者组合商品不显示预售信息
		spvVo.setPresell(null);
		spvVo.setDiscount(null);
		return spvVo;
	}

	/**
	 * 获取权益信息
	 * 
	 * @param cand
	 * @return
	 */
	public List<InterestGoodsProvideVO> getProvideRecord(InterestGoodsCond cand) {
		List<InterestGoodsProvideVO> list = iInterestGoodsService.getProvideRecord(cand);
		return list;
	}
}