package com.thebeastshop.pegasus.channelservice.adapter.exchange;

import com.google.common.collect.Lists;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.common.utils.DateUtil;
import com.thebeastshop.common.utils.NumberUtil;
import com.thebeastshop.exchange.enums.ExchgConfigStatusEnum;
import com.thebeastshop.exchange.enums.ExchgTypeEnum;
import com.thebeastshop.exchange.vo.ExchgSpuConfigVO;
import com.thebeastshop.pegasus.merchandise.cond.PsMatchCampaignCond;
import com.thebeastshop.pegasus.merchandise.service.McPsProductService;
import com.thebeastshop.pegasus.merchandise.vo.PsBrandVO;
import com.thebeastshop.pegasus.merchandise.vo.PsProductVO;
import com.thebeastshop.stock.service.SStockService;
import com.thebeastshop.stock.vo.SSkuPointExchangeStockVO;
import com.thebeastshop.support.enums.CampaignState;
import com.thebeastshop.support.file.FileConstant;
import com.thebeastshop.support.vo.exchange.ExchangeGroupVO;
import com.thebeastshop.support.vo.exchange.ExchangeSecKill;
import com.thebeastshop.support.vo.exchange.ExchangeSimpleProductGoods;
import com.thebeastshop.support.vo.product.SpvVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author feilong.gao
 * @Date 2018/5/23 16:52.
 * @Description
 */
@Component
public class ExchangeProductGoodsAdapter {

    @Autowired
    private McPsProductService mcPsProductService;

    @Autowired
    private SStockService stockService;

    public List<ExchangeSimpleProductGoods> getExchangeSimpleProduct(List<ExchgSpuConfigVO> exchangeList,
                                                                     String channelCode, Integer accessWayId) {

        if (CollectionUtils.isEmpty(exchangeList)) {
            return null;
        }

        List<String> productCodes = Lists.newArrayList();
        for (ExchgSpuConfigVO simple : exchangeList) {
            productCodes.add(simple.getReferenceCode());
        }
        MemberLevelEnum memberLevel = MemberLevelEnum.NORMAL;
        PsMatchCampaignCond cond = new PsMatchCampaignCond();
        cond.setChannelCode(channelCode);
        cond.setMatchCampaign(true);
        cond.setAccessWay(accessWayId);
        cond.setMemberLevel(memberLevel.getCode());
        cond.setDiscount(memberLevel.getLevelAmountCondition());

        List<PsProductVO> products = mcPsProductService.findTopicByCodes(channelCode, productCodes, cond, 1,
                productCodes.size());

        return convertProduct2ExchangeSimpleList(products, exchangeList);
    }

    /**
     * 商品信息转换为ExchangeGoods
     *
     * @param products
     * @return
     */
    private List<ExchangeSimpleProductGoods> convertProduct2ExchangeSimpleList(List<PsProductVO> products,
                                                                               List<ExchgSpuConfigVO> exchangeList) {
        if (CollectionUtils.isEmpty(products)) {
            return null;
        }
        List<ExchangeSimpleProductGoods> targets = Lists.newArrayList();
        Map<String, Integer> stockMap = getExchangeProductStockMap(products);
        Map<String,PsProductVO> prodMap = new HashMap<>();
        for (PsProductVO product : products) {
            prodMap.put(product.getCode(),product);
        }

        for (ExchgSpuConfigVO exchgSpuConfigVO : exchangeList) {
            String productCode = exchgSpuConfigVO.getReferenceCode();
            ExchangeSimpleProductGoods simpleProductGoods = this.buildExchangeSimpleProduct(prodMap.get(productCode),
                    exchgSpuConfigVO, stockMap.get(productCode));
            targets.add(simpleProductGoods);
        }
        return targets;
    }

    /**
     * 获取兑换商品库存信息
     *
     * @param products
     * @return
     */
    private Map<String, Integer> getExchangeProductStockMap(List<PsProductVO> products) {
        Map<String, Integer> prodStockMap = new HashMap<>();
        if (CollectionUtils.isEmpty(products))
            return prodStockMap;
        List<String> skuCodes = Lists.newArrayList();
        for (PsProductVO product : products) {
            skuCodes.addAll(product.getSkuCodes());
        }
        Map<String, Integer> stockMap = new HashMap<>();
        ServiceResp<List<SSkuPointExchangeStockVO>> resp = stockService.getPointExchangeStocks(ExchgTypeEnum.SPU,
                skuCodes);
        if (resp.isSuccess() && CollectionUtils.isNotEmpty(resp.getBean())) {
            for (SSkuPointExchangeStockVO exchangeStock : resp.getBean()) {
                stockMap.put(exchangeStock.getReferenceCode(), exchangeStock.getCanUseQuantity());
            }
        }

        for (PsProductVO product : products) {
            Integer prodStock = 0;
            for (String skuCode : product.getSkuCodes()) {
                prodStock += stockMap.get(skuCode) == null ? 0 : stockMap.get(skuCode);
            }
            prodStockMap.put(product.getCode(), prodStock);
        }
        return prodStockMap;
    }

    /**
     * 商品对象构建
     *
     * @param product
     * @return
     */
    private ExchangeSimpleProductGoods buildExchangeSimpleProduct(PsProductVO product, ExchgSpuConfigVO spuConf,
                                                                  Integer stock) {
        ExchangeSimpleProductGoods basic = new ExchangeSimpleProductGoods();
        basic.setOnline(ExchgConfigStatusEnum.ONLINE.equals(spuConf.getStatus()));
        basic.setSecKill(spuConf.getSpikeMark());
        basic.setMode(spuConf.getExchangeMode().name());
        basic.setStartTime(spuConf.getBeginTime());
        basic.setId(spuConf.getId().intValue());
        List<Integer> levels = new ArrayList<>();
        for (MemberLevelEnum levelEnum : spuConf.getApplyMemberLevel()) {
            levels.add(levelEnum.getCode());
        }
        basic.setCode(spuConf.getReferenceCode());
        basic.setLevelIds(levels);
        basic.setPoint(spuConf.getExchgAmount());
        if(null != spuConf.getExchgPrice() && null != product.getListPrice()){
            basic.setExchangePrice(spuConf.getOrignalPrice().setScale(2,BigDecimal.ROUND_HALF_UP).subtract(spuConf.getExchgPrice().setScale(2, BigDecimal.ROUND_HALF_UP)));
        }
        basic.setTotal(spuConf.getTotalStock());
        if (NumberUtil.isNullOrZero(stock) || stock < 0) {
            basic.setSoldOut(Boolean.TRUE);
        }
        basic.setPrice(spuConf.getOrignalPrice());
        if (product != null) {
            basic.setProductCode(product.getCode());
            basic.setName(product.getNameCn());
            basic.setSummary(product.getDescription());
            // featureImage
            String featureImage = product.getPicList();
            if (StringUtils.isNotBlank(featureImage) && !featureImage.startsWith(FileConstant.BASE_URL)) {
                featureImage = FileConstant.BASE_URL + featureImage;
            }
            basic.setFeatureImage(featureImage);
            // images
            List<String> images = product.getPicDetailMuls();
            for (int i = 0; i < images.size(); i++) {
                if (StringUtils.isNotBlank(images.get(i)) && !images.get(i).startsWith(FileConstant.BASE_URL)) {
                    String image = FileConstant.BASE_URL + images.get(i);
                    images.set(i, image);
                }
            }
            basic.setImages(images);

            // brand
            if (product.getBrands().size() > 0) {
                PsBrandVO brand = product.getBrands().get(0);
                if (brand != null) {
                    SpvVO.BrandVO brandVO = new SpvVO.BrandVO();
                    brandVO.setId(brand.getBrandId());
                    brandVO.setName(brand.getBrandName());
                    basic.setBrand(brandVO);
                }
            }
        }
        return basic;
    }

    /**
     * 构建商品兑换数据，设置商品和秒杀列表
     *
     * @param group
     * @param simpleProductGoods
     */
    public void buildExchageSupGoodsSetProductAndSecKillList(ExchangeGroupVO group, List<ExchangeSimpleProductGoods> simpleProductGoods) {
        if (CollectionUtils.isNotEmpty(simpleProductGoods)) {
            List<ExchangeSimpleProductGoods> products = Lists.newArrayList();

            Map<Date, ExchangeSecKill> secKillMap = new HashMap<>();
            for (ExchangeSimpleProductGoods simpleProductGood : simpleProductGoods) {
                if (simpleProductGood.getSecKill()) {

                    if (!secKillMap.containsKey(simpleProductGood.getStartTime())) {
                        ExchangeSecKill secKill = new ExchangeSecKill();
                        secKill.setExchangeGoods(Lists.newArrayList(simpleProductGood));
                        secKill.setStartTime(simpleProductGood.getStartTime());
                        secKillMap.put(simpleProductGood.getStartTime(), secKill);
                    } else {
                        secKillMap.get(simpleProductGood.getStartTime()).getExchangeGoods().add(simpleProductGood);
                    }
                } else {
                    products.add(simpleProductGood);
                }
            }
            // 设置兑换商品和抢购商品
            group.setProducts(products);
            if (MapUtils.isEmpty(secKillMap)) {
                group.setSecKills(Lists.<ExchangeSecKill>newArrayList());
            } else {
                List<ExchangeSecKill> secKills = Lists.newArrayList(secKillMap.values());
                buildSetSecKilTitleAndState(secKills);
                group.setSecKills(secKills);
            }
        }
    }

    private void buildSetSecKilTitleAndState(List<ExchangeSecKill> secKills) {
        if (CollectionUtils.isEmpty(secKills)) return;

        Collections.sort(secKills, new Comparator<ExchangeSecKill>() {
            @Override
            public int compare(ExchangeSecKill o1, ExchangeSecKill o2) {
                if (o1.getStartTime().after(o2.getStartTime())) {
                    return 1;
                }
                return -1;
            }
        });

        ExchangeSecKill nowSecKill = null;
        for (ExchangeSecKill secKill : secKills) {
            secKill.setState(getSecKillState(secKill.getStartTime()));
            secKill.setTitle(getSecKillTitle(secKill.getStartTime()));
            if (secKill.getState().equals(CampaignState.ING)) {
                nowSecKill = secKill;
            }
        }
        if (null != nowSecKill) {
            nowSecKill.setState(CampaignState.NOW);
        }
    }

    private CampaignState getSecKillState(Date startTime) {
        Date nowDate = DateUtil.getNow();
        if (nowDate.after(startTime)) {
            return CampaignState.ING;
        } else {
            return CampaignState.SOON;
        }
    }


    private String getSecKillTitle(Date startTime) {
        Date nowDate = DateUtil.getNow();
        String killTitle = "";
        if(DateUtils.isSameDay(nowDate,startTime)){
            killTitle = DateUtil.format(startTime, "HH:mm");
        }else if(DateUtils.isSameDay(DateUtils.addDays(nowDate,1),startTime)){
            killTitle = DateUtil.format(startTime, "明日HH:mm");
        }else{
            killTitle = DateUtil.format(startTime, "dd日HH:mm");
        }
        return killTitle;
    }

}
