package com.thebeastshop.pegasus.channelservice.adapter.exchange;

import com.google.common.collect.Lists;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.common.utils.NumberUtil;
import com.thebeastshop.coupon.dto.CpCouponSetCondDTO;
import com.thebeastshop.coupon.service.CpCouponSampleService;
import com.thebeastshop.coupon.service.CpCouponSetService;
import com.thebeastshop.coupon.vo.CpCouponSampleVO;
import com.thebeastshop.coupon.vo.CpCouponSetVO;
import com.thebeastshop.exchange.enums.ExchgConfigStatusEnum;
import com.thebeastshop.exchange.enums.ExchgTypeEnum;
import com.thebeastshop.exchange.vo.ExchgSpuConfigVO;
import com.thebeastshop.stock.service.SStockService;
import com.thebeastshop.stock.vo.SSkuPointExchangeStockVO;
import com.thebeastshop.support.vo.exchange.ExchangeCoupon;
import com.thebeastshop.support.vo.exchange.ExchangeGoods;
import com.thebeastshop.support.vo.exchange.ExchangeGroupCouponGoods;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author feilong.gao
 * @Date 2018/12/13 15:42.
 * @Description 兑换套券对象
 */
@Component
public class ExchangeGroupCouponGoodsAdapter {

    @Autowired
    private CpCouponSetService cpCouponSetService;

    @Autowired
    private CpCouponSampleService cpCouponSampleService;

    @Autowired
    private SStockService stockService;


    /**
     * 获取套券的兑换商品信息
     *
     * @param exchangeList
     * @return
     */
    public List<? extends ExchangeGoods> getGroupCouponExchangeGoods(List<ExchgSpuConfigVO> exchangeList) {

        if (CollectionUtils.isEmpty(exchangeList)) {
            return null;
        }

        List<Long> ids = Lists.newArrayList();
        for (ExchgSpuConfigVO simple : exchangeList) {
            ids.add(Long.parseLong(simple.getReferenceCode()));
        }
        Map<String, Integer> stockMap = getExchangeCouponStockMap(ids);
        Map<Long, CpCouponSetVO> couponSetMap = getExchangeCouponSetMap(ids);
        Map<Long, List<ExchangeCoupon>> exchangeGroupCouponMap = buildGroupCouponExchangeCouponMap(couponSetMap);
        return convertGroupCoupon2ExchangeGoodsList(exchangeGroupCouponMap, couponSetMap, exchangeList, stockMap);
    }


    /**
     * 获取兑换套券map
     *
     * @param groupCouponIds
     * @return
     */
    private Map<Long, CpCouponSetVO> getExchangeCouponSetMap(List<Long> groupCouponIds) {
        Map<Long, CpCouponSetVO> couponSetMap = new HashMap<>();
        if (CollectionUtils.isEmpty(groupCouponIds)) return couponSetMap;

        CpCouponSetCondDTO cond = new CpCouponSetCondDTO();
        cond.setIds(groupCouponIds);
        cond.setPagenum(groupCouponIds.size());
        List<CpCouponSetVO> sampleList = cpCouponSetService.listCouponSet(cond);
        if (CollectionUtils.isEmpty(sampleList)) return couponSetMap;

        for (CpCouponSetVO cpCouponSetVO : sampleList) {
            couponSetMap.put(cpCouponSetVO.getId(), cpCouponSetVO);
        }
        return couponSetMap;
    }

    /**
     * 构建套券兑换优惠券子劵对象
     *
     * @param couponSetMap
     * @return
     */
    private Map<Long, List<ExchangeCoupon>> buildGroupCouponExchangeCouponMap(Map<Long, CpCouponSetVO> couponSetMap) {
        Map<Long, List<ExchangeCoupon>> groupCouponMap = new HashMap<>();
        if (MapUtils.isEmpty(couponSetMap)) return groupCouponMap;

        Map<Long, ExchangeCoupon> exchangeCouponMap = new HashMap<>();
        List<CpCouponSampleVO> sampleVoList = getCpCouponSampleVOS(couponSetMap);
        if (CollectionUtils.isEmpty(sampleVoList)) return groupCouponMap;
        for (CpCouponSampleVO sampleVO : sampleVoList) {
            ExchangeCoupon exchangeCoupon = new ExchangeCoupon();
            exchangeCoupon.setTitle(sampleVO.getTitle());
            exchangeCoupon.setDescription(sampleVO.getDescription());
            exchangeCoupon.setThresholdDesc(sampleVO.getProductDesc());
            exchangeCoupon.setNote(sampleVO.getNote());
            exchangeCoupon.setStartTime(sampleVO.getStartTime());
            exchangeCoupon.setExpireTime(sampleVO.getExpireTime());
            exchangeCoupon.setDays(sampleVO.getDateNum());
            exchangeCouponMap.put(sampleVO.getId(), exchangeCoupon);
        }
        for (CpCouponSetVO cpCouponSetVO : couponSetMap.values()) {
            List<ExchangeCoupon> exchangeCoupons = Lists.newArrayList();
            String csId = cpCouponSetVO.getCouponSampleId();
            String[] cSampleIds = csId.split(",");
            for (int i = 0; i < cSampleIds.length; i++) {
                if (null != exchangeCouponMap.get(Long.valueOf(cSampleIds[i]))) {
                    exchangeCoupons.add(exchangeCouponMap.get(Long.valueOf(cSampleIds[i])));
                }
            }
            groupCouponMap.put(cpCouponSetVO.getId(), exchangeCoupons);
        }
        return groupCouponMap;
    }

    /**
     * 获取优惠券模板
     *
     * @param couponSetMap
     * @return
     */
    private List<CpCouponSampleVO> getCpCouponSampleVOS(Map<Long, CpCouponSetVO> couponSetMap) {
        List<Long> sampleIds = Lists.newArrayList();
        for (CpCouponSetVO cpCouponSetVO : couponSetMap.values()) {
            String csId = cpCouponSetVO.getCouponSampleId();
            String[] cSampleIds = csId.split(",");
            for (int i = 0; i < cSampleIds.length; i++) {
                sampleIds.add(Long.valueOf(cSampleIds[i]));
            }
        }
        return cpCouponSampleService.getCouponSampleByIds(sampleIds);
    }


    /**
     * 套券转换为ExchangeGoods
     *
     * @param exchangeGroupCouponMap
     * @param spuConfList
     * @param stockMap
     * @return
     */
    private List<ExchangeGroupCouponGoods> convertGroupCoupon2ExchangeGoodsList(Map<Long, List<ExchangeCoupon>> exchangeGroupCouponMap, Map<Long, CpCouponSetVO> couponSetMap, List<ExchgSpuConfigVO> spuConfList, Map<String, Integer> stockMap) {

        if (MapUtils.isEmpty(exchangeGroupCouponMap)) {
            return null;
        }
        List<ExchangeGroupCouponGoods> targets = Lists.newArrayList();

        for (ExchgSpuConfigVO spuConfigVO : spuConfList) {
            String referenceCode = spuConfigVO.getReferenceCode();
            ExchangeGroupCouponGoods groupCouponGoods = this.buildExchangeGroupCoupon(exchangeGroupCouponMap.get(Long.valueOf(referenceCode)), couponSetMap.get(Long.valueOf(referenceCode)), spuConfigVO, stockMap.get(referenceCode));
            targets.add(groupCouponGoods);
        }
        return targets;
    }

    /**
     * 获取优化券库存信息
     *
     * @param sampleIds
     * @return
     */
    private Map<String, Integer> getExchangeCouponStockMap(List<Long> sampleIds) {

        Map<String, Integer> stockMap = new HashMap<>();
        if (CollectionUtils.isEmpty(sampleIds)) return stockMap;
        List<String> stockKeys = Lists.newArrayList();
        for (Long sampleId : sampleIds) {
            stockKeys.add(sampleId + "");
        }
        ServiceResp<List<SSkuPointExchangeStockVO>> resp = stockService.getPointExchangeStocks(ExchgTypeEnum.GROUP_COUPON, stockKeys);
        if (resp.isSuccess() && CollectionUtils.isNotEmpty(resp.getBean())) {
            for (SSkuPointExchangeStockVO exchangeStock : resp.getBean()) {
                stockMap.put(exchangeStock.getReferenceCode(), exchangeStock.getCanUseQuantity());
            }
        }
        return stockMap;
    }

    /**
     * 构建优化券返回对象
     *
     * @param exchangeCoupons
     * @param spuConf
     * @param stock
     * @return
     */
    private ExchangeGroupCouponGoods buildExchangeGroupCoupon(List<ExchangeCoupon> exchangeCoupons, CpCouponSetVO cpCouponSet, ExchgSpuConfigVO spuConf, Integer stock) {
        ExchangeGroupCouponGoods goods = new ExchangeGroupCouponGoods();
        goods.setOnline(ExchgConfigStatusEnum.ONLINE.equals(spuConf.getStatus()));
        goods.setMode(spuConf.getExchangeMode().name());
        goods.setId(spuConf.getId().intValue());
        List<Integer> levels = new ArrayList<>();
        for (MemberLevelEnum levelEnum : spuConf.getApplyMemberLevel()) {
            levels.add(levelEnum.getCode());
        }
        goods.setCode(spuConf.getReferenceCode());
        goods.setLevelIds(levels);
        goods.setPoint(spuConf.getExchgAmount());
        goods.setExchangePrice(spuConf.getExchgPrice());
        goods.setTotal(spuConf.getTotalStock());
        if (NumberUtil.isNullOrZero(stock)) {
            goods.setSoldOut(Boolean.TRUE);
        }
        goods.setTitle(cpCouponSet.getName());
        goods.setDescription("");
        goods.setNote(cpCouponSet.getNote());
        if (CollectionUtils.isNotEmpty(exchangeCoupons)) {
            goods.setSubCoupons(exchangeCoupons);
        }
        return goods;
    }


}
