package com.thebeastshop.pegasus.channelservice.adapter.exchange;

import com.google.common.collect.Lists;
import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.common.utils.NumberUtil;
import com.thebeastshop.coupon.service.CpCouponSampleService;
import com.thebeastshop.coupon.vo.CpCouponSampleVO;
import com.thebeastshop.exchange.enums.ExchgConfigStatusEnum;
import com.thebeastshop.exchange.enums.ExchgTypeEnum;
import com.thebeastshop.exchange.vo.ExchgSpuConfigVO;
import com.thebeastshop.stock.service.SStockService;
import com.thebeastshop.stock.vo.SSkuPointExchangeStockVO;
import com.thebeastshop.support.enums.exchange.ExchangeGoodsClassify;
import com.thebeastshop.support.vo.exchange.ExchangeCouponGoods;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author feilong.gao
 * @Date 2018/5/23 16:52.
 * @Description
 */
@Component
public class ExchangeCouponGoodsAdapter {


    @Autowired
    private CpCouponSampleService cpCouponSampleService;

    @Autowired
    private SStockService stockService;

    public List<ExchangeCouponGoods> getExchangeCoupon(List<ExchgSpuConfigVO> exchangeList) {

        if (CollectionUtils.isEmpty(exchangeList)) {
            return null;
        }

        List<Long> sampleIds = Lists.newArrayList();
        Map<String, ExchgSpuConfigVO> exchgConfMap = new HashMap<>();
        for (ExchgSpuConfigVO simple : exchangeList) {
            sampleIds.add(Long.parseLong(simple.getReferenceCode()));
            exchgConfMap.put(simple.getReferenceCode(), simple);
        }
        Map<String, Integer> stockMap = getExchangeCouponStockMap(sampleIds);
        List<CpCouponSampleVO> sampleList = cpCouponSampleService.getCouponSampleByIds(sampleIds);

        return convertCoupon2ExchangeSimpleList(sampleList, exchgConfMap, stockMap);
    }


    /**
     * 商品信息转换为ExchangeGoods
     *
     * @param sampleList
     * @param spuConfMap
     * @param stockMap
     * @return
     */
    private List<ExchangeCouponGoods> convertCoupon2ExchangeSimpleList(List<CpCouponSampleVO> sampleList, Map<String, ExchgSpuConfigVO> spuConfMap, Map<String, Integer> stockMap) {

        if (CollectionUtils.isEmpty(sampleList)) {
            return null;
        }
        List<ExchangeCouponGoods> targets = Lists.newArrayList();

        for (CpCouponSampleVO sampleVO : sampleList) {
            String sampleId = sampleVO.getId().toString();
            ExchangeCouponGoods couponGoods = this.buildExchangeCoupon(sampleVO, spuConfMap.get(sampleId), stockMap.get(sampleId));
            targets.add(couponGoods);
        }
        return targets;
    }

    /**
     * 获取优化券库存信息
     *
     * @param sampleIds
     * @return
     */
    private Map<String, Integer> getExchangeCouponStockMap(List<Long> sampleIds) {

        Map<String, Integer> stockMap = new HashMap<>();
        if (CollectionUtils.isEmpty(sampleIds)) return stockMap;
        List<String> stockKeys = Lists.newArrayList();
        for (Long sampleId : sampleIds) {
            stockKeys.add(sampleId + "");
        }
        ServiceResp<List<SSkuPointExchangeStockVO>> resp = stockService.getPointExchangeStocks(ExchgTypeEnum.COUPON, stockKeys);
        if (resp.isSuccess() && CollectionUtils.isNotEmpty(resp.getBean())) {
            for (SSkuPointExchangeStockVO exchangeStock : resp.getBean()) {
                stockMap.put(exchangeStock.getReferenceCode(), exchangeStock.getCanUseQuantity());
            }
        }
        return stockMap;
    }

    /**
     * 构建优化券返回对象
     *
     * @param sampleVO
     * @param spuConf
     * @param stock
     * @return
     */
    private ExchangeCouponGoods buildExchangeCoupon(CpCouponSampleVO sampleVO, ExchgSpuConfigVO spuConf, Integer stock) {
        ExchangeCouponGoods goods = new ExchangeCouponGoods();
        goods.setClassify(ExchangeGoodsClassify.COUPON);
        goods.setOnline(ExchgConfigStatusEnum.ONLINE.equals(spuConf.getStatus()));
        goods.setMode(spuConf.getExchangeMode().name());
        goods.setId(spuConf.getId().intValue());
        List<Integer> levels = new ArrayList<>();
        for (MemberLevelEnum levelEnum : spuConf.getApplyMemberLevel()) {
            levels.add(levelEnum.getCode());
        }
        goods.setCode(spuConf.getReferenceCode());
        goods.setLevelIds(levels);
        goods.setPoint(spuConf.getExchgAmount());
        goods.setExchangePrice(spuConf.getExchgPrice());
        goods.setTotal(spuConf.getTotalStock());
        if (NumberUtil.isNullOrZero(stock) || stock < 0) {
            goods.setSoldOut(Boolean.TRUE);
        }
        if (sampleVO != null) {
            goods.setTitle(sampleVO.getTitle());
            goods.setDescription(sampleVO.getDescription());
            goods.setThresholdDesc(sampleVO.getProductDesc());
            goods.setNote(sampleVO.getNote());
            goods.setStartTime(sampleVO.getStartTime());
            goods.setExpireTime(sampleVO.getExpireTime());
            goods.setDays(sampleVO.getDateNum());
        }
        return goods;
    }


}
