package com.thebeastshop.pegasus.channelservice.adapter.exchange;

import com.thebeastshop.common.ServiceResp;
import com.thebeastshop.common.enums.MemberLevelEnum;
import com.thebeastshop.exchange.dto.ExchgAdmissionCheckDTO;
import com.thebeastshop.exchange.enums.ExchgErrorCodeEnum;
import com.thebeastshop.exchange.enums.ExchgTypeEnum;
import com.thebeastshop.exchange.service.ExchangeService;
import com.thebeastshop.exchange.vo.MemberExchgVO;
import com.thebeastshop.member.vo.MemberVO;
import com.thebeastshop.pegasus.channelservice.exception.ChannelException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @author feilong.gao
 * @Date 2018/5/25 16:05.
 * @Description
 */
@Component
public class ExchangeAdapter {

    @Autowired
    ExchangeService exchangeService;


    /**
     * 校验兑换优惠券商品
     *
     * @param member
     * @param referenceCode
     * @throws ChannelException
     */
    public MemberExchgVO checkExchangeCouponGoods(MemberVO member, String referenceCode) throws ChannelException {
        return this.checkExchangeGoods(member, referenceCode, null, 1);
    }


    /**
     * 校验兑换商品，抛出校验异常
     * 1、校验记录是否存在
     * 2、校验是否有库存
     * 3、校验当前会员是否可以兑换
     * 4、校验当前会员是否超过限定兑换次数
     * 5、校验会员是否有充足积分
     *
     * @param member
     * @param referenceCode
     * @param skuCode
     */
    public MemberExchgVO checkExchangeGoods(MemberVO member, String referenceCode, String skuCode, Integer exchangeNum) throws ChannelException {

        // 查询要兑换的数据
        List<MemberExchgVO> exchgResp = exchangeService.queryMemberExchg(member.getCode(), referenceCode);
        MemberExchgVO checkExchg = null;
        // 取出要校验的商品
        if (CollectionUtils.isNotEmpty(exchgResp)) {
            for (MemberExchgVO exchg : exchgResp) {
                if (ExchgTypeEnum.SPU.equals(exchg.getExchangeType())) {
                    if (StringUtils.isNotEmpty(exchg.getSkuCode()) && exchg.getSkuCode().equalsIgnoreCase(skuCode)) {
                        checkExchg = exchg;
                        break;
                    }
                } else {
                    checkExchg = exchg;
                }
            }
        } else {
            throw new ChannelException(4000, "抱歉，兑换商品不存在！");
        }

        // 设置错误提示信息
        StringBuffer levelMsg = new StringBuffer();
        StringBuffer limitMsg = new StringBuffer();
        if (checkExchg != null) {
            levelMsg.append("抱歉，仅限");
            List<MemberLevelEnum> levels = checkExchg.getApplyMemberLevel();
            String delimiter = "";
            for (MemberLevelEnum level : levels) {
                levelMsg.append(delimiter);
                levelMsg.append(level.getName());
                delimiter = ",";
            }
            levelMsg.append("专享！");

            Integer limitNum = checkExchg.getLimitPerMember();
            limitMsg.append("抱歉，每人限兑").append(limitNum).append("个，您已超过兑换上限");
        }else {
            throw new ChannelException(4000, "抱歉，兑换商品不存在！");
        }

        // 校验规则
        ExchgAdmissionCheckDTO checkQuery = new ExchgAdmissionCheckDTO();
        checkQuery.setMemberLevel(MemberLevelEnum.getEnumByCode(member.getMemberLevel()));
        checkQuery.setMemberCode(member.getCode());
        checkQuery.setMemberId(member.getId().intValue());
        checkQuery.setReferenceCode(checkExchg.getReferenceCode());
        checkQuery.setSkuCode(checkExchg.getSkuCode());
        checkQuery.setExchangeNum(exchangeNum);
        ServiceResp<Boolean> resp = exchangeService.checkAdmission(checkQuery);
        if (resp.isFailure()) {
            if (ExchgErrorCodeEnum.NO_ADMISSION_ERROR.getErrorCode().equalsIgnoreCase(resp.getRespCode())) {
                throw new ChannelException(4000, "没有兑换权限！");
            } else if (ExchgErrorCodeEnum.CONFIG_LACK_STOCK_ERROR.getErrorCode().equalsIgnoreCase(resp.getRespCode())) {
                throw new ChannelException(4000, "抱歉，商品已兑完！");
            } else if (ExchgErrorCodeEnum.MEMBER_LEVEL_DISSATISFY_ERROR.getErrorCode().equalsIgnoreCase(resp.getRespCode())) {
                throw new ChannelException(4000, levelMsg.toString());
            } else if (ExchgErrorCodeEnum.BEYOND_LIMIT_ERROR.getErrorCode().equalsIgnoreCase(resp.getRespCode())) {
                throw new ChannelException(4000, limitMsg.toString());
            } else if (ExchgErrorCodeEnum.LACK_INTEGRAL_ERROR.getErrorCode().equalsIgnoreCase(resp.getRespCode())) {
                throw new ChannelException(4000, "抱歉，您当前积分不足！");
            } else {
                throw new ChannelException(4000, resp.getRespMsg());
            }
        }
        return checkExchg;
    }


}
